package ru.yandex.msearch.proxy.api.async.suggest.history;

import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.apache.http.HttpException;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;

import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.SortedSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;

import ru.yandex.msearch.proxy.api.async.suggest.united.Target;

public class MixedHistorySuggestRule
    implements SuggestRule<Suggests<? extends Suggest>> {

    private static final Comparator<Suggest> COMPARATOR = (o1, o2) -> {
        if (o1.target() == o2.target()) {
            return 0;
        }

        if (o1.target() == Target.HISTORY) {
            return 1;
        }

        return -1;
    };

    private final SuggestRule<Suggests<? extends Suggest>> historyRule;
    private final SuggestRule<Suggests<? extends Suggest>> subjectRule;

    public MixedHistorySuggestRule(
        final SuggestRule<Suggests<? extends Suggest>> historyRule,
        final SuggestRule<Suggests<? extends Suggest>> mixedRule)
    {
        this.historyRule = historyRule;
        this.subjectRule = mixedRule;
    }

    @Override
    public void execute(
        final SuggestRequest<Suggests<? extends Suggest>> request)
        throws HttpException
    {
        if (!request.cgiParams().getBoolean("subject", false)) {
            historyRule.execute(request);
            return;
        }

        MultiFutureCallback<Suggests<? extends Suggest>> mfCallback
            = new MultiFutureCallback<>(new AggregateCallback(request));
        historyRule.execute(request.withCallback(mfCallback.newCallback()));
        subjectRule.execute(request.withCallback(mfCallback.newCallback()));
        mfCallback.done();
    }

    private static class AggregateCallback
        extends AbstractFilterFutureCallback<List<Suggests<? extends Suggest>>, Suggests<? extends Suggest>>
    {
        private final int limit;
        public AggregateCallback(
            final SuggestRequest<Suggests<? extends Suggest>> request)
        {
            super(request.callback());

            limit = request.requestParams().length();
        }

        @Override
        public void completed(
            final List<Suggests<? extends Suggest>> suggests)
        {
            SortedSuggests<Suggest> result =
                new SortedSuggests<>(COMPARATOR, limit);
            Iterator<? extends Suggest> history = suggests.get(0).iterator();
            Iterator<? extends Suggest> subject = suggests.get(1).iterator();

            while (!result.limitReached()
                && (history.hasNext()
                || subject.hasNext()))
            {
                if (history.hasNext()) {
                    result.add(history.next());
                }

                if (subject.hasNext()) {
                    result.add(
                        new BasicSuggest(
                            Target.SUBJECT,
                            subject.next().showText()));
                }
            }

            callback.completed(result);
        }
    }
}
