package ru.yandex.msearch.proxy.api.async.suggest.history;

import java.util.List;
import java.util.Locale;

import ru.yandex.dbfields.MailIndexFields;

import ru.yandex.msearch.proxy.api.async.suggest.BasicRequestTextField;
import ru.yandex.msearch.proxy.api.async.suggest.BasicRequestTextField
    .FieldFunction;

/**
 * Describes fields in lucene/msearch-proxy for stored request. Also accumulates
 * normalization functions for field, if they applied.
 *
 */
public final class StoredRequestFields {
    /**
     * Raw request
     **/
    public static final StorableRequestTextField RAW =
        new StorableRequestTextField(MailIndexFields.REQUEST_RAW);
    /**
     * Normalized unicode version of request, separated to words
     * Each word without non word symbols
     **/
    public static final StorableRequestTextField NORMALIZED =
        new StorableRequestTextField("request_normalized");
    /**
     * Lemmed request
     **/
    public static final StorableRequestTextField MORPHOLOGY =
        new StorableRequestTextField(
            "request_morph",
            new Join(Join.SPACE),
            new LowercasedJoin(Join.ESCAPED_SPACE, true));
    /**
     * Normalized unicode version of request without non word symbols
     * and also without spaces
     **/
    public static final StorableRequestTextField SPACELESS =
        new StorableRequestTextField(
            "request_spaceless",
            new Spaceless(),
            new Spaceless(true));
    /**
     * If request was corrected by misspell
     **/
    public static final StorableRequestTextField ORIGINAL =
        new StorableRequestTextField(MailIndexFields.REQUEST_ORIGINAL);

    /**
     * Field is filled when on request, suggest by misspell occured
     **/
    public static final StorableRequestTextField SUGGESTED =
        new StorableRequestTextField(MailIndexFields.REQUEST_SUGGEST);

    /**
     * * Timestamp for request
     **/
    public static final BasicRequestTextField DATE =
        new BasicRequestTextField(MailIndexFields.REQUEST_DATE);

    /**
     * Day for request
     **/
    public static final BasicRequestTextField DAY =
        new BasicRequestTextField("request_day");

    /**
     * How much user did this request at all
     */
    public static final BasicRequestTextField COUNT =
        new BasicRequestTextField(MailIndexFields.REQUEST_COUNT);

    private StoredRequestFields() {
    }

    public static final class StorableRequestTextField
        extends BasicRequestTextField
    {
        private final FieldFunction storeFunction;

        public StorableRequestTextField(final String name) {
            this(name, new Join(Join.SPACE), new Join(Join.ESCAPED_SPACE));
        }

        public StorableRequestTextField(
            final String name,
            final FieldFunction storeFunction,
            final FieldFunction requestFunction)
        {
            super(name, requestFunction);
            this.storeFunction = storeFunction;
        }

        public StringBuilder processForStore(
            final StringBuilder sb,
            final List<String> words)
        {
            return this.storeFunction.apply(sb, words);
        }

        public String processForStore(final List<String> words) {
            return this.storeFunction.apply(words);
        }
    }


    public static final class Spaceless extends FieldFunction {

        public Spaceless(boolean wildcard) {
            super(wildcard);
        }

        public Spaceless() {
        }

        @Override
        public StringBuilder apply(
            final StringBuilder sb,
            final List<String> words)
        {
            for (String word: words) {
                sb.append(word.toLowerCase(Locale.ENGLISH).replace('ё', 'е'));
            }

            if (wildcard) {
                sb.append("*");
            }

            return sb;
        }
    }

    public static final class LowercasedJoin extends Join {
        public LowercasedJoin(String separator, boolean wildcard) {
            super(separator, wildcard);
        }

        public LowercasedJoin(String separator) {
            super(separator);
        }

        @Override
        protected StringBuilder joinWord(
            final StringBuilder sb,
            final String word)
        {
            return sb.append(word.toLowerCase(Locale.ENGLISH));
        }
    }

    public static class Join extends FieldFunction {
        public static final String ESCAPED_SPACE = "\\ ";
        public static final String SPACE = " ";

        private final String separator;

        public Join(final String separator, final boolean wildcard) {
            super(wildcard);
            this.separator = separator;
        }

        public Join(final String separator) {
            super();
            this.separator = separator;
        }

        protected StringBuilder joinWord(
            final StringBuilder sb,
            final String word)
        {
            return sb.append(word);
        }

        @Override
        public StringBuilder apply(
            final StringBuilder sb,
            final List<String> words)
        {
            for (int i = 0; i < words.size(); i++) {
                if (i != 0) {
                    sb.append(separator);
                }

                if (!words.get(i).isEmpty()) {
                    joinWord(sb, words.get(i));
                }
            }

            if (wildcard) {
                sb.append("*");
            }

            return sb;
        }
    }
}
