package ru.yandex.msearch.proxy.api.async.suggest.history.rules;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.http.HttpException;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.RequestTextField;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequestParams;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequestText;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;
import ru.yandex.msearch.proxy.api.async.suggest.history.StoredRequestFields;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

/**
 * TokenizeRule for replacing requests to actual lucene request text
 * with fields and request syntax
 *
 */
public class HistoryRewriteSuggestRule
    implements SuggestRule<Suggests<? extends Suggest>>
{

    private static final List<? extends RequestTextField> DEFAULT_SCOPE =
        Arrays.asList(
            StoredRequestFields.MORPHOLOGY,
            StoredRequestFields.SPACELESS);

    private static final String GET_FIELDS =
        StoredRequestFields.RAW.name() + ','
        + StoredRequestFields.NORMALIZED.name() + ','
        + StoredRequestFields.COUNT.name() + ','
        + StoredRequestFields.DATE.name() + ','
        + MailIndexFields.REQUEST_MIDS_COUNT + ','
        + MailIndexFields.REQUEST_COUNT;

    private final SuggestRule<Suggests<? extends Suggest>> next;

    public HistoryRewriteSuggestRule(
        final SuggestRule<Suggests<? extends Suggest>> next)
    {
        this.next = next;
    }

    @Override
    public void execute(
        final SuggestRequest<Suggests<? extends Suggest>> request)
        throws HttpException
    {
        CgiParams params = new CgiParams(request.cgiParams());
        params.replace("get", GET_FIELDS);
        params.replace("group", StoredRequestFields.NORMALIZED.name());

        List<String> requests = params.getAll("request");
        if (requests != null) {
            params.put("requests", new ArrayList<>(requests));
            SuggestRequestText requestText =
                new HistorySuggestRequestText(requests);

            if (requestText.isEmpty() && !params.getBoolean("empty")) {
                request.callback().completed(
                    BasicSuggests.empty(request.requestParams().target()));
                return;
            }

            StringBuilder sb = requestText.buildFields(DEFAULT_SCOPE);
            params.replace("request", sb.toString());
        } else {
            if (!params.getBoolean("empty")) {
                request.callback().completed(
                    BasicSuggests.empty(request.requestParams().target()));

                return;
            }
        }

        next.execute(request.withCgiParams(params));
    }

    private static class HistorySuggestRequestText extends SuggestRequestText {
        public HistorySuggestRequestText(final List<String> texts) {
            super(texts);
        }

        @Override
        public StringBuilder buildFields(
            final StringBuilder sb,
            final List<? extends RequestTextField> fields)
        {
            if (!isEmpty()) {
                return super.buildFields(sb, fields);
            }

            sb.append("(");
            // Ok, if we are on empty request here, we suppose
            // we need just last requests
            for (int j = 0; j < fields.size(); j++) {
                if (j != 0) {
                    sb.append(OR);
                }

                sb.append(fields.get(j).name());
                sb.append(KEY_VALUE_SEP);
                sb.append(ASTERISK);
            }

            sb.append(")");

            return sb;
        }
    }
}
