package ru.yandex.msearch.proxy.api.async.suggest.history.rules;

import java.util.Collection;
import java.util.Locale;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.CaseInsensitiveSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestFactors;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestFactors.SuggestFactor;
import ru.yandex.msearch.proxy.api.async.suggest.highlight.HighlightedSuggest;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestLuceneRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;
import ru.yandex.msearch.proxy.api.async.suggest.history.StoredRequestFields;
import ru.yandex.msearch.proxy.api.async.suggest.rules
    .AbstractPlainSuggestCallback;

import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

/**
 * History Suggest search request callback class
 *
 */
public class PlainHistorySuggestCallback
    extends AbstractPlainSuggestCallback<Suggests<? extends Suggest>>
{
    private final boolean highlight;
    private final Collection<String> requests;
    private final String requestNorm;
    private final int wordsCount;
    private final int requestLen;
    private final int hasAt;

    public PlainHistorySuggestCallback(
        final AsyncHttpServer server,
        final SuggestLuceneRequest searchRequest,
        final SuggestRequest<Suggests<? extends Suggest>> suggestRequest)
        throws BadRequestException
    {
        super(server, searchRequest, suggestRequest);

        this.highlight =
            suggestRequest.cgiParams().getBoolean(
                "highlight",
                server.config().suggestConfig().highlight());
        this.requests =
            suggestRequest.cgiParams().getAll("requests");
        requestNorm =
            SearchRequestText.normalize(
                suggestRequest.requestParams()
                    .request()).trim().toLowerCase(Locale.ROOT);

        String[] words = requestNorm.split("\\s+");
        wordsCount = words.length;
        requestLen = requestNorm.length();

        if (requestNorm.contains("@")) {
            hasAt = 1;
        } else {
            hasAt = 0;
        }
    }

    @Override
    public synchronized void completed(final SearchResult searchResult) {
        CaseInsensitiveSuggests suggests =
            new CaseInsensitiveSuggests(
                suggestRequest.requestParams().target(),
                (int) searchResult.hitsCount());

        for (SearchDocument document: searchResult.hitsArray()) {
            if (!document.attrs().containsKey(StoredRequestFields.RAW.name())) {
                continue;
            }

            String raw = document.attrs().get(StoredRequestFields.RAW.name());

            if (raw != null) {
                final String text = raw.trim();
                Suggest suggest =
                    new BasicSuggest(
                        suggestRequest.requestParams().target(),
                        text);

                long requestCount = 0;
                long docsCount = 0;
                long date = 0;
                try {
                    date = Long.parseLong(
                        document.attrs()
                            .getOrDefault(
                                StoredRequestFields.DATE.name(),
                                "0"));

                    requestCount =
                        Long.parseLong(
                            document.attrs()
                                .getOrDefault(
                                    MailIndexFields.REQUEST_COUNT,
                                    "0"));
                    docsCount =
                        Long.parseLong(
                            document.attrs().getOrDefault(
                                MailIndexFields.REQUEST_MIDS_COUNT,
                                "0"));
                } catch (NumberFormatException nfe) {
                }

                double[] factors = suggest.factors();
                SuggestFactors.set(
                    factors,
                    SuggestFactor.TARGET,
                    Target.HISTORY.ordinal());
                SuggestFactors.set(
                    factors,
                    SuggestFactor.DOCS_COUNT,
                    docsCount);
                SuggestFactors.set(
                    factors,
                    SuggestFactor.REUSE_COUNT,
                    requestCount);
                SuggestFactors.set(
                    factors,
                    SuggestFactor.FRESHNESS,
                    System.currentTimeMillis() - date);
                SuggestFactors.set(factors, SuggestFactor.REQUEST_WORDS, wordsCount);
                SuggestFactors.set(factors, SuggestFactor.REQUEST_SYMBOLS, requestLen);
                SuggestFactors.set(factors, SuggestFactor.REQUEST_AT, hasAt);

                if (highlight) {
                    String show_text =
                        HtmlHighlighter.INSTANCE.highlight(requests, text, true);

                    suggest = new HighlightedSuggest(suggest, show_text);
                }

                suggests.add(suggest);
            }
        }

        suggestRequest.logger().info(
            "Search completed, size " + suggests.size());

        callback.completed(suggests);
    }

    @Override
    protected BasicSuggests empty() {
        return BasicSuggests.empty(suggestRequest.requestParams().target());
    }
}
