package ru.yandex.msearch.proxy.api.async.suggest.label;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.client.wmi.Labels;
import ru.yandex.client.wmi.LabelsConsumer;
import ru.yandex.collection.IntList;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.SearchRequest;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;
import ru.yandex.msearch.proxy.api.async.suggest.highlight.HighlightedSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.util.string.StringUtils;

public class LabelSuggestRule
    implements SuggestRule<Suggests<? extends Suggest>>
{
    private static final String WMI_SUFFIX = "caller=msearch";
    private static final String UNREAD_LABEL_LID = "FAKE_SEEN_LBL";

    private static final Function<String, String> NORMALIZER =
        s -> s.toLowerCase(Locale.ROOT).replace('ё', 'е');
    private static final GenericFunction<JsonMap, String, JsonException>
        LABEL_PROCESSOR = label -> SearchRequestText.normalize(label.get("name").asString());

    private final AsyncHttpServer server;

    public LabelSuggestRule(final AsyncHttpServer server) {
        this.server = server;
    }

    @Override
    public void execute(
        final SuggestRequest<Suggests<? extends Suggest>> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        if (!params.containsKey("ql")
            && params.getString("request", "").trim().isEmpty())
        {
            request.callback().completed(
                new BasicSuggests(
                    Target.LABEL,
                    request.requestParams().length()));
            return;
        }

        String mdb = params.getString(ProxyParams.MDB);
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix suid;
        Prefix uid;
        Prefix prefix;
        if (mdb.equals("pg")) {
            suid = params.get(ProxyParams.SUID, null, prefixType);
            uid = params.get(ProxyParams.UID, prefixType);
            prefix = uid;
        } else {
            suid = params.get(ProxyParams.SUID, prefixType);
            uid = params.get(ProxyParams.UID, null, prefixType);
            prefix = suid;
        }

        boolean corp = SearchRequest.corp(prefix);
        ImmutableURIConfig labelsConfig;
        if (corp) {
            labelsConfig = server.config().corpLabelsConfig();
        } else {
            labelsConfig = server.config().labelsConfig();
        }

        QueryConstructor query = new QueryConstructor(
            new StringBuilder(labelsConfig.uri().toASCIIString())
                .append(labelsConfig.firstCgiSeparator())
                .append(WMI_SUFFIX));
        query.append(ProxyParams.MDB, mdb);
        if (uid != null) {
            query.append(ProxyParams.UID, uid.toString());
        }

        if (suid != null) {
            query.append(ProxyParams.SUID, suid.toString());
        }

        long timeout = params.getLong("timeout", -1L);
        String labelsRequest = query.toString();

        AsyncClient labelsClient =
            server.labelsClient(corp).adjust(request.session().context());
        try {
            if (timeout < 0) {
                labelsClient.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        new AsyncGetURIRequestProducerSupplier(labelsRequest),
                        new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                            server.filterSearchTvm2Ticket(corp))),
                    NormalizedLabelsConsumerFactory.OK,
                    request.session().listener()
                        .createContextGeneratorFor(labelsClient),
                    new LabelsCallback(request));
            } else {
                BasicAsyncRequestProducerGenerator producerGenerator =
                    new BasicAsyncRequestProducerGenerator(labelsRequest);
                producerGenerator.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    server.filterSearchTvm2Ticket(corp));
                labelsClient.execute(
                    Collections.singletonList(labelsConfig.host()),
                    producerGenerator,
                    request.session().requestStartTime() + timeout,
                    NormalizedLabelsConsumerFactory.OK,
                    request.session().listener()
                        .createContextGeneratorFor(labelsClient),
                    new LabelsCallback(request));
            }

        } catch (URISyntaxException ue) {
            request.callback().failed(ue);
        }
    }

    private Suggest buildSuggest(
        final Target target,
        final String key,
        final String lid,
        final String prefix,
        final int highlightLen)
    {
        return buildSuggest(target, key, lid, prefix, 0, highlightLen);
    }

    private Suggest buildSuggest(
        final Target target,
        final String key,
        final String lid,
        final String prefix,
        final int index,
        final int highlightLen)
    {
        Suggest suggest = new LabelSuggest(
            target,
            key,
            StringUtils.concat(prefix, key),
            lid
        );

        if (key != null && highlightLen >= 0) {
            suggest = new HighlightedSuggest(
                suggest,
                HtmlHighlighter.INSTANCE.highlightString(key, index, index + highlightLen));
        }

        return suggest;
    }

    private void suggest(
        final SuggestRequest<Suggests<? extends Suggest>> suggestRequest,
        final Labels userLabels,
        final boolean pure,
        final boolean highlight)
    {
        CgiParams params = suggestRequest.cgiParams();

        SuggestLanguagePack langPack =
            suggestRequest.requestParams().language();

        StringBuilder sb =
            new StringBuilder(params.getString("requestPrefix", ""));

        if (!pure) {
            sb.append(langPack.label());
            sb.append(':');
        }

        final String prefix = sb.toString();

        BasicSuggests suggests =
            new BasicSuggests(
                Target.LABEL,
                suggestRequest.requestParams().length());

        List<SuggestLabel> labels = new ArrayList<>(userLabels.lids().size());
        userLabels.lids().forEach((k, v) -> labels.add(new SuggestLabel(k)));

        for (String request: params.getAll(ProxyParams.REQUEST)) {
            String normRequest = NORMALIZER.apply(request);
            final int highlightLen = highlight ? normRequest.length(): -1;

            labels.forEach((label) -> {
                int index = label.match(normRequest);
                if (index >= 0) {
                    for (String lid: userLabels.lids().get(label.name)) {
                        suggests.add(
                            buildSuggest(
                                Target.LABEL,
                                label.name,
                                lid,
                                prefix,
                                index,
                                highlightLen));
                    }
                }
            });

            if (NORMALIZER.apply(langPack.unreadLabel())
                .startsWith(normRequest))
            {
                suggests.add(
                    buildSuggest(
                        Target.UNREAD,
                        langPack.unreadLabel(),
                        UNREAD_LABEL_LID,
                        prefix,
                        highlightLen));
            }

            if (NORMALIZER.apply(langPack.importantLabel())
                .startsWith(normRequest))
            {
                suggests.add(
                    buildSuggest(
                        Target.IMPORTANT,
                        langPack.importantLabel(),
                        userLabels.importantLid(),
                        prefix,
                        highlightLen));
            }
        }

        suggestRequest.callback().completed(suggests);
    }

    private class LabelsCallback implements FutureCallback<Labels> {
        final SuggestRequest<Suggests<? extends Suggest>> request;

        private final boolean pure;
        private final boolean highlight;

        public LabelsCallback(
            final SuggestRequest<Suggests<? extends Suggest>> request)
            throws HttpException
        {
            this.request = request;
            this.pure = request.cgiParams().getBoolean("pure", false);
            this.highlight =
                request.cgiParams().getBoolean(
                    "highlight",
                    server.config().suggestConfig().highlight());
        }

        @Override
        public void completed(final Labels labels) {
            suggest(request, labels, pure, highlight);
        }

        @Override
        public void failed(final Exception e) {
            request.callback().failed(e);
        }

        @Override
        public void cancelled() {
            request.callback().cancelled();
        }
    }

    private enum NormalizedLabelsConsumerFactory
        implements HttpAsyncResponseConsumerFactory<Labels>
    {
        INSTANCE;

        public static final StatusCheckAsyncResponseConsumerFactory<Labels> OK =
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                INSTANCE);

        @Override
        public LabelsConsumer create(
            final HttpAsyncRequestProducer producer,
            final HttpResponse response)
            throws HttpException
        {
            return new LabelsConsumer(LABEL_PROCESSOR, response.getEntity());
        }
    }

    private final class SuggestLabel {
        private final String name;
        private final String normalized;
        private final IntList indicies;

        public SuggestLabel(final String name) {
            this.name = name;
            this.normalized = NORMALIZER.apply(name);
            this.indicies = normalizeLabel(normalized);
        }

        private IntList normalizeLabel(final String name) {
            IntList indexes = new IntList(name.length() >> 1);

            boolean ws = true;
            for (int i = 0; i < name.length(); i++) {
                char c = name.charAt(i);
                if (Character.isWhitespace(c)
                    || Character.isSpaceChar(c)
                    || Character.isISOControl(c))
                {
                    if (!ws) {
                        ws = true;
                    }
                } else if (ws) {
                    indexes.add(i);
                    ws = false;
                }
            }

            return indexes;
        }

        public int match(final String request) {
            for (int i = 0; i < indicies.size(); i++) {
                if (normalized.startsWith(request, indicies.get(i))) {
                    return indicies.get(i);
                }
            }

            return -1;
        }
    }
}
