package ru.yandex.msearch.proxy.api.async.suggest.mail;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.dbfields.PgFields;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;

import ru.yandex.json.parser.JsonException;

import ru.yandex.json.writer.JsonType;
import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;

import ru.yandex.msearch.proxy.api.async.suggest.highlight.HighlightedSuggest;

import ru.yandex.msearch.proxy.config.ImmutableSuggestConfig;
import ru.yandex.msearch.proxy.config.MailSuggestConfig;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;

import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;

import ru.yandex.msearch.proxy.api.async.suggest.united.SuggestAdapter;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.api.async.suggest.united.UnitedSuggests;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.request.util.SearchRequestText;

public class MailSuggestAdapter implements SuggestAdapter {
    private static final int FETCH_COUNT= 400;
    private static final String SUBJECT = "hdr_subject";
    private static final String GET =
        "mid,fid,hid,attachname,mimetype,disposition_type,hdr_subject,hdr_from,"
            + "received_date,folder_type,message_type,"
            + "has_attachments,clicks_serp_count,clicks_total_count";

    private static final BasicSuggests EMPTY
        = new BasicSuggests(Target.MAIL, 0);

    private final AsyncHttpServer server;
    private final String rankModel;
    private final MailSuggestConfig mailConfig;
    private final ImmutableSuggestConfig unitedConfig;

    public MailSuggestAdapter(final AsyncHttpServer server) {
        this.server = server;
        this.unitedConfig = server.config().suggestConfig();
        this.mailConfig = unitedConfig.mailConfig();
        this.rankModel = mailConfig.mailSuggestRelevanceModel();
    }

    private void adjustParams(
        final QueryConstructor qc,
        final CgiParams params,
        final String param)
        throws BadRequestException
    {
        for (String value: params.getAll(param)) {
            qc.append(param, value);
        }
    }

    @Override
    public void execute(
        final SuggestRequest<UnitedSuggests> request,
        final CgiParams params,
        final FutureCallback<Suggests<? extends Suggest>> callback)
        throws HttpException
    {
        String requestStr = params.getString(ProxyParams.REQUEST, "");

        int minSymbols = mailConfig.minSymbols();

        if (params.getBoolean("pure", false)
            || requestStr.length() < minSymbols)
        {
            callback.completed(EMPTY);
            return;
        }

        AsyncClient client =
            server.proxyClient().adjust(request.session().context());

        QueryConstructor qc =
            new QueryConstructor("/api/async/mail/search/direct?");

        Long uid = params.getLong(ProxyParams.UID, null);
        String mdb = params.getString(ProxyParams.MDB, null);
        Long suid = params.getLong(ProxyParams.SUID, null);

        boolean corp  = false;
        if (uid != null) {
            qc.append(ProxyParams.UID, uid);
            corp = BlackboxUserinfo.corp(uid);
        }

        if (mdb != null) {
            qc.append(ProxyParams.MDB, mdb);
        }

        if (suid != null) {
            qc.append(ProxyParams.SUID, suid);
            corp = BlackboxUserinfo.corp(suid);
        }


        adjustParams(qc, params, ProxyParams.FROM);
        adjustParams(qc, params, ProxyParams.TO);
        adjustParams(qc, params, ProxyParams.FID);
        adjustParams(qc, params, ProxyParams.LID);
        adjustParams(qc, params, ProxyParams.LIDS);
        adjustParams(qc, params, ProxyParams.SCOPE);
        adjustParams(qc, params, ProxyParams.UNREAD);
        adjustParams(qc, params, ProxyParams.HAS_ATTACHMENTS);
        adjustParams(qc, params, ProxyParams.SIDE);
        adjustParams(qc, params, ProxyParams.FOLDER);

        qc.append("merge_func", "values");
        qc.append("get", GET);
        qc.append(
            ProxyParams.REQUEST,
            params.getString(ProxyParams.REQUEST, null));
        qc.append(ProxyParams.FIRST, "0");
        qc.append(
            ProxyParams.COUNT,
            String.valueOf(
                Math.max(request.requestParams().length(), FETCH_COUNT)));

        if (rankModel != null) {
            qc.append("model", rankModel);
            qc.append("ranking", "true");
        }

        long timeout = request.session().params().getLong("timeout", -1L);

        try {
            if (timeout > 0) {
                request.session().subscribeForCancellation(
                    client.execute(
                        Collections.singletonList(server.httpHost()),
                        new BasicAsyncRequestProducerGenerator(qc.toString()),
                        request.session().requestStartTime() + timeout,
                        JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                        request.session().requestsListener()
                            .createContextGeneratorFor(client),
                        new MailSearchCallback(request, corp, callback)));
            } else {
                request.session().subscribeForCancellation(
                    client.execute(
                        Collections.singletonList(server.httpHost()),
                        new BasicAsyncRequestProducerGenerator(qc.toString()),
                        JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                        request.session().requestsListener()
                            .createContextGeneratorFor(client),
                        new MailSearchCallback(request, corp, callback)));
            }
        } catch (IOException e) {
            // Quite impossible to fail server binding
            throw new ServiceUnavailableException(e);
        }
    }

    @Override
    public Target target() {
        return Target.MAIL;
    }

    private class MailSearchCallback
        extends AbstractFilterFutureCallback<JsonObject, Suggests<? extends Suggest>>
    {
        private final String request;
        private final boolean highlight;
        private final boolean corp;
        private final SuggestRequest<?> suggestRequest;

        public MailSearchCallback(
            final SuggestRequest<?> suggestRequest,
            final boolean corp,
            final FutureCallback<? super Suggests<? extends Suggest>> callback)
            throws BadRequestException
        {
            super(callback);

            this.corp = corp;
            this.highlight =
                suggestRequest.cgiParams().getBoolean(
                    "highlight",
                    server.config().suggestConfig().highlight());
            this.request =
                suggestRequest.cgiParams().getString(ProxyParams.REQUEST, "");
            this.suggestRequest = suggestRequest;
        }

        @Override
        public void completed(final JsonObject documentsObj) {
            int limit = suggestRequest.requestParams().length();

            System.out.println(JsonType.NORMAL.toString(documentsObj));
            BasicSuggests suggests = new BasicSuggests(Target.MAIL, limit);
            try {
                JsonList envelopes =
                    documentsObj.asMap().get("hitsArray").asList();

                String requestStr =
                    suggestRequest.requestParams().request().toLowerCase(
                        Locale.ROOT);

                String normalizedRequest =
                    SearchRequestText.normalizeSuggest(requestStr)
                        .replaceAll("ё", "е");

                for (JsonObject docObj: envelopes) {
                    if (suggests.limitReached()) {
                        break;
                    }

                    JsonMap doc = docObj.asMap();
                    Suggest suggest =
                        parseSuggestOrNull(
                            doc, corp, normalizedRequest);

                    if (suggest != null) {
                        if (highlight) {
                            suggest =
                                new HighlightedSuggest(
                                    suggest,
                                    HtmlHighlighter.INSTANCE.highlight(
                                        Collections.singletonList(request),
                                        doc.getString(SUBJECT, ""),
                                        true));
                        }

                        suggests.add(suggest);
                    }
                }

                callback.completed(suggests);
            } catch (JsonException e) {
                failed(e);
                return;
            }
        }
    }

    protected SuggestAttach applyForAttach(final JsonMap doc) {
        String depType =
            doc.getString("disposition_type", "");

        String hid = doc.getString("hid", null);

        if (!"attachment".equalsIgnoreCase(depType)) {
            return null;
        }
        String attachname =
            doc.getString("attachname", null);
        String mimetype =
            doc.getString("mimetype", null);

        if (hid == null || attachname == null || mimetype == null) {
            return null;
        }

        int typeSepInd = mimetype.indexOf("/");
        if (typeSepInd < 0) {
            return null;
        }

        String attType = mimetype.substring(0, typeSepInd);
        String attSubType = mimetype.substring(typeSepInd + 1);

        String fileExt = "";
        int extInd = attachname.lastIndexOf('.');
        if (extInd >= 0) {
            fileExt =
                attachname.substring(extInd).toLowerCase(Locale.ENGLISH);
        }

        return new SuggestAttach(
            hid,
            attachname,
            fileExt,
            attType,
            attSubType);
    }

    public Suggest parseSuggestOrNull(
        final JsonMap doc,
        final boolean corp,
        final String normalizedRequest)
        throws JsonException
    {
        boolean hid0 = false;
        String hid = doc.get(MailMetaInfo.HID).asStringOrNull();
        String mid = doc.get(PgFields.MID).asStringOrNull();
        String fid = doc.get(OracleFields.FID).asStringOrNull();
        String folderType = doc.get("folder_type").asStringOrNull();

        if (!corp
            && (folderType == null
            || folderType.equalsIgnoreCase("spam")
            || folderType.equalsIgnoreCase("trash")))
        {
            return null;
        }

        String from =
            doc.get(MailIndexFields.HDR + MailIndexFields.FROM)
                .asStringOrNull();

        if (from == null) {
            return null;
        }

        boolean hasAttachments =
            doc.getBoolean("has_attachments", false);

        String date =
            doc.get(MailIndexFields.RECEIVED_DATE)
                .asString();

        String subject = doc.get(SUBJECT).asStringOrNull();
        if (mid == null || subject == null || fid == null) {
            return null;
        }

        if ("0".equalsIgnoreCase(hid)) {
            hid0 = true;
        }

        List<JsonObject> merged = doc.getListOrNull("merged_docs");
        if (merged == null) {
            merged = Collections.emptyList();
        }

        List<SuggestAttach> attaches = new ArrayList<>(merged.size() + 1);
        SuggestAttach attach = applyForAttach(doc);
        if (attach != null) {
            attaches.add(attach);
        }

        System.out.println("Hid0 " + hid0 + " " + attaches);
        if (!hid0) {

            for (JsonObject md: merged) {
                if ("0".equalsIgnoreCase(
                    doc.get(MailMetaInfo.HID).asStringOrNull()))
                {
                    hid0 = true;
                    break;
                }
                attach = applyForAttach(md.asMap());
                if (attach != null) {
                    attaches.add(attach);
                }
            }
        }

        String normalizedSubject =
            SearchRequestText.normalizeSuggest(subject)
                .toLowerCase(Locale.ROOT)
                .replaceAll("ё", "е");

        Set<MailSuggestScope> scopes = new LinkedHashSet<>();
        if (normalizedSubject.contains(normalizedRequest)) {
            scopes.add(MailSuggestScope.SUBJECT);
        }

        if (hid0) {
            if (scopes.isEmpty()) {
                scopes.add(MailSuggestScope.BODY);
            }
            // if hid 0 returned, hit could be anywhere
            return new MailSuggest(
                mid,
                fid,
                from,
                date,
                subject,
                scopes,
                hasAttachments);
        }

        if (attaches.size() < merged.size() + 1) {
            scopes.add(MailSuggestScope.BODY);
        }

        if (attaches.size() > 0) {
            scopes.add(MailSuggestScope.ATTACHMENT);
        }

        return new MailSuggest(
            mid,
            fid,
            from,
            date,
            subject,
            scopes,
            attaches,
            hasAttachments);
    }
}
