package ru.yandex.msearch.proxy.api.async.suggest.ql;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.commons.collections4.trie.PatriciaTrie;
import org.apache.http.HttpException;
import org.joda.time.DateTime;

import ru.yandex.msearch.proxy.api.async.mail.rules.LuceneQueryConstructor;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;
import ru.yandex.msearch.proxy.api.async.suggest.highlight.HighlightedSuggest;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestMultiLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.api.async.suggest.united.rules.FilterSuggestRule;
import ru.yandex.msearch.proxy.config.ImmutableSuggestConfig;

import ru.yandex.parser.string.PositiveLongValidator;

import ru.yandex.parser.uri.CgiParams;

public class QueryLanguageSuggestRule
    implements SuggestRule<Suggests<? extends Suggest>>
{
    private final PatriciaTrie<String> qlKeys;
    private final ImmutableSuggestConfig config;

    public QueryLanguageSuggestRule(final ImmutableSuggestConfig config) {
        this.qlKeys = new PatriciaTrie<>();
        this.config = config;

        SuggestMultiLanguagePack.INSTANCE.values()
            .forEach(v -> this.qlKeys.put(v + ':', v + ':'));
    }

    @Override
    public void execute(
        final SuggestRequest<Suggests<? extends Suggest>> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        String requestStr = params.getString("request", "").trim();

        if (requestStr.isEmpty()) {
            request.callback().completed(
                new BasicSuggests(
                    Target.CATEGORY,
                    request.requestParams().length()));
            return;
        }

        BasicSuggests suggests =
            new BasicSuggests(
                Target.CATEGORY,
                request.requestParams().length());

        final String normRequest = requestStr.toLowerCase(Locale.ROOT);

        String requestPrefix = params.getString("requestPrefix", "");

        final boolean highlight =
            request.cgiParams().getBoolean("highlight", config.highlight());

        qlKeys.prefixMap(normRequest)
            .forEach((k, v) -> {
                Suggest suggest =
                    new BasicSuggest(Target.QL, k, requestPrefix + k);
                if (highlight) {
                    suggest =
                        new HighlightedSuggest(
                            suggest,
                            HtmlHighlighter.INSTANCE.highlight(normRequest, k));
                }

                suggests.add(suggest);
            });

        int colon = FilterSuggestRule.colonIndex(requestStr);
        SuggestLanguagePack.QL token =
            FilterSuggestRule.extractToken(requestStr, colon);
        ++colon;
        while (colon < requestStr.length()) {
            char c = requestStr.charAt(colon);
            if (Character.isWhitespace(c) || Character.isSpaceChar(c)) {
                ++colon;
            } else {
                break;
            }
        }
        if (token == SuggestLanguagePack.QL.MONTH) {
            long fixedTimestamp = request.cgiParams().get(
                "fixed-timestamp",
                System.currentTimeMillis(),
                PositiveLongValidator.INSTANCE);
            DateTime date = new DateTime(
                fixedTimestamp,
                request.requestParams().timezone());
            int month = date.monthOfYear().get();
            List<String> suggestedMonths = new ArrayList<>(12);
            String prefix = requestStr.substring(colon);
            int prefixLength = prefix.length();
            if (prefixLength == 0) {
                List<String> months =
                    request.requestParams().language().months();
                for (int i = 0; i < 12; ++i) {
                    suggestedMonths.add(months.get((11 + month - i) % 12));
                }
            } else {
                String lower = prefix.toLowerCase(Locale.ROOT);
                List<Map.Entry<String, Integer>> entries = new ArrayList<>(2);
                for (Map.Entry<String, Integer> entry
                    : LuceneQueryConstructor.MONTHS.entrySet())
                {
                    String key = entry.getKey();
                    if (prefixLength <= 3) {
                        if (key.length() <= 3 && key.startsWith(lower)) {
                            entries.add(entry);
                        }
                    } else if (key.length() > 3 && key.startsWith(lower)) {
                        entries.add(entry);
                    }
                }
                List<Map.Entry<String, Integer>> partialEntries =
                    new ArrayList<>(entries.size());
                for (Map.Entry<String, Integer> entry: entries) {
                    if (entry.getValue() <= month) {
                        partialEntries.add(entry);
                    }
                }
                partialEntries.sort(
                    (x, y) -> y.getValue().compareTo(x.getValue()));
                for (Map.Entry<String, Integer> entry: partialEntries) {
                    suggestedMonths.add(entry.getKey());
                }
                partialEntries.clear();
                for (Map.Entry<String, Integer> entry: entries) {
                    if (entry.getValue() > month) {
                        partialEntries.add(entry);
                    }
                }
                partialEntries.sort(
                    (x, y) -> y.getValue().compareTo(x.getValue()));
                for (Map.Entry<String, Integer> entry: partialEntries) {
                    suggestedMonths.add(entry.getKey());
                }
            }
            for (String suggestedMonth: suggestedMonths) {
                Suggest suggest;
                if (prefixLength == 0) {
                    suggest = new BasicSuggest(
                        Target.QL,
                        requestStr.substring(0, colon) + suggestedMonth);
                } else {
                    suggest = new BasicSuggest(
                        Target.QL,
                        requestStr.substring(0, colon) + prefix
                        + suggestedMonth.substring(prefix.length()));
                }
                if (highlight) {
                    suggest = new HighlightedSuggest(
                        suggest,
                        HtmlHighlighter.INSTANCE.highlightString(
                            suggest.showText(),
                            0,
                            requestStr.length()));
                }
                suggests.add(suggest);
            }
        }

        request.callback().completed(suggests);
    }
}
