package ru.yandex.msearch.proxy.api.async.suggest.rules;

import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;

import ru.yandex.parser.uri.CgiParams;

public class MultiwordSuggestRule<T extends Suggests<? extends Suggest>>
    implements SuggestRule<T>
{
    private SuggestRule<T> next;

    public MultiwordSuggestRule(final SuggestRule<T> next) {
        this.next = next;
    }

    @Override
    public void execute(
        final SuggestRequest<T> sRequest)
        throws HttpException
    {
        CgiParams params = sRequest.cgiParams();
        String request = params.getString("request", "");
        int lastWsIndex =  -1;

        for (int i = 0; i < request.length(); i++) {
            if (Character.isWhitespace(request.charAt(i))
                || Character.isSpaceChar(request.charAt(i)))
            {
                lastWsIndex = i;
            }
        }

        if (params.containsKey("ql")
            || request.isEmpty()
            || lastWsIndex == -1)
        {
            next.execute(sRequest.withCgiParams(params));
            return;
        }

        String lastWord =
            request.substring(lastWsIndex + 1, request.length());

        if (lastWord.isEmpty()) {
            next.execute(sRequest.withCgiParams(params));
            return;
        }

        String prefix = request.substring(0, lastWsIndex + 1);
        CgiParams lastWordParams = new CgiParams(params);
        lastWordParams.replace("request", lastWord);
        lastWordParams.replace("requestPrefix", prefix);
        MultiwordSession session =
            new MultiwordSession(sRequest.logger(), sRequest.callback());

        sRequest.logger().info(
            "Multiword suggest, lastword "
                + lastWord + " prefix " + prefix);

        next.execute(
            new BasicSuggestRequest<>(
                sRequest.session(),
                lastWordParams,
                sRequest.requestParams(),
                new MultiwordCallback(session, true),
                sRequest.prefix() + "-lw"));
        next.execute(
            sRequest.withCallback(new MultiwordCallback(session, false)));
    }

    private class MultiwordCallback implements FutureCallback<T> {
        private final MultiwordSession session;
        private final boolean lastWord;

        public MultiwordCallback(
            final MultiwordSession session,
            final boolean lastWord)
        {
            this.session = session;
            this.lastWord = lastWord;
        }

        @Override
        public void completed(final T suggests) {
            if (lastWord) {
                session.completeLastword(suggests);
            } else {
                session.completeEntire(suggests);
            }
        }

        @Override
        public void failed(final Exception e) {
            session.partFailed(e);
        }

        @Override
        public void cancelled() {
            session.partCancelled();
        }
    }

    private class MultiwordSession implements FutureCallback<T> {
        private volatile boolean done = false;
        private volatile T savedSuggests = null;
        private final AtomicInteger left = new AtomicInteger(2);
        private final FutureCallback<? super T> callback;
        private final PrefixedLogger logger;

        public MultiwordSession(
            final PrefixedLogger logger,
            final FutureCallback<? super T> callback)
        {
            this.logger = logger;
            this.callback = callback;
        }

        @Override
        public void completed(final T suggests) {
            synchronized (this) {
                if (done) {
                    return;
                }

                done = true;
            }

            callback.completed(suggests);
        }

        @Override
        public void failed(final Exception e) {
            synchronized (this) {
                if (done) {
                    return;
                }

                done = true;
            }

            callback.failed(e);
        }

        @Override
        public void cancelled() {
            synchronized (this) {
                if (done) {
                    return;
                }

                done = true;
            }

            callback.cancelled();
        }


        private void partFailed(final Exception e) {
            if (done) {
                return;
            }

            if (left.decrementAndGet() == 0) {
                if (savedSuggests != null) {
                    completed(savedSuggests);
                } else {
                    failed(e);
                }
            }
        }

        private void partCancelled() {
            if (done) {
                return;
            }

            if (left.decrementAndGet() == 0) {
                if (savedSuggests != null) {
                    completed(savedSuggests);
                } else {
                    cancelled();
                }
            }
        }

        private void completeEntire(final T suggests) {
            logger.info(
                "Entire suggest size " + suggests.size() + " " + done);

            if (done) {
                return;
            }

            if (suggests.size() > 0) {
                completed(suggests);
            } else {
                if (left.decrementAndGet() == 0) {
                    if (savedSuggests != null) {
                        completed(savedSuggests);
                    } else {
                        completed(suggests);
                    }
                }

                // else we just rely on other one
            }
        }

        private void completeLastword(final T suggests) {
            if (done) {
                return;
            }

            logger.info("Lastword size " + suggests.size());

            if (suggests.size() > 0) {
                savedSuggests = suggests;
            }

            if (left.decrementAndGet() == 0) {
                logger.info("Lastword is last");
                completed(suggests);
            }
        }
    }
}
