package ru.yandex.msearch.proxy.api.async.suggest.rules;

import java.util.LinkedHashSet;
import java.util.Set;

import java.util.logging.Level;


import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.mail.Side;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.EmptyUnitedSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestFactors.SuggestFactor;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;

import ru.yandex.msearch.proxy.api.async.suggest.united.Target;

import ru.yandex.msearch.proxy.api.suggest.Translit;

import ru.yandex.parser.uri.CgiParams;

public class ParallelTranslitSuggestRule
    implements SuggestRule<Suggests<? extends Suggest>>
{
    private final SuggestRule<Suggests<? extends Suggest>> next;
    private final boolean failOnOriginalFail;

    public ParallelTranslitSuggestRule(
        final SuggestRule<Suggests<? extends Suggest>> next,
        final boolean failOnOriginalFail)
    {
        this.next = next;
        this.failOnOriginalFail = failOnOriginalFail;
    }

    @Override
    public void execute(
        final SuggestRequest<Suggests<? extends Suggest>> request)
        throws HttpException
    {
        int limit = request.requestParams().length();
        TranslitSession translitSession = new TranslitSession(
            request,
            failOnOriginalFail,
            limit);

        next.execute(
            request.withCallback(new OriginalRequestCallback(translitSession)));

        CgiParams params = new CgiParams(request.cgiParams());
        String requestText = params.getString("request", "");
        Set<String> requestSet = new LinkedHashSet<>();
        Side side = request.requestParams().side();
        requestSet.add(requestText);
        requestSet.addAll(Translit.suggestSet(requestText, side));

        // deduplicate original request
        requestSet.remove(requestText);

        if (requestSet.size() <= 0) {
            request.logger().info("No transliterations");
            translitSession.complete(
                TranslitSession.Request.Translit,
                EmptyUnitedSuggests.INSTANCE);

            return;
        }

        params.remove("request");
        StringBuilder logInfo =
            new StringBuilder((requestText.length() + 1)* requestSet.size());

        logInfo.append("Transliterations [");
        for (String translit : requestSet) {
            params.add("request", translit);
            logInfo.append(translit);
            logInfo.append(',');
        }

        logInfo.setLength(logInfo.length() - 1);

        logInfo.append("]");
        request.logger().info(logInfo.toString());

        next.execute(
            new BasicSuggestRequest<>(
                request.session(),
                params,
                request.requestParams(),
                new TranslitRequestCallback(translitSession),
                request.prefix() + "-translit"));
    }

    private static class TranslitSession {
        enum Request {
            Translit,
            Original
        }

        private final FutureCallback<? super Suggests<? extends Suggest>> callback;
        private final PrefixedLogger logger;
        private final int limit;
        private final Target target;
        private final boolean failOnOriginalFail;
        private Suggests<? extends Suggest> originalSuggest;
        private Suggests<? extends Suggest> translitSuggest;
        private int requestsLeft;
        private volatile boolean done = false;

        private TranslitSession(
            final SuggestRequest<Suggests<? extends Suggest>> request,
            final boolean failOnOriginalFail,
            final int limit)
        {
            this.callback = request.callback();
            this.logger = request.logger();
            this.target = request.requestParams().target();
            this.limit = limit;
            this.failOnOriginalFail = failOnOriginalFail;
            this.requestsLeft = 2;
        }

        public void complete(
            final Request request,
            final Suggests<? extends Suggest> result)
        {
            if (done) {
                return;
            }

            BasicSuggests suggests = null;

            synchronized (this) {
                if (done) {
                    return;
                }

                requestsLeft--;

                if (request == Request.Original) {
                    originalSuggest = result;
                } else {
                    translitSuggest = result;
                }

                if (originalSuggest != null && translitSuggest != null) {
                    suggests = new BasicSuggests(target, limit);
                    suggests.copy(originalSuggest);
                    suggests.copy(translitSuggest);
                    done = true;
                } else if (originalSuggest != null
                    && (requestsLeft <= 0 || limit <= originalSuggest.size()))
                {
                    suggests = new BasicSuggests(target, limit);
                    suggests.copy(originalSuggest);
                    done = true;
                } else if (requestsLeft <= 0) {
                    suggests = new BasicSuggests(target, limit);
                    suggests.copy(translitSuggest);
                    done = true;
                }
            }

            if (suggests != null) {
                callback.completed(suggests);
            }
        }

        public void failed(final Request request, final Exception e) {
            boolean failed = false;
            synchronized (this) {
                if (done) {
                    logger.log(Level.WARNING, "Request failed " + request, e);
                    return;
                }

                if (--requestsLeft <= 0
                    || (failOnOriginalFail && request == Request.Original))
                {
                    failed = true;
                    done = true;
                } else {
                    logger.log(Level.WARNING, "Request failed " + request, e);
                }
            }

            if (failed) {
                callback.failed(e);
            }
        }

        public void cancelled(final Request request) {
            boolean cancelled = false;
            synchronized (this) {
                if (done) {
                    logger.log(Level.WARNING, "Request cancelled " + request);
                    return;
                }

                if (--requestsLeft <= 0) {
                    cancelled = true;
                    done = true;
                } else {
                    logger.log(Level.WARNING, "Request cancelled " + request);
                }
            }

            if (cancelled) {
                callback.cancelled();
            }
        }
    }

    private static class OriginalRequestCallback
        implements FutureCallback<Suggests<? extends Suggest>>
    {
        private final TranslitSession translitSession;

        public OriginalRequestCallback(final TranslitSession translitSession)
        {
            this.translitSession = translitSession;
        }

        @Override
        public void completed(final Suggests<? extends Suggest> suggests) {
            if (suggests.size() > 0) {
                translitSession.logger.info(
                    "NonTranslit got " + suggests.size());
            }
            translitSession.complete(
                TranslitSession.Request.Original, suggests);
        }

        @Override
        public void failed(final Exception e) {
            translitSession.failed(TranslitSession.Request.Original, e);
        }

        @Override
        public void cancelled() {
            translitSession.cancelled(TranslitSession.Request.Original);
        }
    }

    private static class TranslitRequestCallback
        implements FutureCallback<Suggests<? extends Suggest>>
    {
        private final TranslitSession translitSession;

        public TranslitRequestCallback(
            TranslitSession translitSession)
        {
            this.translitSession = translitSession;
        }

        @Override
        public void completed(final Suggests<? extends Suggest> suggests)
        {
            suggests.forEach(
                s -> s.factors()[SuggestFactor.TRANSLIT.ordinal()] = 1);

            if (suggests.size() > 0) {
                translitSession.logger.info("Translit got " + suggests.size());
            }

            translitSession.complete(
                TranslitSession.Request.Translit, suggests);
        }

        @Override
        public void failed(final Exception e) {
            translitSession.failed(TranslitSession.Request.Translit, e);
        }

        @Override
        public void cancelled() {
            translitSession.cancelled(TranslitSession.Request.Translit);
        }
    }
}
