package ru.yandex.msearch.proxy.api.async.suggest.subject.rules;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.function.UnaryOperator;

import org.apache.http.HttpException;

import ru.yandex.dbfields.MailIndexFields;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.msearch.proxy.api.async.suggest.BasicRequestTextField;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.RequestTextField;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequestText;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;

import ru.yandex.msearch.proxy.config.SubjectSuggestConfig;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.request.util.SearchRequestText;

public class SubjectRewriteSuggestRule
    implements SuggestRule<Suggests<? extends Suggest>>
{
    private static final NormalizeFunction fieldFunc = new NormalizeFunction();
    public static final RequestTextField SUBJECT =
        new BasicRequestTextField("hdr_subject", fieldFunc);
    public static final RequestTextField SUBJECT_NORMALIZED =
        new BasicRequestTextField("hdr_subject_normalized", fieldFunc);
    public static final RequestTextField SUBJECT_KEYWORD =
        new BasicRequestTextField("hdr_subject_keyword", null);
    private static final String GET_FIELDS =
        SUBJECT.name() + ','
            + SUBJECT_NORMALIZED.name() + ','
            + MailIndexFields.THREAD_ID;

    private static final List<? extends RequestTextField> PG_SCOPE =
        Arrays.asList(SUBJECT_NORMALIZED);
    private static final List<? extends RequestTextField> ORACLE_SCOPE =
        Arrays.asList(SUBJECT);

    private static final List<String> NO_SUBJECT =
        Arrays.asList("no", "subject");
    private static final String[] NO_SUBJECT_RU = {"без", "темы"};

    private final SubjectSuggestWordsModifier wordsModifier;
    private final SuggestRule<Suggests<? extends Suggest>> next;

    public SubjectRewriteSuggestRule(
        final SubjectSuggestConfig suggestConfig,
        final SuggestRule<Suggests<? extends Suggest>> next)
    {
        this.next = next;
        this.wordsModifier = new SubjectSuggestWordsModifier(
            suggestConfig.subjectMinimumWordLength());
        fieldFunc.setMorphoMinLength(suggestConfig.minimumRequestLenToMorpho());
    }

    private static boolean isNoSubject(List<String> words) {
        if (words.isEmpty()) {
            return false;
        }

        if (words.size() > NO_SUBJECT_RU.length) {
            return false;
        }

        for (int i = 0; i < words.size(); i++) {
            String word = words.get(i);
            if (word.length() > NO_SUBJECT_RU[i].length()) {
                return false;
            }

            if (NO_SUBJECT_RU[i].startsWith(word)) {
                if (word.length() < NO_SUBJECT_RU[i].length()) {
                    if (i == words.size() - 1) {
                        return true;
                    } else {
                        return false;
                    }
                }
            } else {
                return false;
            }
        }

        return true;
    }

    @Override
    public void execute(
        final SuggestRequest<Suggests<? extends Suggest>> request)
        throws HttpException
    {
        CgiParams params = new CgiParams(request.cgiParams());
        String scope = params.getString(ProxyParams.SCOPE, null);

        List<String> requests = params.getAll("request");
        params.put("requestRaw", new ArrayList<>(requests));

        if (requests.isEmpty()
            || (scope != null && !"hdr_subject".equalsIgnoreCase(scope)))
        {
            request.callback().completed(
                BasicSuggests.empty(request.requestParams().target()));
            return;
        }

        params.replace("get", GET_FIELDS);
        params.replace("group", "thread_id");
        params.replace("merge_func", "count");

        SuggestRequestText requestText = new SuggestRequestText(requests,
            wordsModifier);
        if (requestText.isEmpty()) {
            request.callback().completed(
                BasicSuggests.empty(request.requestParams().target()));
            return;
        }

        boolean noSubject = false;
        for (List<String> part: requestText.getParts()) {
            if (noSubject = isNoSubject(part)) {
                break;
            }
        }

        StringBuilder text;
        if (noSubject) {
            text = new StringBuilder("((");
        } else {
            text = new StringBuilder("(");
        }


        String mdb = params.getString(ProxyParams.MDB);
        if (mdb.equalsIgnoreCase(ProxyParams.PG)) {
            text = requestText.buildFields(text, PG_SCOPE);
        } else {
            text = requestText.buildFields(text, ORACLE_SCOPE);
        }

        if (noSubject) {
            text.append(") OR ");
            text.append(SUBJECT_KEYWORD.name());
            text.append(":");
            text.append("No\\ subject");
        }

        text.append(")");
        String fid = params.getString("fid", null);
        FolderType folder =
            params.getEnum(FolderType.class, "folder", null);
        if (folder == null && fid == null) {
            if (params.getBoolean("excludeTrash", true)) {
                text.append(" AND NOT folder_type:trash");
            }

            if (params.getBoolean("excludeSpam", true)) {
                text.append(" AND NOT folder_type:spam");
            }
        } else {
            if (folder != null) {
                text.append(" AND folder_type:");
                text.append(SearchRequestText.fullEscape(folder.name().toLowerCase(Locale.ENGLISH), false));
            } else {
                text.append(" AND fid:");
                text.append(fid);
            }
        }

        text.append(" AND hid:0");

        if (params.getBoolean(ProxyParams.HAS_ATTACHMENTS, false)) {
            text.append(" AND has_attachments:1");
        }

        if (params.getBoolean(ProxyParams.UNREAD, false)) {
            text.append(" AND unread:1");
        }

        params.replace("request", text.toString());

        next.execute(request.withCgiParams(params));
    }

    private static final class SubjectSuggestWordsModifier
        implements UnaryOperator<String>
    {
        private final int length;

        public SubjectSuggestWordsModifier(int length) {
            this.length = length;
        }

        @Override
        public String apply(final String s) {
            if (s == null || s.length() < length) {
                return null;
            }
            return s;
        }
    }

    private static final class NormalizeFunction
        extends BasicRequestTextField.FieldFunction
    {

        // minimum length of request for morpho
        private int morphoMinLength;

        public void setMorphoMinLength(int morphoMinLength) {
            this.morphoMinLength = morphoMinLength;
        }

        private String normalize(final String word) {
            return word.toLowerCase(Locale.ENGLISH).replace('ё', 'е');
        }

        @Override
        public StringBuilder apply(
            final StringBuilder sb,
            final List<String> words)
        {
            boolean multiWord = words.size() > 1;
            if (multiWord) {
                sb.append('(');
            }

            int length = 0;
            for (int i = 0; i < words.size(); i++) {
                String normalized = normalize(words.get(i));
                length += normalized.length();

                if (normalized.isEmpty()) {
                    continue;
                }

                if (i != 0) {
                    sb.append(" AND ");
                }

                if (i == words.size() - 1) {
                    sb.append('(');
                    sb.append(normalized);
                    sb.append(SuggestRequestText.ASTERISK);
                    sb.append(" OR ");

                    if (length < morphoMinLength) {
                        sb.append("\"");
                        sb.append(normalized);
                        sb.append("\"");
                    } else {
                        sb.append(normalized);
                    }

                    sb.append(')');
                } else {
                    sb.append("\"");
                    sb.append(normalized);
                    sb.append("\"");
                }


            }

            if (multiWord) {
                sb.append(')');
            }

            return sb;
        }
    }
}
