package ru.yandex.msearch.proxy.api.async.suggest.united;

import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.RequestErrorType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public class AdapterCallback<T extends Suggests<? extends Suggest>>
    implements FutureCallback<T>
{
    private final FutureCallback<? super Suggests<? extends Suggest>> callback;
    private final RequestsStater stater;
    private final long start = System.currentTimeMillis();

    private PrefixedLogger logger;

    public AdapterCallback(
        final FutureCallback<Suggests<? extends Suggest>> callback,
        final RequestsStater stater)
    {
        this.callback = callback;
        this.stater = stater;
    }

    public void logger(final PrefixedLogger logger) {
        this.logger = logger;
    }

    @Override
    public void completed(final T result) {
        RequestInfo info = new RequestInfo(
            TimeSource.INSTANCE.currentTimeMillis(),
            HttpStatus.SC_OK,
            start,
            start,
            0L,
            0L);
        if (stater != null) {
            stater.accept(info);
        }

        if (logger != null) {
            logger.info(
                "Suggest completed with "
                    + result.size()
                    + " suggests in "
                    + info.requestTime() + " ms");
        }

        this.callback.completed(result);
    }

    @Override
    public void failed(final Exception e) {
        RequestErrorType errorType =
            RequestErrorType.ERROR_CLASSIFIER.apply(e);

        long requestTime = System.currentTimeMillis() - start;

        if (stater != null) {
            switch (errorType) {
                case NON_RETRIABLE:
                case HOST_NON_RETRIABLE:
                    stater.accept(
                        new RequestInfo(
                            TimeSource.INSTANCE.currentTimeMillis(),
                            HttpStatus.SC_BAD_REQUEST,
                            start,
                            start,
                            0L,
                            0L));
                    break;
                case HTTP:
                case IO:
                    stater.accept(
                        new RequestInfo(
                            TimeSource.INSTANCE.currentTimeMillis(),
                            HttpStatus.SC_INTERNAL_SERVER_ERROR,
                            start,
                            start,
                            0L,
                            0L));
                    break;
            }
        }

        if (logger != null) {
            logger.info(
                "Suggest failed with "
                    + e
                    + " in "
                    + requestTime + " ms");
        }

        this.callback.failed(e);
    }

    @Override
    public void cancelled() {
        long requestTime = System.currentTimeMillis() - start;

        if (this.logger != null) {
            this.logger.info("Request cancelled in " + requestTime + " ms");
        }

        this.callback.cancelled();
    }
}