package ru.yandex.msearch.proxy.api.async.suggest.united.rules;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import java.util.logging.Level;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.suggest.SortedSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;

import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;

import ru.yandex.msearch.proxy.api.async.suggest.united.SuggestAdapter;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.api.async.suggest.united.TargetWeight;
import ru.yandex.msearch.proxy.api.async.suggest.united.UnitedSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.united.WrappingUnitedSuggests;

import ru.yandex.msearch.proxy.config.ImmutableSuggestConfig;

public class AggregateSuggestRule
    implements SuggestRule<UnitedSuggests>
{
    private static final int DEFAULT_STATUS;

    static {
        DEFAULT_STATUS =
            (1 << Target.SENDER.ordinal()) | (1 << Target.RECEIVER.ordinal());
    }

    private final List<SuggestAdapter> adapters;
    private final ImmutableSuggestConfig config;
    private final Set<String> targetSet;

    public AggregateSuggestRule(
        final ImmutableSuggestConfig config,
        final List<SuggestAdapter> adapters)
    {
        this.adapters = adapters;
        this.config = config;
        this.targetSet = new HashSet<>();

        for (SuggestAdapter adapter: adapters) {
            targetSet.add(adapter.target().toString());
        }
    }

    @Override
    public void execute(
        final SuggestRequest<UnitedSuggests> request)
        throws HttpException
    {
        AggregationSessionCallback session =
            new AggregationSessionCallback(request, DEFAULT_STATUS);

        Set<String> targets = new HashSet<>(request.cgiParams().getAll("type"));
        if (targets.isEmpty()) {
            targets = targetSet;
        }

        for (SuggestAdapter adapter: adapters) {
            if (targets.contains(adapter.target().toString())) {
                adapter.execute(
                    request,
                    session.suggestCallback(adapter.target()));
                request.logger().info(
                    adapter.target().toString() + " executed");
            } else {
                session.skipTarget(adapter.target());
            }
        }

        session.done();
    }

    private static class SingleSuggestCallback
        implements FutureCallback<Suggests<? extends Suggest>>
    {
        private final Target target;
        private final AggregationSessionCallback callback;

        public SingleSuggestCallback(
            final Target target,
            final AggregationSessionCallback callback)
        {
            this.target = target;
            this.callback = callback;
        }

        @Override
        public void completed(final Suggests<? extends Suggest> suggests) {
            callback.completed(target, suggests);
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(target, e);
        }

        @Override
        public void cancelled() {
            callback.cancelled(target);
        }
    }

    private class AggregationSessionCallback {
        private final FutureCallback<? super WrappingUnitedSuggests> callback;
        private final int limit;
        private final PrefixedLogger logger;
        private final Map<Integer, Map<Target, List<Suggest>>> sm;
        private final Map<Target, TargetWeight> targetWeights;
        private int maxStatus;
        private int status;
        private int totalSuggests;
        private boolean allSent;
        private int onAir;

        public AggregationSessionCallback(
            final SuggestRequest<UnitedSuggests> request,
            final int status)
        {
            this.callback = request.callback();
            this.status = status;
            this.maxStatus = status;

            this.limit = request.requestParams().length();
            this.logger = request.logger();

            this.sm = new HashMap<>();
            this.totalSuggests = 0;
            this.onAir = 0;
            this.allSent = false;
            this.targetWeights = config.weights(request);
        }

        private SingleSuggestCallback suggestCallback(final Target target) {
            return this.suggestCallback(target, false);
        }

        private SingleSuggestCallback suggestCallback(
            final Target target,
            final boolean minor)
        {
            synchronized (this) {
                if (!minor) {
                    onAir += 1;
                }

                maxStatus |= 1 << target.ordinal();
            }

            return new SingleSuggestCallback(target, this);
        }

        private void skipTarget(final Target target) {
            status |= 1 << target.ordinal();
            maxStatus |= 1 << target.ordinal();
        }

        private void done() {
            boolean completed = false;
            synchronized (this) {
                allSent = true;
                if (onAir == 0) {
                    completed = true;
                }
            }

            if (completed) {
                completed();
            }
        }

        public void completed(
            final Target target,
            final Suggests<? extends Suggest> suggests)
        {
            boolean completed = false;
            logger.info("Completed " + target);
            synchronized (this) {
                status |= 1 << target.ordinal();
                this.totalSuggests += suggests.size();

                for (Suggest s: suggests) {
                    TargetWeight weight = targetWeights.get(s.target());
                    Integer gw = weight.groupWeight();

                    Map<Target, List<Suggest>> gm = sm.get(gw);

                    if (gm == null) {
                        gm = new HashMap<>();
                        sm.put(gw, gm);
                    }

                    List<Suggest> sl = gm.get(s.target());

                    if (sl == null) {
                        sl = new ArrayList<>();
                        gm.put(s.target(), sl);
                    }

                    sl.add(s);
                }

                if (--onAir == 0 && allSent) {
                    completed = true;
                }
            }

            logger.info(target + " completed, current now status " + status);

            if (completed) {
                completed();
            }
        }

        private void fill(
            final SortedSuggests<Suggest> result,
            final List<Iterator<Suggest>> iterators)
        {
            while (!result.limitReached()) {
                boolean aliveIterators = false;
                for (Iterator<? extends Suggest> it : iterators) {
                    if (it.hasNext()) {
                        result.add(it.next());
                        aliveIterators = true;
                    }
                }

                if (!aliveIterators) {
                    break;
                }
            }
        }

        public void completed() {
            SortedSuggests<Suggest> result = new SortedSuggests<>(
                new ScoreComparator(targetWeights),
                limit,
                true);

            this.logger.info(
                "Got total suggests " + totalSuggests + " limit is " + limit);

            int resultStatus;

            synchronized (this) {
                List<Integer> weights = new ArrayList<>(sm.keySet());
                weights.sort(Comparator.reverseOrder());

                List<Iterator<Suggest>> spareIterators = new ArrayList<>();

                for (Integer weight: weights) {
                    Map<Target, List<Suggest>> targetMap = sm.get(weight);
                    List<Iterator<Suggest>> iterators =
                        new ArrayList<>(targetMap.size());

                    for (Map.Entry<Target, List<Suggest>> entry:
                        targetMap.entrySet())
                    {
                        logger.info(
                            "Target " + entry.getKey()
                                + " suggest size " + entry.getValue().size());

                        TargetWeight tw =
                            this.targetWeights.get(
                                entry.getKey());
                        if (tw.limit() >= 0) {
                            // hook for fixed size targets, like contacts
                            Iterator<Suggest> fsIt =
                                entry.getValue().iterator();

                            fill(
                                result,
                                Collections.singletonList(
                                    new LimitedIterator<>(fsIt, tw.limit())));

                            if (fsIt.hasNext()) {
                                // smth left in iterator
                                // we could use it if all other are null
                                spareIterators.add(fsIt);
                            }
                        } else {
                            iterators.add(entry.getValue().iterator());
                        }
                    }

                    fill(result, iterators);

                    if (result.limitReached()) {
                        break;
                    }
                }

                if (status == maxStatus) {
                    resultStatus = -1;
                } else {
                    resultStatus = status;
                }

                if (!result.limitReached() && !spareIterators.isEmpty()) {
                    fill(result, spareIterators);
                }
            }

            callback.completed(
                new WrappingUnitedSuggests(result, resultStatus));
        }

        public void failed(
            final Target target,
            final Exception e)
        {
            logger.log(Level.INFO, "Suggest for " + target + " failed", e);

            boolean completed = false;
            synchronized (this) {
                if (--onAir == 0 && allSent) {
                    completed = true;
                }
            }

            if (completed) {
                completed();
            }
        }

        public void cancelled(final Target target) {
            logger.info("Suggest for " + target + " cancelled");
            boolean completed = false;
            synchronized (this) {
                if (--onAir == 0 && allSent) {
                    completed = true;
                }
            }

            if (completed) {
                completed();
            }
        }
    }

    private static class ScoreComparator implements Comparator<Suggest> {
        private Map<Target, TargetWeight> weights;

        public ScoreComparator(final Map<Target, TargetWeight> weights) {
            this.weights = weights;
        }

        @Override
        public int compare(final Suggest o1, final Suggest o2) {
            double score1 = o1.calcScore(weights);
            double score2 = o2.calcScore(weights);
            return Double.compare(score1, score2);
        }
    }

    private static class LimitedIterator<T> implements Iterator<T> {
        private final Iterator<T> wrapped;
        private int left;

        public LimitedIterator(final Iterator<T> wrapped, final int limit) {
            this.wrapped = wrapped;
            this.left = limit;
        }

        public Iterator<T> wrapped() {
            return wrapped;
        }

        @Override
        public boolean hasNext() {
            return wrapped.hasNext() && (left > 0);
        }

        @Override
        public T next() {
            if (left > 0) {
                left -= 1;
                return wrapped.next();
            }

            return null;
        }

        @Override
        public void remove() {
            wrapped.remove();
        }
    }
}
