package ru.yandex.msearch.proxy.api.async.suggest.united.rules;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.msearch.proxy.api.async.suggest.EmptyUnitedSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;

import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack.QL;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestMultiLanguagePack;

import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;

import ru.yandex.msearch.proxy.api.async.suggest.united.Target;

import ru.yandex.msearch.proxy.api.async.suggest.united.UnitedSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.united.WrappingUnitedSuggests;
import ru.yandex.parser.uri.CgiParams;

public class FilterSuggestRule implements SuggestRule<UnitedSuggests> {
    private static final SuggestLanguagePack LANGUAGE
        = SuggestMultiLanguagePack.INSTANCE;

    private static final List<String> DEFAULT_TYPES;
    private static final List<String> EXCLUDE_TYPES_VALUES;

    private static final int LARGE_REQUEST_LENGTH = 70;

    private static final List<String> LARGE_REQUEST_DEFAULT_TYPES;

    private static final Target[] LARGE_REQUEST_DEFAULT_TARGETS =
        new Target[]{Target.MAIL, Target.SUBJECT, Target.UNITED, Target.QL};

    static {
        List<String> types = new ArrayList<>();
        for (Target target: Target.values()) {
            types.add(target.toString());
        }

        DEFAULT_TYPES = Collections.unmodifiableList(types);
        List<String> excludeTypes = new ArrayList<>();
        for (Target target: Target.values()) {
            excludeTypes.add("type_" + target.toString());
        }
        excludeTypes.add(Target.HISTORY.toString());
        EXCLUDE_TYPES_VALUES = Collections.unmodifiableList(excludeTypes);

        List<String> largeRequestTypes = new ArrayList<>();
        for (Target target: LARGE_REQUEST_DEFAULT_TARGETS) {
            largeRequestTypes.add(target.toString());
        }
        LARGE_REQUEST_DEFAULT_TYPES =
            Collections.unmodifiableList(largeRequestTypes);
    }

    private final SuggestRule<UnitedSuggests> next;

    public FilterSuggestRule(final SuggestRule<UnitedSuggests> next) {
        this.next = next;
    }

    private List<String> excludedTypes(
        final CgiParams params)
        throws HttpException
    {
        List<String> filteredTypes = new ArrayList<>();
        for (String suggestType : EXCLUDE_TYPES_VALUES) {
            if (!params.getBoolean(suggestType, true)) {
                filteredTypes.add(suggestType);
            }
        }

        return filteredTypes;
    }

    private List<String> forcedTypes(
        final CgiParams params)
        throws HttpException
    {
        String types = params.getString("types", "").trim();
        if (!types.isEmpty()) {
            params.remove("type");
            for (String type: types.split(",")) {
                params.add("type", type);
            }
        }

        return params.getAll("type");
    }

    @Override
    public void execute(
        final SuggestRequest<UnitedSuggests> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();

        List<String> forcedTypes = forcedTypes(params);
        List<String> excludedTypes = excludedTypes(params);

        String requestTxt = params.getString("request", "");
        int colon = colonIndex(requestTxt);

        QL token =
            extractToken(requestTxt, colon);

        List<String> updatedTypes = new ArrayList<>();

        if (token != null) {
            switch (token) {
                case FROM:
                case TO:
                case BCC:
                case CC:
                    updatedTypes.add(Target.CONTACT.toString());
                    break;
                case SUBJECT:
                    updatedTypes.add(Target.SUBJECT.toString());
                    break;
                case FOLDER:
                    updatedTypes.add(Target.FOLDER.toString());
                    break;
                case LABEL:
                    updatedTypes.add(Target.LABEL.toString());
                    break;
                case FILTER:
                    updatedTypes.add(Target.CATEGORY.toString());
                    break;
            }
        }

        List<String> suggestTypes = new ArrayList<>();
        if (forcedTypes.isEmpty()) {
            suggestTypes.addAll(DEFAULT_TYPES);
        } else {
            suggestTypes.addAll(forcedTypes);
        }

        suggestTypes.removeAll(excludedTypes);

        if (requestTxt.length() >= LARGE_REQUEST_LENGTH) {
            request.session().logger().info(
                "Large request: " + requestTxt.length()
                + ", ignore all targets except " + LARGE_REQUEST_DEFAULT_TYPES);
            suggestTypes.retainAll(LARGE_REQUEST_DEFAULT_TYPES);
        }

        if (updatedTypes.isEmpty()) {
            if (suggestTypes.isEmpty()) {
                request.session().logger()
                    .info("All targets filtered out, empty suggest");
                request.callback().completed(new EmptyUnitedSuggests());
                return;
            }

            params = new CgiParams(params);
            params.remove("type");
            for (String type: suggestTypes) {
                params.add("type", type);
            }
            this.next.execute(request.withCgiParams(params));
            return;
        }

        updatedTypes.retainAll(suggestTypes);

        if (updatedTypes.isEmpty()) {
            request.session().logger()
                .info("All targets filtered out, empty suggest");
            request.callback().completed(new EmptyUnitedSuggests());
            return;
        }

        params = new CgiParams(params);
        params.remove("type");
        for (String type: updatedTypes) {
            params.add("type", type);
        }

        String tokenRequest = "";
        String prefixRequest = requestTxt;
        if (colon < requestTxt.length() - 1) {
            tokenRequest = requestTxt.substring(colon + 1);
            prefixRequest = requestTxt.substring(0, colon + 1);
        }

        params.replace("request", tokenRequest.trim());
        params.replace("ql", token.toString());
        // we do not need prefixes or smth like that
        params.replace("pure", "true");

        request.session().logger().info(
            "Suggesting for token " + updatedTypes
                + " request " + tokenRequest);

        this.next.execute(
            request.with(
                params,
                new FilterSuggestCallback(request.callback(), prefixRequest)));
    }

    public static int colonIndex(final String request) {
        int wsPos = -1;
        for (int i = request.length() - 1; i >= 0; i--) {
            char c = request.charAt(i);
            if (Character.isWhitespace(c) || Character.isSpaceChar(c)) {
                if (wsPos != -1) {
                    break;
                }

                wsPos = i;
            }

            if (c == ':') {
                if (wsPos != -1 && wsPos != i + 1) {
                    break;
                }

                return i;
            }
        }

        return -1;
    }

    public static QL extractToken(final String request, int colon) {
        if (colon < 0) {
            return null;
        }

        int lastSpace = -1;
        for (int i = colon - 1; i >= 0; i--) {
            if (Character.isWhitespace(request.charAt(i))) {
                lastSpace = i;
                break;
            }
        }

        // if we have separate colon, just skip
        if (lastSpace + 1 == colon) {
            return null;
        }

        String token =
            request.substring(lastSpace + 1, colon).toLowerCase(Locale.ROOT);

        return LANGUAGE.key(token);
    }

    private static class FilterSuggestCallback
        implements FutureCallback<UnitedSuggests>
    {
        private final String prefix;
        private final FutureCallback<? super WrappingUnitedSuggests> cb;

        public FilterSuggestCallback(
            final FutureCallback<? super WrappingUnitedSuggests> cb,
            final String prefix)
        {
            this.prefix = prefix;
            this.cb = cb;
        }

        @Override
        public void completed(final UnitedSuggests suggests) {
            ListSuggests result = new ListSuggests(suggests.size());

            for (Suggest suggest: suggests) {
                result.add(suggest.withPrefix(prefix));
            }

            cb.completed(new WrappingUnitedSuggests(result, suggests.status()));
        }

        @Override
        public void failed(final Exception e) {
            cb.failed(e);
        }

        @Override
        public void cancelled() {
            cb.cancelled();
        }
    }

    private static class ListSuggests implements Suggests<Suggest> {
        private final List<Suggest> suggests;

        public ListSuggests(final int size) {
            this.suggests = new ArrayList<>(size);
        }

        public void add(final Suggest suggest) {
            this.suggests.add(suggest);
        }

        @Override
        public int size() {
            return this.suggests.size();
        }

        @Override
        public Iterator<Suggest> iterator() {
            return this.suggests.iterator();
        }
    }
}
