package ru.yandex.msearch.proxy.api.async.suggest.united.rules;

import java.util.Iterator;

import com.google.common.collect.Iterators;
import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.FilterFutureCallback;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequestParams;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;

import ru.yandex.msearch.proxy.api.async.suggest.SuggestWeights;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.api.async.suggest.united.UnitedSuggests;

import ru.yandex.msearch.proxy.config.ImmutableSuggestConfig;

import ru.yandex.parser.uri.CgiParams;

public class TimeoutSuggestRule implements SuggestRule<UnitedSuggests> {
    private final SuggestRule<UnitedSuggests> next;
    private final ImmutableSuggestConfig config;

    public TimeoutSuggestRule(
        final AsyncHttpServer server,
        final SuggestRule<UnitedSuggests> next)
    {
        this.next = next;
        this.config = server.config().suggestConfig();
    }

    @Override
    public void execute(
        final SuggestRequest<UnitedSuggests> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();

        if (!params.getBoolean("twoSteps", config.twoStepsRequest())) {
            request.logger().info("Two steps disabled");
            this.next.execute(request);
            return;
        }

        if (!params.containsKey("status")) {
            long timeout = params.getLong("timeout", config.timeout());
            params.replace("timeout", String.valueOf(timeout));
            params.replace("fast", "1");
            request.logger().info("First request");
            this.next.execute(request.withCallback(
                new TimeoutCallback(
                    request.requestParams(), request.callback())));
        } else {
            int status = params.getInt("status", 0);
            params.remove("type");
            params.remove("types");
            for (Target target: Target.values()) {
                if ((status & (1 << target.ordinal())) == 0) {
                    params.add("type", target.toString());
                }
            }

            request.logger().info("Left " + params.getAll("type"));
            this.next.execute(request);
        }
    }

    private final class TimeoutCallback
        extends FilterFutureCallback<UnitedSuggests>
    {
        private final int expected = Target.CONTACT.ordinal();
        private final int limit;
        private final int reducedLimit;

        public TimeoutCallback(
            final SuggestRequestParams params,
            final FutureCallback<? super UnitedSuggests> callback)
        {
            super(callback);

            this.limit = params.length();
            this.reducedLimit =
                Math.max(
                    1,
                    this.limit - SuggestWeights.DEFAULT_WEIGHTS.get(Target.CONTACT).limit());
        }

        @Override
        public void completed(final UnitedSuggests suggests) {
            if ((suggests.status() & expected) == expected) {
                callback.completed(suggests);
            } else {
                callback.completed(
                    new LimitedWrappingUnitedSuggests(suggests, reducedLimit));
            }
        }
    }

    private static final class LimitedWrappingUnitedSuggests
        implements UnitedSuggests
    {
        private final UnitedSuggests wrapped;
        private final int limit;

        public LimitedWrappingUnitedSuggests(
            final UnitedSuggests wrapped,
            final int limit)
        {
            this.wrapped = wrapped;
            this.limit = limit;
        }

        @Override
        public int status() {
            return wrapped.status();
        }

        @Override
        public int size() {
            return Math.min(limit, wrapped.size());
        }

        @Override
        public Iterator<Suggest> iterator() {
            return Iterators.limit(wrapped.iterator(), limit);
        }
    }

}
