package ru.yandex.msearch.proxy.api.async.suggest.zero;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;

import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import org.apache.http.protocol.HttpContext;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;

import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.Side;
import ru.yandex.msearch.proxy.api.async.mail.searcher.ProducerParallelSearcher;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestSession;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParser;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggestBuilder;
import ru.yandex.msearch.proxy.api.suggest.Translit;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.json.fieldfunction.SumMapFunction;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.util.string.StringUtils;

public class ZeroSuggestHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final String NON_PEOPLE_FILTER_POSTFIX
        = "+AND+NOT+senders_message_types:4&sort=senders_last_contacted";

    private static final Set<String> MAIL_TYPES =
        new LinkedHashSet<>(
            Arrays.asList(
                "1",
                "2",
                "3",
                "4", "5", "6", "7", "8", "12", "13", "22", "27","28", "35", "36", "38", "42","40", "64", "65", "66", "67", "68"));
    private static final String ZERO_NAME = "0";
    private static final Set<String> EXCLUDE_SET =
        new HashSet<>(
            Arrays.asList(
                "No address",
                "no_address",
                "undisclosed-recipients:;",
                "No_address <>",
                "No address <>",
                "no_address <>",
                "no address <>",
                ">",
                "<"));

    private final AsyncHttpServer server;

    public ZeroSuggestHandler(final AsyncHttpServer server) {
        this.server = server;
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest httpRequest,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest httpRequest,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        SuggestSession session = new SuggestSession(server, exchange, context);

        CgiParams params = session.params();
        Long length = 10 * params.getLong(ProxyParams.LENGTH, 10L);
        Long uid = params.getLong(ProxyParams.UID);
        User user = new User(server.config().pgQueue(), new LongPrefix(uid));

        String request = params.getString("request", "");

        Set<String> requests;
        if (request == null || request.isEmpty()) {
            requests = Collections.emptySet();
        } else {
            requests = Translit.suggestSet(request, Side.parse(params));
            requests.removeIf((s) -> s == null || s.length() <= 0);
        }

        StringBuilder text = new StringBuilder();
        String urlBase =
            StringUtils.concat(
                "senders_uid_",
                user.prefix().toStringFast(),
                "_");

        if (requests.size() > 0) {
            text.append("url:");
            text.append('(');
            for (String translit: requests) {
                text.append(urlBase);
                text.append(translit.trim().toLowerCase(Locale.ROOT));
                text.append("*");
                text.append(" OR ");
            }

            text.setLength(text.length() - 4);
            text.append(")");
        }

        QueryConstructor query =
            new QueryConstructor("/search-async-mail-zero-suggest?");

        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());
        query.append("offset", "0");
        query.append("length", length / 2);
        query.append("get", "url,senders_message_types,senders_names,senders_last_contacted");

        ProducerParallelSearcher searcher =
            new ProducerParallelSearcher(server, session, user);
        DoubleFutureCallback<SearchResult, SearchResult> mfcb =
            new DoubleFutureCallback<>(
                new ZeroSuggestPrinter(session, user, requests));

        String peopleFilter;
        if (uid == 210714881 || uid == 227356512 || uid == 414728756) {
            peopleFilter =
                "senders_mail_type:people&dp=map_contains"
                    + "(senders_store_folders,spam+has_spam)"
                    + "&postfilter=has_spam+==+0&dp=map_contains"
                    + "(senders_store_folders,unsubscribe+unsub)"
                    + "&postfilter=unsub+==+0&dp=const(1536306192+cur_ts)"
                    + "&dp=fallback(senders_last_contacted,"
                    + "has_spam+last_contact_ts)&dp=sub(cur_ts,"
                    + "last_contact_ts+ts_diff)&dp=const(86400+secs_in_day)"
                    + "&dp=div(ts_diff,secs_in_day+days_passed)&dp=fallback"
                    + "(senders_from_read_count,has_spam+read_cnt)&dp=poly"
                    + "(read_cnt,2+read_score)&dp=sub(days_passed,"
                    + "read_cnt+score)&sort=score&asc";
            if (requests.size() > 0) {
                peopleFilter = "+AND+" + peopleFilter;
            }
            peopleFilter += "&sort=score";
        } else {
            if (requests.size() > 0) {
                peopleFilter = "+AND+senders_mail_type:people";
                peopleFilter += "&dp=map_contains(senders_store_folders,spam+has_spam)&postfilter=has_spam+==+0";
                peopleFilter += "&dp=map_contains(senders_store_folders,unsubscribe+unsub)&postfilter=unsub+==+0";
            } else {
                peopleFilter = "senders_mail_type:people";
                peopleFilter += "&dp=map_contains(senders_store_folders,spam+has_spam)&postfilter=has_spam+==+0";
                peopleFilter += "&dp=map_contains(senders_store_folders,unsubscribe+unsub)&postfilter=unsub+==+0";
            }

            peopleFilter += "&sort=senders_last_contacted";
        }

        query.append("text", text.toString());
        StringBuilder queryBase = query.sb();
        queryBase.append(peopleFilter);
        session.logger().info("query " + queryBase.toString());
        searcher.search(
            new BasicAsyncRequestProducerGenerator(queryBase.toString()),
            mfcb.first());

        queryBase.setLength(queryBase.length() - peopleFilter.length());
        if (requests.size() > 0) {
            queryBase.append(NON_PEOPLE_FILTER_POSTFIX);
        } else {
            queryBase.append("senders_uid:");
            queryBase.append(user.prefix());
            queryBase.append(NON_PEOPLE_FILTER_POSTFIX);
        }

        searcher.search(
            new BasicAsyncRequestProducerGenerator(queryBase.toString()),
            mfcb.second());
    }

    private static final class ZeroSuggestPrinter
        extends AbstractProxySessionCallback<
        Map.Entry<SearchResult, SearchResult>>
    {
        private final User user;
        private final Set<String> requests;
        private final JsonType jsonType;
        private final long length;

        public ZeroSuggestPrinter(
            final ProxySession session,
            final User user,
            final Set<String> requests)
            throws BadRequestException
        {
            super(session);

            this.user = user;
            this.length =
                session.params().getInt(ProxyParams.LENGTH, 10);
            this.requests = requests;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        private ContactSuggest build(
            final SearchDocument doc,
            final boolean people)
        {
            String dateStr = doc.attrs().get("senders_last_contacted");

            if (dateStr == null) {
                session.logger()
                    .log(Level.WARNING, "Empty received_date " + doc);
                return null;
            }

            Map<String, Long> map = new LinkedHashMap<>();
            String mapStr =
                doc.attrs().getOrDefault(
                    MailIndexFields.SENDERS_MESSAGE_TYPES,
                    "");

            float peoplePcnt = 0;
            long ts;
            try {
                ts = Long.parseLong(dateStr);
                if (people) {
                    SumMapFunction.parseMap(map, mapStr);
                    long sum = 0L;
                    long peopleCnt = map.getOrDefault("4", 0L);
                    for (Map.Entry<String, Long> item: map.entrySet()) {
                        if (MAIL_TYPES.contains(item.getKey())) {
                            sum += item.getValue();
                        }
                    }

                    if (sum > 0) {
                        peoplePcnt = peopleCnt / sum;
                    }

                    session.logger().info("");
                }
            } catch (NumberFormatException nfe) {
                session.logger()
                    .log(Level.WARNING, "Bad received_date " + doc, nfe);
                return null;
            }

            if (people && peoplePcnt <= 0.5) {
                return null;
            }

            String url = doc.attrs().get("url");
            if (url == null) {
                session.logger()
                    .log(Level.WARNING, "No url " + doc);
                return null;
            }

            int sepIndex = url.lastIndexOf('_');
            if (sepIndex < 0) {
                session.logger()
                    .log(Level.WARNING, "No _ in url" + doc);
                return null;
            }

            String address = url.substring(sepIndex + 1).trim();

            String name = null;
            String names = doc.attrs().get("senders_names");
            if (names != null) {
                String[] split = names.split("\\n");
                int nameIndex = split.length - 1;

                while (nameIndex >= 0
                    && ((split[nameIndex] == null)
                    || split[nameIndex].isEmpty()
                    || ZERO_NAME.equalsIgnoreCase(split[nameIndex])))
                {
                    nameIndex--;
                }

                if (nameIndex >= 0) {
                    name = split[nameIndex];
                }
            }

            int atIndex = address.indexOf('@');
            if (atIndex <= 0
                || EXCLUDE_SET.contains(name)
                || EXCLUDE_SET.contains(address))
            {
                return null;
            }

            ContactParser.Email email;
            if (name == null) {
                email = ContactParser.parseEmail(address);
            } else {
                email = new ContactParser.Email(name, address);
            }

            if (email == null || EXCLUDE_SET.contains(email.address())) {
                return null;
            }

            ContactSuggestBuilder builder =
                ContactSuggestBuilder
                    .create(email)
                    .ts(ts);

            builder.highlight(requests);

            return builder.build();
        }

        private void write(
            final JsonWriter writer,
            final SearchResult result,
            final boolean people)
            throws IOException
        {
            int index = 0;
            for (SearchDocument document: result.hitsArray()) {
                ContactSuggest suggest = build(document, people);
                if (suggest != null) {
                    suggest.writeValue(writer);
                    index += 1;
                }

                if (index >= length / 2) {
                    break;
                }
            }
        }

        @Override
        public void completed(
            final Map.Entry<SearchResult, SearchResult> result)
        {
            DecodableByteArrayOutputStream out =
                new DecodableByteArrayOutputStream();
            CharsetEncoder encoder = session.acceptedCharset().newEncoder()
                .onMalformedInput(CodingErrorAction.REPLACE)
                .onUnmappableCharacter(CodingErrorAction.REPLACE);

            try (OutputStreamWriter outWriter = new OutputStreamWriter(out, encoder);
                 JsonWriter writer = jsonType.create(outWriter))
            {
                writer.startObject();
                writer.key("people");
                writer.startArray();
                write(writer, result.getKey(), true);
                writer.endArray();
                writer.key("other");
                writer.startArray();
                write(writer, result.getValue(), false);
                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                failed(ioe);
            }

            NByteArrayEntity entity =
                out.processWith(NByteArrayEntityFactory.INSTANCE);
            entity.setContentType(
                ContentType.APPLICATION_JSON.withCharset(
                    session.acceptedCharset())
                    .toString());
            session.response(HttpStatus.SC_OK, entity);
        }
    }
}
