package ru.yandex.msearch.proxy.api.chemodan;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;

import java.io.IOException;

import org.apache.http.HttpException;
import org.apache.http.client.*;
import org.apache.http.impl.client.*;
import org.apache.http.client.config.RequestConfig;

import ru.yandex.blackbox.BlackboxUserinfo;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.HttpServer.HttpParams;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.api.BadRequestException;
import ru.yandex.msearch.proxy.api.async.mail.chemodan.AttachShield;
import ru.yandex.msearch.proxy.api.async.mail.chemodan.AttachShieldMailPS3183;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.document.Document;
import ru.yandex.msearch.proxy.collector.LinkedHashCollector;
import ru.yandex.msearch.proxy.collector.Collector;
import ru.yandex.msearch.proxy.httpclient.CommonHttpClient;
import ru.yandex.msearch.proxy.mail.FolderList;
import ru.yandex.msearch.proxy.mail.FolderListFactory;
import ru.yandex.msearch.proxy.ora.WmiFilterSearchClient;
import ru.yandex.msearch.proxy.search.MailSearcher;

import ru.yandex.parser.config.ConfigException;

public class ChemodanMailSearcher implements ChemodanSearcher {
    private static final HashSet<String> manualKeys = new HashSet<>();
    private static final HashMap<String, String> folderAttrs = new HashMap<>();
    private static final HashMap<String, String> folderMetaAttrs =
        new HashMap<>();
    private static final HashSet<String> previewableExtentions =
        new HashSet<>();
    private static final HttpRequestRetryHandler httpRetryHandler =
        new DefaultHttpRequestRetryHandler(2, true);
    private static final Pattern NORMALIZE =
        Pattern.compile("(^[/\\\\]+|[/\\\\]+$)");

    private static String ATTACHES_URL;
    private static String CORP_ATTACHES_URL;
    private static int ATTACHES_TTL;
    private static boolean NEW_ATTACH_SID;
    private final Map<String, ChemodanDocument> foldersFidMap =
        new HashMap<>();
    private final Map<String, List<ChemodanDocument>> foldersParentMap =
        new HashMap<>();
    private final UserInfo userInfo;
    private final HttpParams params;
    private AttachShield shield;
    private final HttpServer.RequestContext ctx;
    private final int offset;
    private final int length;
    private final Attr sortAttr;
    private final boolean desc;

    private static HttpClient httpClient;

    public static void init(
        final ImmutableMsearchProxyConfig config)
        throws ConfigException
    {
        ATTACHES_URL = config.chemodanConfig().webattachUrl();
        CORP_ATTACHES_URL = config.chemodanConfig().corpWebattachUrl();
        ATTACHES_TTL = config.chemodanConfig().webattachTtl();
        NEW_ATTACH_SID = config.chemodanConfig().ps3183Shield();

        RequestConfig rqConfig = RequestConfig.custom()
            .setSocketTimeout(3000)
            .setConnectTimeout(2000)
            .setExpectContinueEnabled(false)
            .setStaleConnectionCheckEnabled(false)
            .build();
        httpClient = CommonHttpClient.httpClientBuilder
            .setDefaultRequestConfig(rqConfig)
            .setRetryHandler(httpRetryHandler)
            .build();
    }

    static {
        folderAttrs.put("fid", "id");
        folderAttrs.put("creation_time", "ctime");
        folderAttrs.put("name", "name");
        folderMetaAttrs.put("parent_fid", "parentId");
        folderMetaAttrs.put("parent_id", "parentId");
        folderMetaAttrs.put("special", "systemLabel");
        folderMetaAttrs.put("symbol", "systemLabel");

//	manualKeys.add( "id" );
        previewableExtentions.add("bmp");
        previewableExtentions.add("jpg");
        previewableExtentions.add("jpeg");
        previewableExtentions.add("png");
        previewableExtentions.add("gif");
        previewableExtentions.add("tif");
        previewableExtentions.add("tiff");
    }

    public ChemodanMailSearcher(
        final HttpServer.RequestContext ctx,
        final HttpParams params,
        final UserInfo userInfo,
        final int offset,
        final int length,
        final Attr sortAttr,
        final boolean desc)
    {
        this.userInfo = userInfo;
        this.params = params;
        this.ctx = ctx;
        this.offset = offset;
        this.length = length;
        this.sortAttr = sortAttr;
        this.desc = desc;
    }

    public UserInfo userInfo() {
        return userInfo;
    }

    private static String normalizePath(final String path) {
        return NORMALIZE.matcher(path.trim()).replaceAll("");
    }

    private static boolean hasPrefix(final String path, final String prefix) {
        int length = prefix.length();
        if (path.length() > length + 1) {
            char c = path.charAt(length);
            return c == ':' && path.startsWith(prefix);
        }
        return false;
    }

    private static boolean isFolderPath(final String path) {
        return hasPrefix(normalizePath(path), "folder");
    }

    public static String encodeFolderId(final String fid) {
        return "folder:" + fid;
    }

    private static String decodeId(final String path, final String type)
        throws BadRequestException
    {
        String normalized = normalizePath(path);
        if (!hasPrefix(normalized, type)) {
            throw new BadRequestException(
                "Invalid " + type + " id: '" + path
                    + "'. Expected format is 'file:\\d+' or 'folder:\\d+'");
        }
        return normalized.substring(type.length() + 1);
    }

    private static String decodeFolderId(final String fid)
        throws BadRequestException
    {
        return decodeId(fid, "folder");
    }

    public static String encodeFileId(final String fid) {
        return "file:" + fid;
    }

    public static String decodeFileId(final String fid)
        throws BadRequestException
    {
        return decodeId(fid, "file");
    }

    private void getFolders() throws ApiException {
        try {
            FolderList folderList = FolderListFactory.create(ctx, userInfo);
            List<Map<String, String>> folders = folderList.getFolders();

            for (Map<String, String> folder : folders) {
                ChemodanDocument doc = new ChemodanDocument(true);
                doc.setAttr("id", folder.get("id"));
                String parentId = folder.get("parent_id");
                if (parentId == null) {
                    parentId = folder.get("parent_fid");
                }
                if (parentId != null) {
                    doc.setMetaAttr("parentId", parentId);
                } else {
                    parentId = "";
                }

                for (Map.Entry<String, String> entry : folder.entrySet()) {
                    String attr = entry.getKey();
                    String value = entry.getValue();
                    String docAttr = folderAttrs.get(attr);
                    if (docAttr != null) {
                        doc.setAttr(docAttr, value);
                    } else {
                        docAttr = folderMetaAttrs.get(attr);
                        if (docAttr != null) {
                            doc.setMetaAttr(docAttr, value);
                        }
                    }
                }

                String system = doc.getMetaAttr("systemLabel");
                if (system != null && !system.equals("0") &&
                    system.length() != 0)
                {
                    doc.setSystem(true);
                }
                foldersFidMap.put(doc.id(), doc);

                List<ChemodanDocument> list = foldersParentMap.get(parentId);
                if (list == null) {
                    list = new LinkedList<ChemodanDocument>();
                    foldersParentMap.put(parentId, list);
                }
                list.add(doc);
            }
            Iterator<String> iter = foldersFidMap.keySet().iterator();
            while (iter.hasNext()) {
                String fid = iter.next();
                ChemodanDocument doc = foldersFidMap.get(fid);
                if (doc == null) {
                    continue;
                }
                doc.setMetaAttr("fullname", getFullName(doc));
                doc.setMetaAttr("fullpath", getFullPath(doc));

                doc.setMetaAttr(
                    "has_folders",
                    foldersParentMap.containsKey(doc.id()) ? "1" : "0");
//	        System.err.println("GETFOLDERS: DOC=" + doc);
            }
        } catch (ApiException e) {
            throw e;
        } catch (MsearchProxyException e) {
            throw new ApiException(e);
        }
    }

    private void collectFilterFids(String path, int depth, List<String> fids)
    {
        if (depth < 0) {
            return;
        }
        List<ChemodanDocument> list = foldersParentMap.get(path);
        if (list == null) {
            return;
        }
        Iterator<ChemodanDocument> iter = list.iterator();
        while (iter.hasNext()) {
            ChemodanDocument folder = iter.next();
            fids.add(folder.id());
            collectFilterFids(folder.id(), depth - 1, fids);
        }
    }

    private final String getFullPath(ChemodanDocument doc)
    {
        String parentId = doc.parentId();
        String id = "/" + encodeFolderId(doc.id());
//	String id = encodeFolderId( doc.id() );
        if (parentId == null) {
            return id;
        }
        if (parentId.isEmpty()) {
            return id;
        }
        ChemodanDocument parent = foldersFidMap.get(parentId);
        if (parent == null) {
            return id;
        }
        return getFullPath(parent) + id;
    }

    private final String getFullName(ChemodanDocument doc)
    {
        String parentId = doc.parentId();
        String name = "/" + doc.getAttr("name");
        if (parentId == null) {
            return name;
        }
        if (parentId.isEmpty()) {
            return name;
        }
        ChemodanDocument parent = foldersFidMap.get(parentId);
        if (parent == null) {
            return name;
        }
        return getFullName(parent) + name;
    }

    private void collectFolders(
        ChemodanCollector collector,
        String path,
        List<String> fids,
        boolean withFilesOnly)
    {
        Iterator<String> iter = fids.iterator();
        while (iter.hasNext()) {
            String fid = iter.next();
            ChemodanDocument doc = foldersFidMap.get(fid);
            if (doc == null) {
                continue;
            }
            if (withFilesOnly && doc.getCount() <= 0) {
                continue;
            }
            doc.setAttr("id", encodeFolderId(doc.id()));


            if (path == null) {
                collector.collect(doc);
            } else if (fid.equals(path)) {
                collector.setCurrentFolder(doc);
            } else {
                if (doc.parentId() != null && !doc.parentId().equals(path)) {
                    continue;
                }
                collector.collect(doc);
            }
        }
    }

    private void findFolder(ChemodanCollector collector, String path)
    {
        ChemodanDocument doc = foldersFidMap.get(path);
        if (doc == null) {
            return;
        }

        doc.setMetaAttr(
            "has_folders",
            foldersParentMap.containsKey(doc.id()) ? "1" : "0");

        doc.setAttr("id", encodeFolderId(doc.id()));
        collector.collect(doc);
    }

    private static String applyFilter(String request)
    {
        String filter =
            "((disposition_type:inline AND "
                + "(headers:\"_$XHIDENX$_content-id:\" OR "
				+ "headers:content-id\\:*)) "
                + "OR attachname:narod_attachment_links.html "
                + "OR mimetype:application/pkcs7-signature "
                + "OR attachname:smime.p7s "
                + "OR headers:\"_$XHIDENX$_content-id:\" "
                + "OR headers:content-id\\:* "
                + "OR headers:\"_$XHIDENX$_content-location:\" "
                + "OR headers:content-location\\:* "
                + "OR headers:\"_$XHIDENX$_list-unsubscribe:\" "
                + "OR headers:list-unsubscribe\\:* "
                + "OR headers:\"_$XHIDENX$_list-id:\" "
                + "OR headers:list-id\\:* "
                + "OR headers:\"_$XHIDENX$_precedence: list\" "
                + "OR headers:\"_$XHIDENX$_precedence: List\" "
                + "OR headers:precedence\\:\\ \\ list* "
                + "OR headers:precedence\\:\\ List "
                + "OR headers:\"_$XHIDENX$_precedence: junk\" "
                + "OR headers:\"_$XHIDENX$_precedence: Junk\" "
                + "OR headers:precedence\\:\\ junk "
                + "OR headers:precedence\\:\\ Junk "
                + "OR headers:\"_$XHIDENX$_precedence: bulk\" "
                + "OR headers:\"_$XHIDENX$_precedence: Bulk\" "
                + "OR headers:precedence\\:\\ bulk "
                + "OR headers:precedence\\:\\ Bulk)";
        if (request == null || request.isEmpty()) {
            request = "attachname:*";
        }
        return '(' + request + ") AND NOT " + filter;
    }

    private static String makeFilesRequest(String request)
    {
        if (request == null || request.isEmpty()) {
            request = "attachname:*";
        } else if (request.contains("attachname:*")) {
            int or = request.indexOf(" OR ");
            if (or == -1) {
                request = request.toLowerCase(Locale.ENGLISH).replace('ё', 'е');
            } else {
                request = request.substring(
                    0,
                    or).toLowerCase(Locale.ENGLISH).replace('ё', 'е')
                    + " OR "
                    + request.substring(
                    or + 4).toLowerCase(Locale.ENGLISH).replace('ё', 'е');
            }
        } else {
            request = request
                .replaceAll("[*\\\\(){}\\[\\]'?~\u00a0+:!^\"-]", " ")
                .trim();
            String asteriskRequest =
                "*"
                    + request
                    .replaceAll(" +", "*")
                    .toLowerCase(Locale.ENGLISH)
                    .replace('ё', 'е')
                    + '*';
            String andedRequest = "(" + request.replaceAll(" +", " AND ") + ')';
            request = "attachname:" + andedRequest
                + " OR attachname:" + asteriskRequest
                + " OR attachname_keyword:" + asteriskRequest
                + " OR (attachname:* AND body_text,album,artist,author,"
                + "comment,composer,description,genre,keywords,subject,title,"
                + "attachtype:"
                + andedRequest + ')';
        }
        return applyFilter(request);
    }

    private final boolean isManualKey(String key)
    {
        if (manualKeys.contains(key)) {
            return true;
        }
        return false;
    }

    private final boolean checkAttach(Document doc)
    {
        String name = doc.getAttr("attachname");
        if (name == null) {
            return false;
        }
        return true;
    }

    public final String makeUrl(
        String name,
        long uid,
        String stid,
        String hid,
        String id) throws ApiException
    {
        StringBuilder sb = new StringBuilder();
        if (BlackboxUserinfo.corp(uid)) {
            sb.append(CORP_ATTACHES_URL);
        } else {
            sb.append(ATTACHES_URL);
        }
        sb.append(name);
        sb.append("?sid=");
        sb.append(shield.encodeSync(stid, hid, id, uid));
        sb.append("&name=");
        sb.append(name);
        sb.append("&ids=");
        sb.append(id);
        return new String(sb);
    }

    public static final boolean previewableFile(ChemodanDocument file)
    {
        String type = file.getAttr("type");
        if (type != null && type.startsWith("image")) {
            return true;
        }
        String extention = file.getAttr("name");
        if (extention == null) {
            return false;
        }
        int dotPos = extention.lastIndexOf('.');
        if (dotPos == -1) {
            return false;
        }
        if (dotPos == extention.length() - 1) {
            return false;
        }
        extention = extention.substring(dotPos + 1).toLowerCase();
        if (previewableExtentions.contains(extention)) {
            return true;
        }
        return false;
    }

    private final void collectMailDocument(
        final ChemodanCollector collector,
        final Document doc,
        final HashSet<String> folders,
        final String path,
        boolean search) throws ApiException
    {
        String fid = doc.getAttr("fid");
        ChemodanDocument folder = foldersFidMap.get(fid);

        int count = 0;
        boolean collect = false;
        if (search) {
            if (folders.isEmpty() || folders.contains(fid)) {
                collect = true;
            }
        } else {
            if (fid.equals(path)) {
                collect = true;
            }
        }
//	if( tree && fid.equals(path) ) collect = true;
//	if( !tree ) collect = true;

        if (checkAttach(doc)) {
            count++;
            if (collect) {
                collector.collect(new MailWrappingChemodanDocument(
                    this,
                    doc,
                    null,
                    folder));
            }
        }

        if (doc.getMergedCount() > 0) {
            Iterator<Document> merged = doc.getMerged().iterator();
            while (merged.hasNext()) {
                Document m = merged.next();
                if (checkAttach(m)) {
                    count++;
                    if (collect) {
                        collector.collect(new MailWrappingChemodanDocument(
                            this,
                            m,
                            doc,
                            folder));
                    }
                }
            }
        }
        if (folder != null) {
            folder.incCount(count);
        }

    }

    private void collectFiles(
        ChemodanCollector collector,
        String path,
        String request,
        List<String> filterFids,
        boolean tree,
        boolean search,
        int toFind)
        throws ApiException, IOException, MsearchProxyException
    {
        MailSearcher msearch =
            new MailSearcher(ctx, userInfo.getMailDb(), userInfo.getUid());
        LinkedHashCollector col = new LinkedHashCollector("mid");
        msearch.setFids(filterFids);
        msearch.setImapSearch(true);
        msearch.setFields(
            "mid,received_date,attachname,attachsize_b,mimetype,url,"
                + sortAttr.luceneName());
        msearch.setRequestParam("outergroup", "url");
        msearch.setExcludeSpam(!"1".equals(params.get("spam")));
        msearch.setExcludeTrash(!"1".equals(params.get("trash")));
        msearch.setServerOrder(sortAttr.luceneName());
        msearch.setDesc(desc);
        msearch.search(col, request, toFind);
        Map<Document, Document> hits = col.hits();

        if (hits.size() > 0) {
            try {
                if (NEW_ATTACH_SID) {
                    shield = new AttachShieldMailPS3183(ATTACHES_TTL);

                    ctx.log.info(
                        "Creating and storing new version of shield PS-3183");
                } else {
                    shield = new AttachShieldOld(ATTACHES_TTL);
                }
            } catch (HttpException e) {
                throw new ApiException(e.getMessage(), e);
            }
        }

        Iterator<Document> docs = hits.keySet().iterator();
        if (!tree) {
            search = true;
        }
        while (docs.hasNext()) {
            collectMailDocument(
                collector,
                docs.next(),
                new HashSet<>(filterFids),
                path,
                search);
        }
    }

    public int collectAllFolders(ChemodanCollector collector, String path)
        throws ApiException
    {
        getFolders();
        List<String> filterFids = new LinkedList<String>();
        if (path.isEmpty()) {
            filterFids.addAll(foldersFidMap.keySet());
        } else {
            collectFilterFids(path, 9999, filterFids);
        }
//	Iterator<String>
//	collectFilterFids( path, 9999, filterFids );
        if (!path.isEmpty()) {
            filterFids.add(path);
        }
        collectFolders(collector, null, filterFids, false);
        return 0;
    }

    public void secondFilter(ChemodanCollector collector)
        throws MsearchProxyException
    {
        int found = 0;

        if (collector.hits().size() <= 0) {
            return;
        }

        ctx.log.info("Second ORACLE filter step");
        Map<ChemodanDocument, ChemodanDocument> hits = collector.hits();
        Iterator<ChemodanDocument> collectedIter = hits.keySet().iterator();

        Map<String, Document> toFilter = new HashMap<String, Document>();
        Map<String, List<ChemodanDocument>> toFilterCmp =
            new HashMap<String, List<ChemodanDocument>>();
        int i;
        for (i = 0; i < offset + length && collectedIter.hasNext(); i++) {
            if (i >= offset) {
                break;
            }
            ChemodanDocument cd = collectedIter.next();
//    	    if( cd.isFolder() ) continue;
        }
        if (i == offset + length) {
            ctx.log.info(
                "offset + length top of collected documents is folders. no "
					+ "need to filter");
            return;
        }

        List<ChemodanDocument> removeList = new ArrayList<ChemodanDocument>(10);

        while (found < length && collectedIter.hasNext()) {
            for (i = 0; i < length && collectedIter.hasNext(); i++) {
                ChemodanDocument cd = collectedIter.next();
                if (cd.isFolder()) {
                    continue;
                }
                MailWrappingChemodanDocument mwcd =
                    (MailWrappingChemodanDocument) cd;
                Document mailDoc = mwcd.getMailDocument();
                String mid = mailDoc.getAttr("mid");
                toFilter.put(mid, mailDoc);
                List<ChemodanDocument> list = toFilterCmp.get(mid);
                if (list == null) {
                    list = new ArrayList<ChemodanDocument>(10);
                    toFilterCmp.put(mid, list);
                }
                list.add(cd);
            }
            if (toFilter.size() > 0) {
                ctx.log.debug("second filter batch size: " + toFilter.size());
                WmiFilterSearchClient filter =
                    new WmiFilterSearchClient(ctx.getSessionId(), ctx);
                filter.setMailDb(userInfo.getMailDb());
                filter.setSuid(userInfo.getSuid());
                filter.setOrder("default");
                filter.setExcludeSpam(!"1".equals(params.get("spam")));
                filter.setExcludeTrash(!"1".equals(params.get("trash")));
                Collector secondCollector = new LinkedHashCollector("mid");
                filter.filter(toFilter, secondCollector);

                Map<Document, Document> successHits = secondCollector.hits();
                Iterator<Document> successIter =
                    successHits.keySet().iterator();
                while (successIter.hasNext()) {
                    Document successDoc = successIter.next();
                    List<ChemodanDocument> list =
                        toFilterCmp.get(successDoc.getAttr("mid"));
                    found += list.size();
                    toFilterCmp.remove(successDoc.getAttr("mid"));
                }

                if (toFilterCmp.size() > 0) {
                    int removed = 0;
                    for (Map.Entry<String, List<ChemodanDocument>> entry :
						toFilterCmp.entrySet()) {
                        List<ChemodanDocument> list = entry.getValue();
                        removeList.addAll(list);
                        removed += list.size();
                    }
                    ctx.log.debug(
                        "second filter has deleted " + toFilterCmp.size()
                            + " / " + removed + " documents from " + length
                            + " requested");
                }

                ctx.log.debug("second filter total found: " + found);

                toFilterCmp.clear();
                toFilter.clear();
            }
        }

        for (ChemodanDocument doc : removeList) {
            doc.setRemove(true);
            hits.remove(doc);
        }

    }

    public int search(
        ChemodanCollector collector,
        String path,
        boolean tree,
        String request,
        int toFind)
        throws ApiException, MsearchProxyException
    {
        if (path == null) {
            path = "";
        }
        if (!path.isEmpty()) {
            path = decodeFolderId(path);
        }
        if (request.equals("foldername:*")) //FIXME: this is an ugly hack
        {
            return collectAllFolders(collector, path);
        }
        getFolders();
        int depth = Integer.MAX_VALUE;

        if (tree) {
            depth = 0;
        }
        List<String> filterFids = new LinkedList<String>();
        boolean search = true;
        if (request.isEmpty()) {
            search = false;
        }

        collectFilterFids(path, depth, filterFids);
        if (!path.isEmpty()) {
            filterFids.add(path);
        }
        ctx.log.debug("FilterFids: " + filterFids.toString());

        try {

            if (!path.isEmpty()) {
                collectFiles(
                    collector,
                    path,
                    makeFilesRequest(request),
                    filterFids,
                    tree,
                    search,
                    toFind); //do not collect file for root list
            } else {
                if (!tree || !request.isEmpty()) {
                    collectFiles(
                        collector,
                        path,
                        makeFilesRequest(request),
                        filterFids,
                        tree,
                        search,
                        toFind); //do not collect file for root list
                }
            }

            if (tree) {
                if (!request.isEmpty()) {
                    //This is another one ugly hack as we must collect requested folder regardless of found files.
                    if (path.isEmpty()) {
                        collectFolders(collector, path, filterFids, true);
                    } else {
                        filterFids.remove(path);
                        List<String> pathList = new LinkedList<String>();
                        pathList.add(path);
                        collectFolders(collector, path, pathList, false);
                        collectFolders(collector, path, filterFids, true);
                    }
                } else {
                    collectFolders(collector, path, filterFids, false);
                }
            }

        } catch (IOException e) {
            throw new ApiException(e);
        }


        return 0;
    }

    public int info(ChemodanCollector collector, String path)
        throws ApiException, MsearchProxyException
    {
        try {
            if (isFolderPath(path)) {
                path = decodeFolderId(path);
                getFolders();
                List<String> filterFids = new LinkedList<String>();
                collectFilterFids(path, 1, filterFids);
                collectFiles(
                    collector,
                    "invalidid",
                    applyFilter(null),
                    filterFids,
                    true,
                    false,
                    9999999);
                findFolder(collector, path);
            } else {
                String decodedPath = decodeFileId(path);
                String mid = decodedPath.substring(0, decodedPath.indexOf('/'));
                List<String> filterFids = new LinkedList<String>();
                ChemodanCollector tempCollector =
                    new SortingCollector(Attr.ID, true, 10000);
                collectFiles(
                    tempCollector,
                    path,
                    "mid_p:" + mid,
                    filterFids,
                    false,
                    false,
                    9999999);
                Map<ChemodanDocument, ChemodanDocument> hits =
                    tempCollector.hits();
                Iterator<ChemodanDocument> iter = hits.keySet().iterator();
                while (iter.hasNext()) {
                    ChemodanDocument doc = iter.next();
                    if (doc.id().equals(encodeFileId(decodedPath))) {
                        collector.collect(doc);
                        break;
                    }
                }
            }
        } catch (IOException e) {
            throw new ApiException(e);
        }
        return 0;
    }
}
