package ru.yandex.msearch.proxy.api.mail;

import java.nio.charset.StandardCharsets;

import java.util.Iterator;
import java.util.TreeMap;
import java.util.Map;
import java.util.Arrays;
import java.io.PrintStream;
import java.io.OutputStreamWriter;
import java.io.IOException;
import org.json.JSONWriter;
import org.json.JSONException;
import java.net.URLEncoder;
import com.tecnick.htmlutils.htmlentities.HTMLEntities;

import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.dom.address.MailboxList;
import org.apache.james.mime4j.field.address.LenientAddressParser;
import org.apache.james.mime4j.stream.ParserCursor;
import org.apache.james.mime4j.util.ByteSequence;
import org.apache.james.mime4j.util.ContentUtil;

import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.document.Document;
import ru.yandex.msearch.proxy.collector.Collector;

public class JSONMailPrinter
{
int perPage;
private int totalCount;
private MailSearchOptions options;
private byte[] decodeBuf = new byte[20];

    public JSONMailPrinter( int perPage )
    {
	this.perPage = perPage;
    }
    
    public void print( MailSearchResult result, PrintStream ps, int page ) throws IOException, JSONException
    {
        Collector collector = result.collector();
	Map<Document,Document> hits = collector.hits();
	totalCount = hits.size();
        options = result.options();
	Iterator<Document> iter = hits.keySet().iterator();

	printJson( ps, iter, page );
    }
    
    private void printJson( PrintStream ps, Iterator<Document> iter, int page ) throws IOException, JSONException
    {
        OutputStreamWriter writer = new OutputStreamWriter( ps );
        JSONWriter jw = new JSONWriter( writer );
        try
        {
            jw.object();
            jw.key( "threaded" );
            jw.value( "" );
            printDetails( jw, page );
            printList( jw, iter, (page - 1) * perPage, perPage );
            jw.endObject();
        }
        finally
        {
            if( writer != null )
            {
                writer.flush();
            }
        }
    }

    private void printDetails( JSONWriter jw, int page ) throws JSONException
    {
        jw.key( "details" );
        jw.object();
            jw.key( "crc32" );
            jw.value( "0" );

            jw.key( "pager" );
            jw.object();
                jw.key( "items-per-page" );
                jw.value( perPage );

                jw.key( "prev" );
                if( page > 1 )
                {
                    jw.object();
                        jw.key( "n" );
                        jw.value( page - 1 );
                    jw.endObject();
                }
                else
                {
                    jw.value( "" );
                }

                jw.key( "next" );
                if( totalCount > page * perPage )
                {
                    jw.object();
                        jw.key( "n" );
                        jw.value( page + 1 );
                    jw.endObject();
                }
                else
                {
                    jw.value( "" );
                }
            jw.endObject();
            options.toJSONWriter(jw);
        jw.endObject();
    }

    private void printList( JSONWriter jw, Iterator<Document> iter, int offset, int length ) throws JSONException
    {
        jw.key( "list" );
        jw.object();
        jw.key( "message" );
        jw.array();
	int i = 0;
	while( iter.hasNext() )
	{
	    Document doc = iter.next();
	    if( i++ < offset ) continue;
	    if( i > length + offset ) break;
	    printMessage( jw, doc );
	}
	jw.endArray();
	jw.endObject();
    }

    private void printMessage( JSONWriter jw, Document doc ) throws JSONException
    {
        jw.object();

        jw.key("id");
        jw.value( doc.getAttr("mid") );

        jw.key( "message" );
        jw.object();
            jw.key( "id" );
            jw.value( doc.getAttr("threadId") );
        jw.endObject();

        jw.key( "folder" );
        jw.object();
            jw.key( "id" );
            jw.value( doc.getAttr("fid") );
        jw.endObject();

        jw.key( "date" );
        jw.object();
            jw.key( "utc_timestamp" );
            jw.value( doc.getAttr("received_date") );
        jw.endObject();

        jw.key( "size" );
        jw.value( doc.getAttr("size") );

        jw.key( "last_status" );
        jw.value( "" );

        jw.key( "subject" );
        jw.value( doc.getAttr("subject") );

        jw.key( "firstline" );
        jw.value( doc.getAttr("firstline") );

        printEmail( jw, doc.getAttr("to"), "to" );
        printEmail( jw, doc.getAttr("from"), "from" );
        printEmail( jw, doc.getAttr("replyTo"), "reply-to" );


        jw.key( "types" );
        jw.value( doc.getAttr("types") );


        String labels = doc.getAttr("labels");
        if( labels != null )
        {
            jw.key( "labels" );
            jw.object();
                jw.key("label");
                jw.array();
                    for( String label : labels.split("\n") )
                    {
                        jw.object();
                            jw.key( "id" );
                            jw.value( label );
                        jw.endObject();
                    }
                jw.endArray();
            jw.endObject();
        }

        jw.key( "other" );
        jw.object();
	    Map<String,String> attrs = doc.getAttrs();
	    Iterator<Map.Entry<String,String>> attrIter = attrs.entrySet().iterator();
	    while( attrIter.hasNext() )
	    {
		Map.Entry<String,String> entry = attrIter.next();
		if( entry.getKey().equals("_relevant_") ) continue;
	        jw.key( entry.getKey() );
	        jw.value( entry.getValue() );
	    }
	jw.endObject();

        jw.endObject();
    }

    private void printEmail( JSONWriter jw, String email, String fieldName ) throws JSONException
    {
        if( email == null || email.length() == 0 ) return;
        jw.key( fieldName );

        MailboxList mboxList = parseMailboxList( email );
        if( mboxList.size() > 1 )
        {
            jw.array();
            for( Mailbox mbox : mboxList )
            {
                printMailbox( jw, mbox );
            }
            jw.endArray();
        }
        else if( mboxList.size() == 1 )
        {
            printMailbox( jw, mboxList.get(0) );
        }
    }

    private void printMailbox( JSONWriter jw, Mailbox mbox ) throws JSONException
    {
        jw.object();
        jw.key( "name" );

        if( mbox.getName() != null && mbox.getName().length() > 0 )
        {
            jw.value( decodeUTF8(mbox.getName()) );
        }
        else
        {
            jw.value( mbox.getAddress() );
        }
        jw.key( "email" );
        jw.value( mbox.getAddress() );

        jw.endObject();
    }

    private MailboxList parseMailboxList( String str )
    {
        ByteSequence raw = ContentUtil.encode(StandardCharsets.UTF_8, str);
        ParserCursor cursor = new ParserCursor(0, raw.length());
        return LenientAddressParser.DEFAULT.parseAddressList(raw, cursor).flatten();
    }

    private String decodeUTF8( String encoded )
    {
        if( encoded.length() > decodeBuf.length )
        {
            decodeBuf = new byte[encoded.length() * 2];
        }
        for( int i = 0; i < encoded.length(); i++ )
        {
            decodeBuf[i] = (byte)encoded.charAt(i);
        }
        return new String( decodeBuf, 0, encoded.length(), StandardCharsets.UTF_8 );
    }

    private String encode( String in )
    {
	return HTMLEntities.htmlAngleBrackets( HTMLEntities.htmlAmpersand( in ) );
    }
}

