package ru.yandex.msearch.proxy.api.mail;

import java.io.IOException;
import java.io.PrintStream;
import java.net.SocketTimeoutException;
import java.nio.charset.CharacterCodingException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;
import java.util.regex.Pattern;

import org.apache.http.nio.reactor.IOReactorException;
import org.json.JSONException;
import org.json.JSONWriter;

import ru.yandex.erratum.ErratumClient;
import ru.yandex.erratum.ErratumConfigBuilder;
import ru.yandex.erratum.ErratumResult;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.server.BaseServerConfigBuilder;
import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.api.NotImplementedException;
import ru.yandex.msearch.proxy.api.mail.rules.ForbiddenRequestsRule;
import ru.yandex.msearch.proxy.api.mail.rules.MailSearchRule;
import ru.yandex.msearch.proxy.api.mail.rules.MisspellRule;
import ru.yandex.msearch.proxy.api.mail.rules.RewriteScopeRule;
import ru.yandex.msearch.proxy.api.mail.rules.SearchRule;
import ru.yandex.msearch.proxy.api.mail.rules.SoCheckRule;
import ru.yandex.msearch.proxy.api.mail.rules.StoreSearchRequestAdapter;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.socheck.SoCheckFactory;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class Mail
{
    private static SearchRule searchRule;
    private static int MAX_ATTACHMENTS_TO_FILTER;

    public static void init(
        final IniConfig config,
        final ImmutableMsearchProxyConfig proxyConfig)
        throws ConfigException, IOReactorException
    {
        MAX_ATTACHMENTS_TO_FILTER =
            config.getInt("MAX_ATTACHMENTS_TO_FILTER", 100);
        SearchRule rule = new MailSearchRule();
        SharedConnectingIOReactor reactor = new SharedConnectingIOReactor(
            proxyConfig,
            proxyConfig.dnsConfig());
        reactor.start();
        if (proxyConfig.erratumConfig() != null) {
            ErratumClient erratumClient =
                new ErratumClient(reactor, proxyConfig.erratumConfig());
            erratumClient.start();
            MisspellRule next = new MisspellRule(erratumClient);
            next.setNext(rule);
            rule = next;
        }

        if (proxyConfig.indexSearchRequests()
            && proxyConfig.producerClientConfig() != null)
        {
            rule =
                StoreSearchRequestAdapter.create(rule, reactor, proxyConfig);
        }

        if (proxyConfig.soCheckConfig() != null) {
            try {
                SoCheckFactory soCheckFactory =
                    new SoCheckFactory(reactor, proxyConfig.soCheckConfig());
                soCheckFactory.start();
                SoCheckRule next = new SoCheckRule(soCheckFactory);
                next.setNext(rule);
                rule = next;
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        Pattern forbiddenRequests = proxyConfig.forbiddenRequests();
        if (forbiddenRequests != null) {
            ForbiddenRequestsRule next =
                new ForbiddenRequestsRule(forbiddenRequests);
            next.setNext(rule);
            rule = next;
        }

        RewriteScopeRule rewriteScopeRule = new RewriteScopeRule();
        rewriteScopeRule.setNext(rule);
        rule = rewriteScopeRule;

        searchRule = rule;
    }

    private HttpServer.RequestContext ctx;
    private String requestString;
    private String requestType;
    private HttpServer.HttpParams params;
    private HttpServer.HttpHeaders headers;
    private PrintStream ps;

    public Mail( HttpServer.RequestContext ctx, String request, String requestType, HttpServer.HttpParams params, HttpServer.HttpHeaders headers, PrintStream ps )
    {
        this.ctx = ctx;
        this.requestString = request;
        this.requestType = requestType;
        this.params = params.copy();
        this.params.remove("attachments");
        this.params.remove("get_fields");
        this.headers = headers;
        this.ps = ps;
    }

    public int dispatch() throws ApiException, MsearchProxyException
    {
        if( requestString.startsWith("/api/mail/search") )
        {
            return search();
        }
        else if( requestString.startsWith("/api/mail/news") )
        {
            return news();
        }
        else if( requestString.startsWith("/api/mail/attachments_count") )
        {
            return attachmentsCount();
        }
        else
        {
            throw new NotImplementedException( requestString );
        }
    }

    public int news() throws ApiException, MsearchProxyException {
        HttpServer.HttpParams searchParams = params.copy();
        searchParams.add("message_type", "13");
        searchParams.add("getfields", "hdr_from");
        searchParams.add("imap", "1");
        MailSearchResult result = searchRule.execute(ctx, searchParams, -1);

        if (result == null) {
            return 0;
        }
        News news = new News(ctx);
        news.collect(result.collector());
        try {
            news.print(ps);
        } catch (IOException | JSONException e) {
            throw new ApiException(e);
        }
        return result.collector().getCount();
    }

    public int search() throws ApiException, MsearchProxyException {
        int page = 1;
        int perPage = 200;
        if( params.get("page") != null )
        {
            page = Integer.parseInt(params.get("page"));
            if( page <= 0 ) page = 1;
        }
        if( params.get("per_page") != null )
        {
            perPage = Integer.parseInt(params.get("per_page"));
            if( perPage <= 0 ) perPage = 200;
        }
        final int toFind = page * perPage + 1;
        MailSearchResult result = searchRule.execute(ctx, params, toFind);
        return print(result, page, perPage);
    }

    private int attachmentsCount() throws ApiException, MsearchProxyException {
        HttpServer.HttpParams searchParams = params.copy();
        searchParams.add("has_attachments", "1");
        searchParams.add("attachments", Boolean.TRUE.toString());
        searchParams.add("imap", "1");
        MailSearchResult result = searchRule.execute(ctx, searchParams,
            MAX_ATTACHMENTS_TO_FILTER);
        int found = 0;
        if (result != null) {
            found = result.collector().getCount();
            if (found >= MAX_ATTACHMENTS_TO_FILTER) {
                found = result.collector().getTotalCount();
            }
        }
        ctx.ps().println(found);
        return found;
    }

    private int print(
        final MailSearchResult result,
        final int page,
        final int perPage)
        throws ApiException
    {
        if (result == null) {
            return 0;
        }
        JSONMailPrinter mout = new JSONMailPrinter(perPage);
        ctx.setContentType("application/json");
        try {
            mout.print(result, ps, page);
        } catch (IOException | JSONException e) {
            throw new ApiException(e);
        }
        return result.collector().getCount();
    }
}

