package ru.yandex.msearch.proxy.api.mail;

import java.io.IOException;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.List;
import java.util.stream.Collectors;

import org.json.JSONException;
import org.json.JSONWriter;

import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.msearch.proxy.api.async.mail.Product;
import ru.yandex.msearch.proxy.api.async.mail.Side;
import ru.yandex.msearch.proxy.api.async.mail.rules.RewriteRequestRule;
import ru.yandex.msearch.proxy.experiment.UserSplit;
import ru.yandex.parser.uri.CgiParams;

public class MailSearchOptions implements JsonValue {
    private final Map<String, List<MailSearchOptions>> requests =
        new IdentityHashMap<>();
    private final String field;
    private String request;
    private String rule;
    private String suggest;
    private UserSplit userSplit;
    private Side side;
    private Product product;
    private boolean pure;

    public MailSearchOptions(final CgiParams params) {
        this(params.getString("request", ""));
        for (String param: RewriteRequestRule.REQUEST_TEXT_PARAMS) {
            List<String> values = params.getAll(param);
            if (!values.isEmpty()) {
                requests.put(
                    param,
                    values.stream()
                        .map(request -> new MailSearchOptions(param, request))
                        .collect(Collectors.toList()));
            }
        }

        this.side = Side.parse(params);
        this.product = Product.parse(params);
    }

    public MailSearchOptions(final String request) {
        this("request", request);
    }

    public MailSearchOptions(
        final String field,
        final String request)
    {
        this.field = field;
        this.request = request;
    }

    public void request(final String request, final String rule) {
        this.request = request;
        this.rule = rule;
        suggest = null;
    }

    public void suggest(final String suggest, final String rule) {
        this.suggest = suggest;
        this.rule = rule;
    }

    public void pure(final boolean pure) {
        this.pure = pure;
    }

    public void userSplit(final UserSplit userSplit) {
        this.userSplit = userSplit;
    }

    public Side side() {
        return side;
    }

    public Product product() {
        return product;
    }

    public Map<String, List<MailSearchOptions>> requests() {
        return requests;
    }

    public String field() {
        return field;
    }

    public String request() {
        return request;
    }

    public String suggest() {
        return suggest;
    }

    public String rule() {
        return rule;
    }

    public UserSplit userSplit() {
        return userSplit;
    }

    public void toJSONWriter(final JSONWriter writer) throws JSONException {
        writer.key("search-options");
        writer.object();
        writer.key("request");
        writer.value(request);
        if (suggest != null) {
            writer.key("suggest");
            writer.value(suggest);
        }
        if (rule != null) {
            writer.key("rule");
            writer.value(rule);
        }
        if (pure) {
            writer.key("pure");
            writer.value(true);
        }
        writer.endObject();
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key("request");
        writer.value(request);
        if (suggest != null) {
            writer.key("suggest");
            writer.value(suggest);
        }
        if (rule != null) {
            writer.key("rule");
            writer.value(rule);
        }
        if (pure) {
            writer.key("pure");
            writer.value(true);
        }
        if (!requests.isEmpty()) {
            writer.key("requests");
            writer.value(requests);
        }

        if (userSplit != null) {
            writer.key("experiments");
            writer.value(userSplit.testsToString());
        }

        writer.endObject();
    }
}

