package ru.yandex.msearch.proxy.api.mail;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintStream;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.json.JSONException;
import org.json.JSONWriter;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.collector.Collector;
import ru.yandex.msearch.proxy.document.Document;

public class News {
    private final Map<Stat, Stat> emails = new HashMap<>();
    private final HttpServer.RequestContext ctx;

    public News(HttpServer.RequestContext ctx) {
        this.ctx = ctx;
    }

    public void collect(Collector collector) {
        int collected = 0;
        for (Document doc: collector.hits().keySet()) {
            String hdrFrom = doc.getAttr("hdr_from");
            if (hdrFrom == null) {
                continue;
            }
            String[] parts = splitEmail(doc.getAttr("hdr_from"));
            cache(new Stat(parts[1], parts[0])).increment();
            collected++;
        }
        ctx.log.info("collected " + collected + " news");
    }

    private Stat cache(final Stat stat) {
        Stat cached = emails.get(stat);
        if (cached == null) {
            emails.put(stat, stat);
            cached = stat;
        }
        return cached;
    }

    public void print(final PrintStream ps) throws IOException, JSONException {
        Map<String, List<Stat>> byEmail = new HashMap<>();
        for (Stat stat: emails.keySet()) {
            List<Stat> list = byEmail.get(stat.email());
            if (list == null) {
                list = new ArrayList<Stat>();
                byEmail.put(stat.email(), list);
            }
            list.add(stat);
        }

        ctx.setContentType("application/json");
        OutputStreamWriter writer = new OutputStreamWriter(ps);
        JSONWriter jw = new JSONWriter(writer);
        try {
            jw.object();
            jw.key("news");
            jw.array();
            for (Map.Entry<String, List<Stat>> entry: byEmail.entrySet()) {
                jw.object();
                jw.key("email");
                jw.value(entry.getKey());
                jw.key("names");
                jw.array();
                List<Stat> list = entry.getValue();
                Collections.sort(list, Stat.COUNT_DESC_COMPARATOR);
                for (Stat stat: list) {
                    jw.object();
                    jw.key("name");
                    jw.value(stat.name());
                    jw.key("count");
                    jw.value(stat.count());
                    jw.endObject();
                }
                jw.endArray();
                jw.endObject();
            }
            jw.endArray();
            jw.endObject();
        } finally {
            if (writer != null) {
                writer.flush();
            }
        }
    }

    private String[] splitEmail(final String header) {
        int beginName = header.indexOf('"');
        int endName = header.lastIndexOf('"');
        if (beginName == -1 || beginName >= endName) {
            return new String[]{"", header};
        }

        int beginEmail = header.lastIndexOf('<');
        int endEmail = header.lastIndexOf('>');
        if (beginEmail == -1 || endEmail == -1 || beginEmail >= endEmail) {
            return new String[]{"", header};
        }

        return new String[]{
            header.substring(beginName + 1, endName),
            header.substring(beginEmail + 1, endEmail)};
    }
    
    static class Stat {
        private final String email;
        private final String name;
        private int count = 0;

        public Stat(final String email, final String name) {
            this.email = email;
            this.name = name;
        }

        public void increment() {
            count++;
        }

        @Override
        public int hashCode() {
            return email.hashCode() ^ name.hashCode();
        }

        @Override
        public boolean equals(final Object other) {
            if (!(other instanceof Stat)) {
                return false;
            }
            Stat stat = (Stat) other;
            return email.equals(stat.email) && name.equals(stat.name);
        }

        public String email() {
            return email;
        }

        public String name() {
            return name;
        }

        public int count() {
            return count;
        }

        public static final Comparator<Stat> COUNT_DESC_COMPARATOR =
            new Comparator<Stat>() {
                @Override
                public int compare(final Stat one, final Stat other) {
                    return -Integer.compare(one.count, other.count);
                }
            };
    }
}

