package ru.yandex.msearch.proxy.api.mail.rules;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import ru.yandex.blackbox.BlackboxUserinfo;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.api.mail.MailSearchOptions;
import ru.yandex.msearch.proxy.api.mail.MailSearchResult;
import ru.yandex.msearch.proxy.collector.Collector;
import ru.yandex.msearch.proxy.collector.LinkedHashCollector;
import ru.yandex.msearch.proxy.document.Document;
import ru.yandex.msearch.proxy.search.MailSearcher;

import ru.yandex.parser.email.MailAliases;

public class MailSearchRule implements SearchRule {
    public static final List<String> PURE_SCOPE = Arrays.asList(
        "mid_p",
        "pure_body",
        "hdr_subject",
        "hdr_from",
        "hdr_from_normalized",
        "hdr_to",
        "hdr_to_normalized",
        "hdr_cc",
        "hdr_cc_normalized",
        "hdr_bcc",
        "hdr_bcc_normalized",
        "reply_to",
        "reply_to_normalized",
        "attachname",
        "attachtype",
        "attachments");
    public static final List<String> EXTENDED_SCOPE = Arrays.asList(
        "body_text",
        "album",
        "artist",
        "author",
        "comment",
        "composer",
        "description",
        "genre",
        "keywords",
        "subject",
        "title");
    public static final List<String> DEFAULT_SCOPE;

    static {
        DEFAULT_SCOPE = new ArrayList<>(PURE_SCOPE);
        DEFAULT_SCOPE.addAll(EXTENDED_SCOPE);
    }

    private static int toDate(final String str) {
        if (str != null) {
            try {
                int value = Integer.parseInt(str);
                if (value > 19000000 && value < 30000000) {
                    return value;
                }
            } catch (NumberFormatException e) {
            }
        }
        return -1;
    }

    public static String rewriteRequest(String request, List<String> scope) {
        request = request
            .replaceAll("[*\\\\(){}\\[\\]'?~\u00a0+:!^]", " ")
            .toLowerCase(Locale.ENGLISH)
            .trim();
        if (request.isEmpty()) {
            return null;
        }

        String[] parts = request.split("\"");
        List<String> partsList = new ArrayList<>(parts.length);
        for (int i = 0; i < parts.length; ++i) {
            String part = parts[i].trim();
            if (!part.isEmpty()) {
                if ((i & 1) == 0) {
                    for (String token: part.split(" ")) {
                        String trimmed = token.trim();
                        if (!trimmed.isEmpty()) {
                            if (trimmed.charAt(0) == '-') {
                                trimmed = '\\' + trimmed;
                            } else {
                                int sep =
                                    MailAliases.emailSeparatorPos(trimmed);
                                if (sep != -1) {
                                    Set<String> emails = MailAliases.INSTANCE
                                        .equivalentEmails(trimmed, sep);
                                    if (emails.size() > 1) {
                                        StringBuilder sb = new StringBuilder();
                                        sb.append('(');
                                        for (String email: emails) {
                                            sb.append(email);
                                            sb.append(' ');
                                        }
                                        sb.setCharAt(sb.length() - 1, ')');
                                        trimmed = new String(sb);
                                    }
                                }
                            }
                            partsList.add(trimmed);
                        }
                    }
                } else {
                    partsList.add('"' + part + '"');
                }
            }
        }
        if (partsList.isEmpty()) {
            return null;
        }

        if (scope == null || scope.isEmpty()) {
            scope = DEFAULT_SCOPE;
        }

        StringBuilder sb = new StringBuilder();
        sb.append('(');
        for (String part: partsList) {
            Iterator<String> iter = scope.iterator();
            sb.append(iter.next());
            sb.append(':');
            sb.append(part);
            while (iter.hasNext()) {
                sb.append(" OR ");
                sb.append(iter.next());
                sb.append(':');
                sb.append(part);
            }
            sb.append(") AND (");
        }
        sb.setLength(sb.length() - 6);
        return new String(sb);
    }

    @Override
    public MailSearchResult execute(
        final HttpServer.RequestContext ctx,
        final HttpServer.HttpParams params,
        final int length)
        throws ApiException, MsearchProxyException
    {
        String db = params.get("mdb");
        String user = params.get("suid");
        String text = params.get("request");

        String request;
        if (text == null || params.getBoolean("imap", false)) {
            request = text;
            if (request != null && request.trim().isEmpty()) {
                request = null;
            }
        } else {
            request = rewriteRequest(text, params.getAll("scope"));
            params.remove("scope");
        }

        if (params.getBoolean("has_attachments", false)) {
            if (request == null) {
                request = "has_attachments:1";
            } else {
                request = '(' + request + ") AND has_attachments:1";
            }
        }

        List<String> messageTypes = params.getAll("message_type");
        if (messageTypes != null && !messageTypes.isEmpty()) {
            Iterator<String> iter = messageTypes.iterator();
            StringBuilder sb = new StringBuilder("message_type:");
            sb.append(iter.next());
            while (iter.hasNext()) {
                sb.append(" OR message_type:");
                sb.append(iter.next());
            }
            if (request == null) {
                request = new String(sb);
            } else {
                request = '(' + request + ") AND (" + sb + ')';
            }
        }

        boolean corp = false;
        try {
            corp = BlackboxUserinfo.corp(Long.parseLong(user));
        } catch (RuntimeException e) {
        }

        MailSearcher ms = new MailSearcher(ctx, db, user);
        ms.setImapSearch(true);
        ms.setUnread(params.getBoolean("unread", false));
        ms.setFids(params.getAllOrEmpty("fid"));
        ms.setLids(params.getAllOrEmpty("lid"));
        ms.setFromDate(toDate(params.get("from")));
        ms.setToDate(toDate(params.get("to")));
        ms.setServerOrder(params.get("order"));
        ms.setFields(params.get("getfields"));
        ms.setAttachmentsSearch(params.getBoolean("attachments", false));
        String folderSet = params.get("folder_set");
        if (folderSet == null) {
            if (corp) {
                folderSet = "default";
            } else {
                folderSet = "";
            }
        }
        ms.setFolderSet(folderSet);

        Collector col = new LinkedHashCollector("mid") {
            @Override
            public Document createDocument() {
                return new Document();
            }
        };
        ms.search(col, request, length);
        return new MailSearchResult(col, new MailSearchOptions(text));
    }
}

