package ru.yandex.msearch.proxy.api.mail.rules;

import java.net.SocketTimeoutException;

import java.nio.charset.CharacterCodingException;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;

import ru.yandex.erratum.ErratumClient;
import ru.yandex.erratum.ErratumResult;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.api.mail.MailSearchOptions;
import ru.yandex.msearch.proxy.api.mail.MailSearchResult;
import ru.yandex.msearch.proxy.logger.Logger;

public class MisspellRule extends ChainedSearchRule {
    private final ErratumClient client;

    public MisspellRule(final ErratumClient client) {
        this.client = client;
    }

    @Override
    public MailSearchResult execute(
        final HttpServer.RequestContext ctx,
        final HttpServer.HttpParams params,
        final int length)
        throws ApiException, MsearchProxyException
    {
        String request = params.get("request");
        if (request == null
            || request.isEmpty()
            || request.indexOf('@') != -1
            || params.getBoolean("imap", false)
            || params.getBoolean("force", false))
        {
            return next(ctx, params, length);
        } else {
            try {
                Future<ErratumResult> erratumFuture = client.execute(request);
                MailSearchResult searchResult = next(ctx, params, length);
                if (searchResult != null) {
                    ErratumResult result = null;
                    long start = System.currentTimeMillis();
                    try {
                        result = erratumFuture.get();
                        ctx.log.warn("erratum result: " + result);
                    } catch (InterruptedException e) {
                        ctx.log.warn("erratum: Request interrupted: "
                            + Logger.exception(e));
                    } catch (ExecutionException e) {
                        Throwable error = e.getCause();
                        if (error instanceof SocketTimeoutException) {
                            ctx.log.warn("erratum: Request timed out: "
                                + Logger.exception(e));
                        } else if (error instanceof TimeoutException) {
                            ctx.log.warn("erratum: No available connections: "
                                + Logger.exception(e));
                        } else {
                            ctx.log.warn("erratum: Request failed: "
                                + Logger.exception(e));
                        }
                    } finally {
                        ctx.log.debug(
                            "erratum: Total erratum request overhead: "
                            + (System.currentTimeMillis() - start));
                    }
                    if (result != null
                        && result.code() == ErratumResult.CODE_CORRECTED)
                    {
                        HttpServer.HttpParams paramsCopy = params.copy();
                        paramsCopy.replace("request", result.text());
                        MailSearchResult fixed = next(ctx, paramsCopy, length);
                        if (fixed == null) {
                            searchResult = null;
                        } else {
                            int count =
                                searchResult.collector().getTotalCount();
                            int fixedCount = fixed.collector().getTotalCount();
                            ctx.log.debug("erratum: original count: " + count
                                + ", fixed count: " + fixedCount);
                            if (fixedCount != 0) {
                                if (count == 0) {
                                    ctx.log.debug(
                                        "erratum: switching to fixed results");
                                    MailSearchOptions options =
                                        new MailSearchOptions("");
                                    options.request(
                                        result.text(),
                                        result.rule());
                                    searchResult = new MailSearchResult(
                                        fixed.collector(),
                                        options);
                                } else {
                                    ctx.log.debug("erratum: adding suggest");
                                    MailSearchOptions options =
                                        new MailSearchOptions(request);
                                    options.suggest(
                                        result.text(),
                                        result.rule());
                                    searchResult = new MailSearchResult(
                                        searchResult.collector(),
                                        options);
                                }
                            }
                        }
                    }
                }
                return searchResult;
            } catch (CharacterCodingException e) {
                throw new ApiException("Failed to encode: " + request, e);
            }
        }
    }
}

