package ru.yandex.msearch.proxy.api.mail.rules;

import java.io.IOException;

import java.text.ParseException;

import org.apache.http.HttpException;
import org.apache.http.entity.ContentType;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;

import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.api.async.suggest.history.AbstractStoredRequest;
import ru.yandex.msearch.proxy.api.async.mail.rules.StoreSearchRequestRule;
import ru.yandex.msearch.proxy.api.async.suggest.history.DeleteStoredRequest;
import ru.yandex.msearch.proxy.api.async.suggest.history.UpdateStoredRequest;
import ru.yandex.msearch.proxy.api.mail.MailSearchResult;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;

import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.User;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

public class StoreSearchRequestAdapter
    extends StoreSearchRequestRule
    implements SearchRule
{
    protected final SearchRule nextSyncRule;
    private final PrefixType defaultPrefixType;

    public StoreSearchRequestAdapter(
        final SearchRule nextSyncRule,
        final AsyncClient client,
        final ImmutableMsearchProxyConfig proxyConfig,
        final SearchMap searchMap)
    {
        super(null, client, proxyConfig, searchMap);
        this.nextSyncRule = nextSyncRule;
        defaultPrefixType = proxyConfig.searchMapConfig().prefixType();
    }

    public StoreSearchRequestAdapter(
        final AsyncClient client,
        final ImmutableMsearchProxyConfig proxyConfig,
        final SearchMap searchMap)
    {
        this(null, client, proxyConfig, searchMap);
    }

    public static StoreSearchRequestAdapter create(
        final SharedConnectingIOReactor reactor,
        final ImmutableMsearchProxyConfig proxyConfig)
    {
        return create(null, reactor, proxyConfig, null);
    }

    public static StoreSearchRequestAdapter create(
        final SearchRule next,
        final SharedConnectingIOReactor reactor,
        final ImmutableMsearchProxyConfig proxyConfig)
    {
        return create(next, reactor, proxyConfig, null);
    }

    public static StoreSearchRequestAdapter create(
        final SearchRule next,
        final SharedConnectingIOReactor reactor,
        final ImmutableMsearchProxyConfig proxyConfig,
        final SearchMap searchMap)
    {
        try {
            AsyncClient client =
                new AsyncClient(reactor, proxyConfig.producerClientConfig());
            client.start();
            return new StoreSearchRequestAdapter(
                next,
                client,
                proxyConfig,
                searchMap);
        } catch (Exception e) {
            throw new RuntimeException("Unable create producer client", e);
        }
    }

    private PrefixType prefixType(final String mdb) {
        if (searchMap != null) {
            return searchMap.prefixType(mdb);
        } else {
            return defaultPrefixType;
        }
    }

    private void storeRequest(
        final HttpServer.RequestContext ctx,
        final HttpServer.HttpParams params,
        final MailSearchResult result,
        final String proxyName,
        final String request)
    {
        String mdb = params.get("mdb");
        String suid = params.get("suid");
        if (mdb == null || suid == null) {
            ctx.log.err("storeRequest failed no suid or mdb");
            return;
        }

        try {
            Prefix prefix = prefixType(mdb).parse(suid);
            User user = new User(
                AsyncHttpServer.resolveService(mdb, prefix, proxyConfig),
                prefix);

            AbstractStoredRequest storedRequest;
            if (result.collector().getTotalCount() == 0) {
                storedRequest = createDeleteRequest(user.prefix(), request);
            } else {
                storedRequest = createUpdateRequest(
                    user.prefix(),
                    request,
                    result.options());
            }

            storeRequest(ctx, storedRequest, proxyName, user);
        } catch (ParseException | IOException | HttpException e) {
            ctx.log.err("storeRequest failed", e);
            return;
        }

        ctx.log.info("storeRequest done");
    }

    protected BasicAsyncRequestProducerGenerator createGenerator(
        final HttpServer.RequestContext ctx,
        final AbstractStoredRequest request,
        final String proxyName,
        final User user)
        throws IOException, BadRequestException
    {
        String uri = request.uri(user, proxyName, ctx.getSessionId());
        String data = request.toJsonString();
        if (data == null) {
            return new BasicAsyncRequestProducerGenerator(uri);
        } else {
            return new BasicAsyncRequestProducerGenerator(
                uri,
                data,
                ContentType.APPLICATION_JSON);
        }
    }

    private void storeRequest(
        final HttpServer.RequestContext ctx,
        final AbstractStoredRequest request,
        final String proxyName,
        final User user)
        throws HttpException, IOException
    {
        producerClient.adjust(ctx.getSessionId(), ctx.referer()).execute(
            proxyConfig.producerClientConfig().host(),
            createGenerator(ctx, request, proxyName, user),
            BasicAsyncResponseConsumerFactory.OK,
            EmptyFutureCallback.INSTANCE);
    }


    protected boolean isStorable(
        final String trimRequest,
        final HttpServer.HttpParams params)
    {
        return trimRequest.length() >= minimumRequestStoreLen
            && params.getBoolean("save-request", true)
            && !params.getBoolean("imap", false)
            && !params.getBoolean("nostore", false);
    }

    public void storeRequest(
        final HttpServer.RequestContext ctx,
        final HttpServer.HttpParams params,
        final String proxyName,
        final String text,
        final boolean delete)
    {
        if (text == null) {
            return;
        }

        String trimRequest = text.trim();
        if (!isStorable(trimRequest, params)) {
            return;
        }

        String mdb = params.get("db");
        String userId = params.get("user");
        if (mdb == null || userId == null) {
            ctx.log.err("StoreRequest invalid user or db");
            return;
        }

        try {
            Prefix prefix = prefixType(mdb).parse(userId);

            User user = new User(
                AsyncHttpServer.resolveService(mdb, prefix, proxyConfig),
                prefix);
            AbstractStoredRequest request;
            if (delete) {
                request = new DeleteStoredRequest(
                    user.prefix(),
                    trimRequest);
            } else {
                request = new UpdateStoredRequest(
                    user.prefix(),
                    trimRequest);
            }

            storeRequest(ctx, request, proxyName, user);
        } catch (ParseException | IOException | HttpException e) {
            ctx.log.err("storeRequest failed", e);
            return;
        }

        ctx.log.info("storeRequest done");
    }

    @Override
    public MailSearchResult execute(
        final HttpServer.RequestContext ctx,
        final HttpServer.HttpParams params,
        final int length) throws MsearchProxyException
    {
        String request = params.get("request");
        MailSearchResult result = nextSyncRule.execute(ctx, params, length);

        if (request == null) {
            return result;
        } else {
            request = request.trim();
        }

        if (isStorable(request, params)) {
            storeRequest(ctx, params, result, "msproxy_sync", request);
            return result;
        }

        return result;
    }
}
