package ru.yandex.msearch.proxy.api.postmaster;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.Weigher;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.UnsupportedEncodingException;

import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

import java.text.ParseException;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.HttpEntity;

import ru.yandex.http.util.CharsetUtils;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.xpath.XPathContentHandler;
import ru.yandex.json.xpath.PathComponent;
import ru.yandex.json.xpath.PrimitiveHandler;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.api.AbstractDispatchableService;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.api.ServiceProcessor;
import ru.yandex.msearch.proxy.dispatcher.Dispatcher;
import ru.yandex.msearch.proxy.dispatcher.DispatcherException;
import ru.yandex.msearch.proxy.dispatcher.DispatcherFactory;
import ru.yandex.msearch.proxy.searchmap.SearchMap;
import ru.yandex.msearch.proxy.logger.Logger;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixParser;
import ru.yandex.search.prefix.PrefixType;

import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.OrderSavingSearchResult;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.result.SearchResultHandler;


public class PostMaster extends AbstractDispatchableService {
    private static String POSTMASTER_SERVICE_NAME;
    private static int DEFAULT_SEARCH_TIMEOUT;
    private static int ZOOLOOSER_INDEX_TIMEOUT;
    private static final String[] checkParams = {
	"prefix",
	"group",
	"text"
    };

    private static Cache<String, byte[]> searchCache = null;
    private static Pattern sessionIdPattern;

    public static void init(final IniConfig config) throws ConfigException {
        POSTMASTER_SERVICE_NAME =
            config.getString("service_name", "postmaster");
        DEFAULT_SEARCH_TIMEOUT = config.getInt("default_search_timeout", 30000);
        ZOOLOOSER_INDEX_TIMEOUT = config.getInt("index_timeout", 30000);
        int cacheExpire = config.getInt("searchcache_expire", 100);
        int cacheSize = config.getInt("searchcache_size", 10000000);
        String sessionIdPattern = config.getString("session_id_pattern",
            "session=[a-zA-Z%0-9]+");
        initCache(cacheExpire, cacheSize, sessionIdPattern);
    }

    public PostMaster() {
        super(checkParams);
    }

    @Override
    public String name() {
        return "postmaster";
    }

    @Override
    public String serviceName() throws MsearchProxyException {
        return POSTMASTER_SERVICE_NAME;
    }

    @Override
    public PrefixParser prefixParser() {
        return PrefixType.STRING;
    }

    @Override
    public int indexTimeout() {
        return ZOOLOOSER_INDEX_TIMEOUT;
    }

    @Override
    public int searchTimeout() {
        return DEFAULT_SEARCH_TIMEOUT;
    }

    @Override
    public boolean allowIndex() {
        return true;
    }

    @Override
    public boolean allowUpdate() {
        return true;
    }

    @Override
    public boolean allowDelete() {
        return true;
    }

    @Override
    public boolean wait(final String api, final HttpServer.HttpParams params) {
        return api.indexOf("_nowait") == -1;
    }

    @Override
    protected int processInternal(
        final ServiceProcessor processor,
        final HttpServer.HttpParams params)
        throws MsearchProxyException
    {
        if (processor.api().equals("search")
            || processor.api().equals("intersect"))
        {
            return search(processor, params);
        }
        return -1;
    }

    public static void initCache(final int expire, final int size,
        final String sessionIdPattern) 
    {
        Cache<String, byte[]> cache = CacheBuilder.newBuilder()
            .maximumWeight(size)
            .expireAfterWrite(expire, TimeUnit.MILLISECONDS)
            .weigher(
                new Weigher<String, byte[]>()
                {
                    public int weigh( String key, byte[] value )
                    {
                        return key.length() * 2 + value.length;
                    }
                }
            )
            .build();
        searchCache = cache;
        PostMaster.sessionIdPattern = Pattern.compile(sessionIdPattern);
    }

    private int search(
        final ServiceProcessor processor,
        final HttpServer.HttpParams params)
        throws MsearchProxyException
    {
        return search(processor, params, false);
    }

    private boolean pruningEmulation(final HttpServer.HttpParams params) {
        final String sort = params.get("sort");
        if (sort != null && sort.equals("multi(year desc,month desc,day desc"
            + ",subject asc,from asc)"))
        {
            return true;
        }
        return false;
    }

    private int pruneSearch(
        final ServiceProcessor processor,
        final HttpServer.HttpParams params)
        throws MsearchProxyException
    {
        final RequestWidener widener =
            RequestWidenerFactory.create(processor, params);
        processor.ctx().log.info("Pruning emulation: " + widener);
        final int length = params.getInt("length", Integer.MAX_VALUE);
        final BasicSearchResult emptyResult = new BasicSearchResult();
        emptyResult.hitsCount(0);

        Prefix prefix = processor.getPrefix(params);
        String timeoutString = params.get("timeout");
        int timeout;
        if (timeoutString != null) {
            timeout = Integer.parseInt(timeoutString);
        } else {
            timeout = DEFAULT_SEARCH_TIMEOUT;
        }

        final SearchContext context = new SearchContext(
            processor,
            params,
            prefix,
            timeout);
        final SearchResult sr = doSearch(context, widener, length, emptyResult);
        processor.ctx().setContentType( "application/json" );
        try (JsonWriter writer =
                new JsonWriter(new OutputStreamWriter(processor.ctx().ps())))
        {
            writer.startObject();
            writer.key("hitsCount");
            writer.value(sr.hitsCount());
            writer.key("hitsArray");
            writer.startArray();
            List<SearchDocument> hitsArray = sr.hitsArray();
            for (SearchDocument hit: hitsArray) {
                writer.value(hit.attrs());
            }
            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            throw new MsearchProxyException("Error writing json output", e);
        }
        return (int)sr.hitsCount();
    }

    private SearchResult doSearch(
        final SearchContext context,
        final RequestWidener widener,
        final int maxLength,
        final SearchResult sr)
        throws ApiException, MsearchProxyException
    {
        final String request = widener.nextRequest();
        if (request == null) {
            return sr;
        }
        context.processor.ctx().log.info("Trying narrowed request: " + request);
        long requestStart = System.currentTimeMillis();
        try {
            Dispatcher dispatcher =
                DispatcherFactory.create(
                    context.prefix(),
                    serviceName(),
                    SearchMap.getInstance(),
                    false,
                    context.processor().ctx());

            try (CloseableHttpResponse response =
                    dispatcher.searchDispatch(
                        new HttpGet(request),
                        context.timeout()))
            {
                HttpEntity httpEntity = response.getEntity();
                int httpCode = response.getStatusLine().getStatusCode();
                if (httpCode < HttpStatus.SC_OK
                    || httpCode >= HttpStatus.SC_BAD_REQUEST)
                {
                    context.processor().ctx().log.err(
                        "PostMaster.search: error: "
                        + "server returned status: " + httpCode);
                    context.processor().ctx().setHttpCode(httpCode);
                    context.processor().ctx().ps().println(
                        "Backend response:  http code: "
                        + httpCode + ", response body: ");
                    if (httpEntity != null) {
                        BufferedReader r =
                            new BufferedReader(
                                new InputStreamReader(
                                    httpEntity.getContent(),
                                    "UTF-8"));
                        String line;
                        while ((line = r.readLine()) != null) {
                            context.processor().ctx().log.err(
                                "PostMaster.search: "
                                + "server response: " + line);
                            context.processor().ctx().ps().println(line);
                        }
                    }
                    context.processor().ctx().log.info(
                        "PostMaster.search backend request "
                        + "execution time: "
                        + (System.currentTimeMillis() - requestStart));
                    throw new ApiException(
                        "PostMaster.search: invalid backend response: code="
                            + httpCode);
                } else {
                    long eStart = System.currentTimeMillis();
                    SearchResult newResult = parseResults(httpEntity);
                    context.processor().ctx().log.info(
                        "Backend response read time: "
                        + (System.currentTimeMillis() - eStart));
                    context.processor().ctx().log.info(
                        "PostMaster.search backend request "
                        + "execution time: "
                        + (System.currentTimeMillis() - requestStart));
                    if (newResult.hitsCount() < maxLength) {
                        return doSearch(context, widener, maxLength, newResult);
                    } else {
                        return newResult;
                    }
                }
            }
        } catch (DispatcherException e) {
            throw new ApiException(
                "PostMaster.search: can't find backends for searching",
                e);
        } catch (IOException e) {
            throw new ApiException("PostMaster.search: backend io error", e);
        } finally {
        }
    }

    private SearchResult parseResults(final HttpEntity entity)
        throws MsearchProxyException
    {
        try (Reader r =
                new BufferedReader(
                    new InputStreamReader(
                        entity.getContent(),
                        StandardCharsets.UTF_8));)
        {
            final OrderSavingSearchResult sr = new OrderSavingSearchResult();
            final SearchResultHandler srHandler =
                new SearchResultHandler(sr, true);
            XPathContentHandler jsonHandler =
                new XPathContentHandler(srHandler);
            JsonParser jsonParser = new JsonParser(jsonHandler);
            jsonParser.parse(r);
            return sr;
        } catch(IOException|JsonException e) {
            throw new MsearchProxyException(e);
        }
    }

    private int search(
        final ServiceProcessor processor,
        final HttpServer.HttpParams params,
        final boolean noRecursion)
        throws MsearchProxyException
    {
        checkSearchParams(params);
        Prefix prefix = processor.getPrefix(params);
        String db = serviceName();
        String timeoutString = params.get("timeout");
        int timeout;
        if (timeoutString != null) {
            timeout = Integer.parseInt(timeoutString);
        } else {
            timeout = DEFAULT_SEARCH_TIMEOUT;
        }

        String client = params.get("client");
        boolean cache = false;
        if (client != null && client.equals("fastsrv")) {
            cache = true;
        }

        if (cache == false && pruningEmulation(params)) {
            return pruneSearch(processor, params);
        }

        String request = processor.request();
        String requestNoSid = sessionIdPattern.matcher(request).replaceAll("");
        if (cache && !noRecursion) {
            byte[] reply = searchCache.getIfPresent(requestNoSid);
            if (reply == null) {
                requestNoSid = requestNoSid.intern();
                synchronized (requestNoSid) {
                    reply = searchCache.getIfPresent(requestNoSid);
                    if (reply == null) {
                        return search(processor, params, true);
                    }
                }
            }
            processor.ctx().setContentType("application/json");
            processor.ctx().ps().write(reply, 0, reply.length);
            processor.ctx().log.info("Returning response from cache");
            return 0;
        }

        long requestStart = System.currentTimeMillis();
        try
        {
            Dispatcher dispatcher = DispatcherFactory.create( prefix, db, SearchMap.getInstance(), false, processor.ctx() );

            try (CloseableHttpResponse response =
                    dispatcher.searchDispatch(new HttpGet(request), timeout))
            {
                HttpEntity httpEntity = response.getEntity();
                int httpCode = response.getStatusLine().getStatusCode();
                if( httpCode < HttpStatus.SC_OK || httpCode >= HttpStatus.SC_BAD_REQUEST )
                {
                    processor.ctx().log.err( "PostMaster.search: error: server returned status: " + httpCode );
                    processor.ctx().setHttpCode( httpCode );
                    processor.ctx().ps().println( "Backend response:  http code: " + httpCode + ", response body: " );
                    if (httpEntity != null) {
                        BufferedReader r = new BufferedReader(new InputStreamReader(httpEntity.getContent(), "UTF-8"));
                        String line;
                        while( (line = r.readLine()) != null )
                        {
                            processor.ctx().log.err( "PostMaster.search: server response: " + line );
                            processor.ctx().ps().println( line );
                        }
                    }
                }
                else
                {
                    processor.ctx().setContentType( "application/json" );
                    long eStart = System.currentTimeMillis();
                    if( cache )
                    {
                        byte[] body = CharsetUtils.toByteArray(httpEntity);
                        // TODO: Body writing must be done outside of lock
                        processor.ctx().ps().write(body);
                        processor.ctx().log.info(
                            "Backend response read time: "
                            + (System.currentTimeMillis() - eStart)
                            + ", size: " + body.length
                            + ". Caching as " + requestNoSid);
                        searchCache.put(requestNoSid, body);
                    }
                    else
                    {
                        if (httpEntity != null) {
                            httpEntity.writeTo(processor.ctx().ps());
                        }
                        processor.ctx().log.info(
                            "Backend response read time: "
                            + (System.currentTimeMillis() - eStart));
                    }
                }
	    }
	}
        catch( DispatcherException e )
        {
            throw new ApiException(
                "PostMaster.search: can't find backends for searching",
                e);
        }
	catch( IOException e )
	{
            throw new ApiException("PostMaster.search: backend io error", e);
	}
	finally
	{
	    processor.ctx().log.info( "PostMaster.search backend request execution time: " + (System.currentTimeMillis() - requestStart) );
	}

	return 0;
    }

    private static class SearchContext {
        private final ServiceProcessor processor;
        private final HttpServer.HttpParams params;
        private final Prefix prefix;
        private final int timeout;

        public SearchContext(
            final ServiceProcessor processor,
            final HttpServer.HttpParams params,
            final Prefix prefix,
            final int timeout)
        {
            this.processor = processor;
            this.params = params;
            this.prefix = prefix;
            this.timeout = timeout;
        }

        public ServiceProcessor processor() {
            return processor;
        }

        public HttpServer.HttpParams params() {
            return params;
        }

        public Prefix prefix() {
            return prefix;
        }

        public int timeout() {
            return timeout;
        }
    }
}
