package ru.yandex.msearch.proxy.api.suggest;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintStream;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.exception.ExceptionUtils;

import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.HttpClient;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;

import org.apache.james.mime4j.codec.DecodeMonitor;
import org.apache.james.mime4j.codec.DecoderUtil;

import org.json.JSONException;
import org.json.JSONWriter;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.client.measurable.MeasurableHttpContext;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.BasicRequestsListener;

import ru.yandex.logger.BackendAccessLoggerConfigDefaults;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.HttpServer.HttpParams;
import ru.yandex.msearch.proxy.HttpServer.RequestContext;
import ru.yandex.msearch.proxy.MsearchProxyException;
import ru.yandex.msearch.proxy.OutputPrinter;
import ru.yandex.msearch.proxy.Synonyms;
import ru.yandex.msearch.proxy.api.ApiException;
import ru.yandex.msearch.proxy.api.async.mail.Side;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestFactors;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestFactors.SuggestFactor;
import ru.yandex.msearch.proxy.collector.Collector;
import ru.yandex.msearch.proxy.collector.GroupingSortingCollector;
import ru.yandex.msearch.proxy.config.ImmutableFactorsLogConfig;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.document.Document;
import ru.yandex.msearch.proxy.httpclient.CommonHttpClient;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.logger.ProxyTskvLogger;
import ru.yandex.msearch.proxy.search.Searcher;
import ru.yandex.msearch.proxy.searchmap.SearchMap;
import ru.yandex.msearch.proxy.util.Strings;

import ru.yandex.parser.config.IniConfig;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.proxy.SearchResultConsumerFactory;

import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

import ru.yandex.util.string.StringUtils;

public class Suggest {
    private static final long DEFAULT_SUGGEST_SWITCH_TO_NEXT_DELAY = 100;
    private static final long DEFAULT_MAX_GET_HEADERS_KEYSET_TIME = 900;
    private static final long DEFAULT_MAX_REQUEST_TIME = 1950;
    private static final String[] excludeNoEmails =
        {
            "No address",
            "\"No_address\" <>",
            "\"No address\" <>",
            "no_address",
            "\"no_address\" <>",
            "undisclosed-recipients:;" };
    private static final String[] searchHeaders = { "hdr_from", "hdr_to", "hdr_cc" };

    private final Synonyms synonyms;
    private final int TOTAL_LIMIT;
    private final boolean ENABLED;
    private final long switchToNextDelay;
    private final long maxGetHeadersKeySetTime;
    private final long maxRequestTime;
    private final HttpClient httpClient;
    private final ThreadPoolExecutor suggestExecutor;
    private final LoadingCache<CacheKey, Set<String>> printKeysCache;
    private final ProxyTskvLogger factorsLogger;

    private final CacheLoader<CacheKey,Set<String>> cacheLoader =
        new CacheLoader<CacheKey,Set<String>>() {
            @Override
            public Set<String> load(CacheKey key) throws ApiException {
                return getHeadersKeySetImpl(key);
            }
        };

    public Suggest(
        final Synonyms synonyms,
        final IniConfig config,
        final ImmutableMsearchProxyConfig proxyConfig)
        throws Exception
    {
        this.synonyms = synonyms;
        TOTAL_LIMIT = config.getInt("total_limit", 10);
        ENABLED = config.getBoolean("enabled", true);
        switchToNextDelay = config.getLong("switch-to-next-host-delay",
            DEFAULT_SUGGEST_SWITCH_TO_NEXT_DELAY);
        maxGetHeadersKeySetTime = config.getLong("max-get-headers-keyset-time",
            DEFAULT_MAX_GET_HEADERS_KEYSET_TIME);
        maxRequestTime = config.getLong("max-request-time",
            DEFAULT_MAX_REQUEST_TIME);
        int threads = config.getInt("threads", 500);
        int queueSize = threads;
        httpClient = CommonHttpClient.httpClientBuilder
            .setDefaultRequestConfig(
                RequestConfig.custom()
                    .setSocketTimeout(
                        config.getIntegerDuration("so_timeout", 500))
                    .setConnectTimeout(
                        config.getIntegerDuration("connect_timeout", 200))
                    .setConnectionRequestTimeout(
                        config.getIntegerDuration("conn_pool_timeout", 500))
                    .setStaleConnectionCheckEnabled(false)
                    .setExpectContinueEnabled(false)
                .build())
            .build();

        suggestExecutor = new ThreadPoolExecutor(
            threads,
            threads,
            1,
            TimeUnit.MINUTES,
            new ArrayBlockingQueue<Runnable>(queueSize),
            new ThreadPoolExecutor.AbortPolicy());
        printKeysCache = CacheBuilder.newBuilder()
                .maximumSize(config.getInt("printkeys_cache_size", 1000))
                .expireAfterAccess(config.getInt("printkeys_cache_expire",
                    30000),TimeUnit.MILLISECONDS)
                .build(cacheLoader);

        ImmutableFactorsLogConfig logFactorsConfig =
            proxyConfig.tskvLogConfig();
        if (logFactorsConfig != null) {
            this.factorsLogger = new ProxyTskvLogger(proxyConfig);
        } else {
            this.factorsLogger = null;
        }
    }


    private static class CacheKey {
        public final HttpServer.RequestContext ctx;
        public final String suid;
        public final String db;
        public final Set<String> excludeSet;
        public final SearchMap searchMap;

        public CacheKey(final HttpServer.RequestContext ctx,
            final String suid,
            final String db,
            final Set<String> excludeSet,
            final SearchMap searchMap)
        {
            this.ctx = ctx;
            this.suid = suid;
            this.db = db;
            this.excludeSet = excludeSet;
            this.searchMap = searchMap;
        }

        @Override
        public int hashCode() {
            return suid.hashCode();
        }

        @Override
        public boolean equals(Object o) {
            CacheKey other = (CacheKey)o;
            return suid.equals(other.suid);
        }

    }

    public long cacheSize() {
        return printKeysCache.size();
    }

    public CacheStats cacheStats() {
        return printKeysCache.stats();
    }
    public Set<CacheKey> cachekeySet() {
        return printKeysCache.asMap().keySet();
    }

    public int search( HttpServer.RequestContext ctx, String request, String requestType, HttpParams params, HttpServer.HttpHeaders headers, PrintStream ps ) throws ApiException, MsearchProxyException
    {
        if (!ENABLED) {
            throw new ApiException("suggest api  is disabled", 403);
        }
        final long requestStartTime = System.currentTimeMillis();
	String requestStr = params.get("q");
	if( requestStr == null )
	{
	    // preserving old alias for now
	    requestStr = params.get("text");
	    if( requestStr == null )
	    {
	    throw new ApiException( "missing parameter q" );
//	    requestStr = "";
	    }
	    if( requestStr.equals("") )
	    {
	    throw new ApiException( "empty request" );
	    }
	}

        requestStr = requestStr.replaceAll( "[\"*]", "" );
        requestStr = requestStr.replace('ё', 'e').replace('Ё', 'Е');

        if( requestStr.startsWith("someqtrnvhkly ") )
        {
            requestStr = requestStr.substring( "someqtrnvhkly ".length() );
        }

        int limit = TOTAL_LIMIT;

        String limitStr = params.get("limit");
        if( limitStr != null ) limit = Integer.parseInt(limitStr);

        String db = params.get("maildb");
        if (db == null || db != null && db.isEmpty()) {
        // preserving old alias for now
            db = params.get("db");
            if (db == null || db != null && db.isEmpty()) {
                db = "mdb303";
            }
        }

        String suid = params.get("uid");
        if (suid == null || suid != null && suid.isEmpty()) {
        // preserving old alias for now
            suid = params.get("kps");
            if (suid == null || suid != null && suid.isEmpty()) {
                throw new ApiException("missing uid");
            }
        }

        boolean mixGroups = true;
        if (params.get("mix_groups") != null) {
            String mix = params.get("mix_groups");
            if (mix.equalsIgnoreCase("false") ||
                mix.equals("0") ||
                mix.equalsIgnoreCase("no")
            ) {
                mixGroups = false;
            }
        }

        HashSet<String> excludeSet = new HashSet<String>();
        HashSet<String> selfEmails = new HashSet<String>();
        if( params.count("emails") > 0 )
        {
            for( int i = 0; i < params.count("emails"); i++ )
            {
                String value = params.get("emails", i);
                if (value == null) continue;
                value = value.trim();
                if (value.length() == 0) continue;
                String[] emails = params.get("emails",i).split(",");
                for (String e : emails) {
                    if (e.trim().length() == 0) continue;
                    selfEmails.add(e.trim());
                }
            }
        }
//        selfEmails.add("uliyana-u@yandex.ru");
//        selfEmails.add("uliyana-u@ya.ru");
//        selfEmails.add("uriko.s@gmail.com");
//        selfEmails.add("julia@centrida.ru");
//        selfEmails.add("juliachina@mail.ru");

        excludeSet.addAll(Arrays.asList(excludeNoEmails));
        excludeSet.addAll(selfEmails);

        SearchMap searchMap = SearchMap.getInstance();

        Set<String> keySet = getHeadersKeySet(
            new CacheKey(ctx, suid, db, excludeSet, searchMap));

        HashSet<String> groupSet = new HashSet<String>();

	GroupingSortingCollector collector = new GroupingSortingCollector("email_address_lower", "date_boost");

        ArrayList<Rule> rulesArray = new ArrayList<Rule>();
        {
//            String[] suggest = Translit.suggest( requestStr );
            String[] suggest = new String[3];
            suggest[0] = requestStr;
            suggest[1] = Translit.layoutEnRu(requestStr);
            suggest[2] = Translit.layoutRuEn(requestStr);
            for( String s : suggest )
            {
                Rule[] rules = parseRequestString( ctx, s, keySet );
                rulesArray.addAll(Arrays.asList(rules));
            }
        }
        Rule[] rules = rulesArray.toArray( new Rule[rulesArray.size()] );

        String[] firstGroupHeaders = new String[] {"hdr_to"};
//        String firstGroupRequest = generateRequest(rules, firstGroupHeaders, "AND message_type:4") + 
//            " OR " +
//            generateRequest(rules, firstGroupHeaders, "AND NOT message_type:*");
        String selfEmailsSuffix = null;
        if (selfEmails.size() > 0) {
            selfEmailsSuffix = "AND hdr_from:(" +
                stringsConcat(selfEmails, " OR ") +
                ")";

        }
        String firstGroupRequest = generateRequest(
            rules,
            firstGroupHeaders,
            selfEmailsSuffix
            );

//        String[] secondGroupHeaders = new String[] {"hdr_from", "hdr_cc"};
        String[] secondGroupHeaders = new String[] {"hdr_from", "hdr_cc", "hdr_to"};
//        String secondGroupRequest = "(" +
//            generateRequest(rules, secondGroupHeaders, null) +
//            ") OR (" +
//            generateRequest(rules, firstGroupHeaders, "AND NOT message_type:4") +
//            ")";
        String invertedFirstRequest = "";
        if (selfEmails.size() > 0) {
            invertedFirstRequest = 
                " OR " + 
                generateRequest(
                    rules,
                    firstGroupHeaders,
                    "AND NOT hdr_from:(" +
                    stringsConcat(selfEmails, " OR ") +
                    ")"
                );
        }
        String secondGroupRequest =
            generateRequest(rules, secondGroupHeaders, null) +
                invertedFirstRequest;
        final long deadline = requestStartTime + maxRequestTime;
        Future<Collector> hdrToCollectorFuture =
            suggestExecutor.submit(
                new EmailSuggestTask(
                    deadline,
                    firstGroupRequest,
                    firstGroupHeaders,
                    suid,
                    db,
                    ctx,
                    searchMap,
                    excludeSet));
        Future<Collector> hdrFromCollectorFuture =
            suggestExecutor.submit(
                new EmailSuggestTask(
                    deadline,
                    secondGroupRequest,
                    secondGroupHeaders,
                    suid,
                    db,
                    ctx,
                    searchMap,
                    excludeSet));
        try
        {
            Collector emailCollector = hdrToCollectorFuture.get();
            Map<Document,Document> emailDocuments = emailCollector.hits();
            ctx.log.info( "EmailHits: " + emailDocuments.size() );
            ctx.setContentType( "application/json" );

            Iterator<Document> iter = emailDocuments.keySet().iterator();
            collectAllEmails( rules, iter, collector, limit, 1, groupSet );

            if( params.count("debug") > 0 )
            {
	        OutputPrinter printer = new OutputPrinter( ctx.getSessionId(), params, ps );
	        printer.print( collector );
            }

            if( groupSet.size() < limit && mixGroups || groupSet.size() == 0 )
            {
                emailCollector = hdrFromCollectorFuture.get();
                emailDocuments = emailCollector.hits();
                iter = emailDocuments.keySet().iterator();

                collectAllEmails( rules, iter, collector, limit, 0, groupSet );

                if( params.count("debug") > 0 )
                {
	            OutputPrinter printer = new OutputPrinter( ctx.getSessionId(), params, ps );
	            printer.print( collector );
                }

            } else {
                emailCollector = hdrFromCollectorFuture.get();
//                hdrFromCollectorFuture.cancel(true);
            }

            ProxyTskvLogger tskvLogger = null;
            String kps = params.get("suid");
            if (kps != null && kps.endsWith("70")) {
                tskvLogger =
                    SuggestFactors.prepareLogger(
                        factorsLogger,
                        requestStr,
                        ctx,
                        params);
            }
            if (params.count("debug") == 0) {
                printResults( collector, ps, requestStr, tskvLogger, TOTAL_LIMIT );
            }
/*
            Iterator<Document> group0Docs = emailDocuments.keySet().iterator();
            Iterator<Document> group1Docs = emailDocuments.keySet().iterator();
        
            collectAfterAtEmails( rules, group1Docs, collector, limit, 0, groupSet );


            if( params.count("debug") > 0 )
            {
	        OutputPrinter printer = new OutputPrinter( ctx.getSessionId(), params, ps );
	        printer.print( collector );
            }


            if( groupSet.size() >= GROUP_1_LIMIT ) limit = GROUP_0_LIMIT;
            else limit = TOTAL_LIMIT - groupSet.size();

            collectBeforeAtEmails( rules, group0Docs, collector, limit, 1, groupSet );


            if( params.count("debug") > 0 )
            {
	        OutputPrinter printer = new OutputPrinter( ctx.getSessionId(), params, ps );
	        printer.print( collector );
            }
            else
            {
                printResults( collector, ps, requestStr, TOTAL_LIMIT );
            }
*/
        }
        catch( InterruptedException e )
        {
            throw new ApiException(e);
        }
        catch( ExecutionException e )
        {
            throw new ApiException(e);
        }
        catch( JSONException e )
        {
            throw new ApiException(e);
        }
        catch( IOException e )
        {
            throw new ApiException(e);
        }

	return 0;
    }

    private static String stringsConcat(
        final Collection<String> strings,
        final String separator)
    {
        StringBuilder sb = new StringBuilder();
        String sep = "";
        for (String s : strings) {
            sb.append(sep);
            sb.append(s);
            sep = separator;
        }
        return sb.toString();
    }

    public Set<String> getHeadersKeySet(CacheKey key) throws ApiException {
        try {
            return printKeysCache.get(key);
        } catch (ExecutionException e) {
            throw new ApiException("Get printkeys error", e, 500);
        }
    }

    public Set<String> getHeadersKeySetImpl(CacheKey key) throws ApiException {
        LinkedList<SearchMap.Host> searchMapHosts =
            key.searchMap.getHosts(key.db, key.suid);
        if (searchMapHosts == null) {
            throw new ApiException("Suggest.GetKeysTask: error: user <"
                + key.suid + "> cannot be bound to any server.");
        }
        ArrayList<HttpHost> httpHosts = new ArrayList<HttpHost>(searchMapHosts.size());
        for (SearchMap.Host smHost : searchMapHosts) {
            httpHosts.add(
                new HttpHost(
                    smHost.params.get("host"),
                    Integer.parseInt(smHost.params.get("search_port"))));
        }
        final AsyncClient client = key.ctx.server().searchClient().adjust(
            key.ctx.getSessionId(),
            key.ctx.referer());

        final BasicRequestsListener requestDetailer =
            new BasicRequestsListener();
        final long currentTime = System.currentTimeMillis();
        final long deadline = currentTime + maxGetHeadersKeySetTime;
        try {
            LinkedList<Future<String>> futures = new LinkedList<>();
            TreeSet<String> keySet =
                new TreeSet<String>();
            for (String header : searchHeaders) {
                final String request = "/?printkeys&user=" + key.suid + "&field="
                    + header + "&prefix="+ key.suid + "%23";
                final BasicAsyncRequestProducerGenerator httpGet =
                    new BasicAsyncRequestProducerGenerator(request);
                futures.add(
                    client.execute(
                        httpHosts,
                        httpGet,
                        deadline,
                        switchToNextDelay,
                        AsyncStringConsumerFactory.OK,
                        requestDetailer.createContextGeneratorFor(client),
                        EmptyFutureCallback.INSTANCE));
            }
            for (Future<String> f : futures) {
                String keysList = f.get();
                for (String line : keysList.split("\n")) {
                    int sep = line.indexOf('#');
                    if (sep == -1) {
                        continue;
                    }
                    final String oneKey = line.substring(sep + 1);
                    if (key.excludeSet.contains(oneKey)) {
                        continue;
                    }
                    keySet.add(oneKey);
                }
            }
            key.ctx.log.debug(
                "Suggest.getHeaders<" + key.suid
                + "> got " + keySet.size() + " keys in "
                + (System.currentTimeMillis() - currentTime) + " ms");
            key.ctx.log.debug(
                "Suggest.getHeaders<" + key.suid
                + "> request details: "
                + requestDetailer);
            return keySet;
        } catch (Exception e) {
            key.ctx.log.debug(
                "Suggest.getHeaders<" + key.suid
                 + "> failed to get keys in "
                 + (System.currentTimeMillis() - currentTime) + " ms"
                 + ": " + requestDetailer);
            throw new ApiException(e);
        }
/*
        try {
            LinkedList<Future> futures = new LinkedList<Future>();
            String firstHeader = null;
            TreeSet<String> keySet =
                new TreeSet<String>();

            for (String header : searchHeaders) {
                if (firstHeader == null) {
                    firstHeader = header;
                    continue;
                }
                futures.add(
                    suggestExecutor.submit(
                        new GetKeysTask(key.suid,
                            key.db,
                            key.ctx,
                            key.searchMap,
                            key.excludeSet,
                            header,
                            keySet)));
            }
            //Process first header in the current thread to save executor
            GetKeysTask task = new GetKeysTask(key.suid,
                key.db,
                key.ctx,
                key.searchMap,
                key.excludeSet,
                firstHeader,
                keySet);
            task.call();

            for (Future f : futures) {
                f.get();
            }
            return keySet;
        } catch (Exception e) {
            throw new ApiException(e);
        }
*/
    }

    private static class Rule {
        public final List<Token[]> tokens;
        public boolean single = false;
        public boolean searchable;

        public Rule(final List<Token[]> tokens, final boolean searchable) {
            this.tokens = tokens;
            if( tokens.size() == 1 ) single = true;
            this.searchable = searchable;
        }
    }

    public static class RuleComparator {
        private final HashSet<Set<Token>> tokens;
        private final int hashCode;
        public RuleComparator(final List<Token[]> tokens) {
            this.tokens = new HashSet<Set<Token>>();
            int hashCode = 0;
            for (Token[] tokensArray : tokens) {
                this.tokens.add(new HashSet<Token>(Arrays.asList(tokensArray)));
                for (Token token : tokensArray) {
                    hashCode += token.hashCode();
                }
            }
            this.hashCode = hashCode;
        }

        @Override
        public int hashCode() {
            return this.hashCode;
        }

        @Override
        public boolean equals(Object o) {
            RuleComparator other = (RuleComparator)o;
            return tokens.equals(other.tokens);
        }

    }

    public static class Token {
        public final String str;
        public final boolean finished;

        public Token(final String str, final boolean finished) {
            this.str = str;
            this.finished = finished;
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof Token) {
                Token other = (Token) o;
                return str.equals(other.str) && finished == other.finished;
            }
            return false;
        }

        @Override
        public int hashCode() {
            return str.hashCode();
        }
    }

    private Rule[] parseRequestString(
        final HttpServer.RequestContext ctx,
        String str,
        final Set<String> keySet)
    {
        ArrayList<Rule> rules = new ArrayList<Rule>();

        str = str.toLowerCase();

        //First rule is the whole request string
        {
            String[] suggest =
                Translit.suggestSet(str, Side.UNKNOWN).toArray(new String[0]);
            Token[] tokens = new Token[suggest.length];
            for( int i = 0; i < suggest.length; i++ )
            {
                tokens[i] = new Token( suggest[i], false );
            }
            List<Token[]> tokensList = new LinkedList<Token[]>();
            tokensList.add( tokens );
            rules.add( new Rule(tokensList, false) );
        }

        //Then tokenize request
        {
//            String[] stringTokens = str.toLowerCase().split( "[\\._\\-, \\t]" );
            String[] stringTokens = str.toLowerCase().split( "[\\._\\-, \\t]" );
            LinkedList<Token[]> tokensList = new LinkedList<Token[]>();
            boolean forceFinishToken = false;

            if( stringTokens.length == 1 && 
                (str.endsWith( " " ) || str.endsWith( "." ) || str.endsWith( "_" ) || str.endsWith( "-" ) || str.endsWith( "@" ))
            )
            {//Single token followed by a separator should be finalized
                forceFinishToken = true;
            }

            for( int i = 0; i < stringTokens.length; i++ )
            {
                if( stringTokens[i].trim().length() == 0 ) continue;
                ctx.log.info( "Request token[" + i + "]: " + stringTokens[i] );

                String[] tokenSuggests =
                    Translit.suggestSet(stringTokens[i], Side.UNKNOWN)
                        .toArray(new String[0]);

                HashSet<Token> set = new HashSet<Token>();

                boolean finishToken = false;

                for( String s : tokenSuggests )
                {
                    boolean getSynonyms = s.length() >= 3 ? true : false;

                    if( i < stringTokens.length - 1 || forceFinishToken ) //Not the last token - should finalize
                    {
                        finishToken = true;
                    }

                    set.add( new Token(s, finishToken) );

                    ctx.log.info( "Request tokens suggest: " + s + " (" + finishToken +"), getSynonyms: " + getSynonyms );

                    if( getSynonyms )
                    {
                        List<String> synonyms;
                        synonyms = this.synonyms.suggest( s );

                        for( String syn : synonyms )
                        {
                            if( keySet.contains(syn) )
                            {
                                if(
                                    set.add( new Token(syn,true) ) //Synonyms are allways finished
                                )
                                {
                                    ctx.log.info( "Request tokens suggest synonym: " + syn );
                                }
                            }
                            syn = Translit.translit(syn);
                            if( keySet.contains(syn) )
                            {
                                if(
                                    set.add( new Token(syn,true) ) //Synonyms are allways finished
                                )
                                {
                                    ctx.log.info( "Request tokens suggest synonym translit: " + syn );
                                }
                            }
                        }
                    }
                }
                tokensList.add( set.toArray( new Token[set.size()] ) );
            }
            rules.add( new Rule(tokensList, false) );
        }

        //Then tokenize request //searchable
        {
//            String[] stringTokens = str.toLowerCase().split( "[\\._\\-, \\t]" );
            String[] stringTokens = str.toLowerCase().split( "[\\._\\-\\@, \\t]" );
            LinkedList<Token[]> tokensList = new LinkedList<Token[]>();
            boolean forceFinishToken = false;

            if( stringTokens.length == 1 && 
                (str.endsWith( " " ) || str.endsWith( "." ) || str.endsWith( "_" ) || str.endsWith( "-" ) || str.endsWith( "@" ))
            )
            {//Single token followed by a separator should be finalized
                forceFinishToken = true;
            }

            for( int i = 0; i < stringTokens.length; i++ )
            {
                if( stringTokens[i].trim().length() == 0 ) continue;
                ctx.log.info( "Request token[" + i + "]: " + stringTokens[i] );

                String[] tokenSuggests =
                    Translit.suggestSet(stringTokens[i], Side.UNKNOWN)
                        .toArray(new String[0]);

                HashSet<Token> set = new HashSet<Token>();

                boolean finishToken = false;

                for( String s : tokenSuggests )
                {
                    boolean getSynonyms = s.length() >= 3 ? true : false;

                    if( i < stringTokens.length - 1 || forceFinishToken ) //Not the last token - should finalize
                    {
                        finishToken = true;
                    }

                    set.add( new Token(s, finishToken) );

                    ctx.log.info( "Request tokens suggest: " + s + " (" + finishToken +"), getSynonyms: " + getSynonyms );

                    if( getSynonyms )
                    {
                        List<String> synonyms;
                        synonyms = this.synonyms.suggest( s );

                        for( String syn : synonyms )
                        {
                            if( keySet.contains(syn) )
                            {
                                if(
                                    set.add( new Token(syn,true) ) //Synonyms are allways finished
                                )
                                {
                                    ctx.log.info( "Request tokens suggest synonym: " + syn );
                                }
                            }
                            syn = Translit.translit(syn);
                            if( keySet.contains(syn) )
                            {
                                if(
                                    set.add( new Token(syn,true) ) //Synonyms are allways finished
                                )
                                {
                                    ctx.log.info( "Request tokens suggest synonym translit: " + syn );
                                }
                            }
                        }
                    }
                }
                tokensList.add( set.toArray( new Token[set.size()] ) );
            }
            rules.add( new Rule(tokensList, true) );
        }


        return rules.toArray( new Rule[rules.size()] );
    }

    private static void printResults(Collector collector, PrintStream ps, String request, ProxyTskvLogger factorsLogger, int limit ) throws JSONException, IOException
    {
        OutputStreamWriter writer = new OutputStreamWriter(ps);
	JSONWriter jw = new JSONWriter( writer );
        jw.object();

//        jw.key( "request" );
//        jw.value( request );
//        writer.write( '\n' );

        Set<Document> docs = collector.hits().keySet();

//        jw.key( "found" );
//        jw.value( Math.min(docs.size(), limit) );
//        writer.write( '\n' );

        int wordsCount = 0;
        int requestLen = 0;
        int hasAt = 0;
        String requestNorm = "";
        if (factorsLogger != null) {
            requestNorm =
                SearchRequestText.normalize(request).trim().toLowerCase(Locale.ROOT);
            String[] words = requestNorm.split("\\s+");
            wordsCount = words.length;
            requestLen = requestNorm.length();

            if (requestNorm.contains("@")) {
                hasAt = 1;
            }
        }

        jw.key( "contacts" );
        jw.array();
        writer.write( '\n' );
        int i = 0;
        for( Document doc : docs )
        {
            if (factorsLogger != null) {
                double[] factors = SuggestFactors.create();
                SuggestFactors.set(factors, SuggestFactor.REQUEST_WORDS, wordsCount);
                SuggestFactors.set(factors, SuggestFactor.REQUEST_SYMBOLS, requestLen);
                SuggestFactors.set(factors, SuggestFactor.REQUEST_AT, hasAt);
                SuggestFactors.oldContacts(factorsLogger, factors, requestNorm, doc);
            }
            String email_address = doc.getAttr("email_address");
            String email_name = doc.getAttr("email_name");

            if (email_name.equals("")) {
                email_name = email_address;
            }
            if (Strings.isUpperCase(email_address)) {
                email_address = email_address.toLowerCase();
            }
            if (email_address.toLowerCase().trim().
                    equals(email_name.toLowerCase().trim())) {
                String[] parts = email_name.split("@");
                if (parts.length > 0) {
                    email_name = parts[0];
                }
            }
            // TODO considering this
//            if (Strings.isUpperCase(email_name) || Strings.isLowerCase(email_name)) {
//                email_name = WordUtils.capitalizeFully(email_name);
//            }

           jw.object();
           jw.key("id");
           jw.value(-2);
           jw.key("email");
           jw.value(email_address);
           jw.key("name");
           jw.value(email_name);
           jw.key("phones");
           jw.array();
           jw.endArray();
           jw.key("t");
           jw.value(doc.getAttr("received_date"));
           jw.key("u");
           jw.value(-1);
            jw.key("target");
            jw.value("contact");
            jw.key("show_text");
            jw.value(email_name);
            jw.key("search_text");
            jw.value(email_address);
           jw.endObject();

           writer.write( '\n' );
           if( ++ i >= limit ) break;
        }
        jw.endArray();
        jw.endObject();
//        jw.flush();
//        jw.close();
        writer.flush();
//        writer.close();
    }

    private static void collectEmails( String header, Rule[] rules, Iterator<Document> docs, Collector collector, int limit, int boost, Set<String> groupSet )
    {
        if( limit <= 0 ) return;
        int added = 0;
        groupSet.clear();
        while( docs.hasNext() && added < limit )
        {
            Document doc = docs.next();
            String emailStr = doc.getAttr( header );
            if( emailStr == null ) continue;
            Email email = parseEmail( emailStr );
            String emailName = email.name;
            String emailAddress = email.address.toLowerCase();
            for( Rule rule : rules )
            {
                if( rule.searchable ) continue;
                boolean matched = true;
                for( Token[] orTokens : rule.tokens )
                {
                    if( !(
                        displayNameMatchesPrefix( emailName, orTokens ) ||
                        addressMatchesPrefixBeforeAt( emailAddress, orTokens ) ||
                        addressMatchesPrefixAfterAt( emailAddress, orTokens )
                        )
                    )
                    {
                        matched = false;
                        break;
                    }
                }
                if( matched )
                {
                    if( !groupSet.contains( emailAddress ) )
                    {
                        groupSet.add( emailAddress );
                        added++;
                    }
                    doc = doc.clone( false );
                    doc.setAttr("date_boost", boost + "_" + doc.getAttr("received_date") );
                    doc.setMerged( null );
                    collector.collect( doc );
                    collector.incTotalCount( 1 );
                    break;
                }
            }
        }
    }

    private static void collectAllEmails( Rule[] rules, Iterator<Document> docs, Collector collector, int limit, int boost, Set<String> groupSet )
    {
        if( limit <= 0 ) return;
        int added = 0;
        groupSet.clear();
        while( docs.hasNext() && added < limit )
        {
            Document doc = docs.next();
            String emailName = doc.getAttr( "email_name" );
            String emailAddress = doc.getAttr( "email_address_lower" );
            for( Rule rule : rules )
            {
                if( rule.searchable ) continue;
                boolean matched = true;
                for( Token[] orTokens : rule.tokens )
                {
                    if( !(
                        displayNameMatchesPrefix( emailName, orTokens ) ||
                        addressMatchesPrefixBeforeAt( emailAddress, orTokens ) ||
                        addressMatchesPrefixAfterAt( emailAddress, orTokens )
                        )
                    )
                    {
                        matched = false;
                        break;
                    }
                }
                if( matched )
                {
                    if( !groupSet.contains( emailAddress ) )
                    {
                        groupSet.add( emailAddress );
                        added++;
                    }
                    doc = doc.clone( false );
                    doc.setAttr("date_boost", boost + "_" + doc.getAttr("received_date") );
                    doc.setMerged( null );
                    collector.collect( doc );
                    collector.incTotalCount( 1 );
                    break;
                }
            }
        }
    }

    private static void collectBeforeAtEmails( Rule[] rules, Iterator<Document> docs, Collector collector, int limit, int boost, Set<String> groupSet )
    {
        if( limit <= 0 ) return;
        int added = 0;
        groupSet.clear();
        while( docs.hasNext() && added < limit )
        {
            Document doc = docs.next();
            for( Rule rule : rules )
            {
                if( rule.searchable ) continue;
                boolean matched = true;
                boolean firstToken = true;
                for( Token[] orTokens : rule.tokens )
                {
                    if( firstToken )
                    {
                        if( !(
                            displayNameMatchesPrefix( doc.getAttr("email_name"), orTokens ) ||
                            addressMatchesPrefixBeforeAt( doc.getAttr("email_address_lower"), orTokens )
                            )
                        )
                        {
                            matched = false;
                            break;
                        }
                        firstToken = false;
                    }
                    else
                    {
                        if( !(
                            displayNameMatchesPrefix( doc.getAttr("email_name"), orTokens ) ||
                            addressMatchesPrefixBeforeAt( doc.getAttr("email_address_lower"), orTokens ) ||
                            addressMatchesPrefixAfterAt( doc.getAttr("email_address_lower"), orTokens )
                            )
                        )
                        {
                            matched = false;
                            break;
                        }
                    }
                }
                if( matched )
                {
                    if( !groupSet.contains( doc.getAttr("email_address_lower") ) )
                    {
                        groupSet.add( doc.getAttr("email_address_lower") );
                        added++;
                    }
                    doc = doc.clone( false );
                    doc.setAttr("date_boost", boost + "_" + doc.getAttr("received_date") );
                    doc.setMerged( null );
                    collector.collect( doc );
                    collector.incTotalCount( 1 );
                    break;
                }
            }
        }
    }

    private static void collectAfterAtEmails( Rule[] rules, Iterator<Document> docs, Collector collector, int limit, int boost, Set<String> groupSet )
    {
        if( limit <= 0 ) return;
        int added = 0;
        groupSet.clear();
        while( docs.hasNext() && added < limit )
        {
            Document doc = docs.next();
            for( Rule rule : rules )
            {
                if( rule.searchable ) continue;
                boolean matched = true;
                boolean firstToken = true;
                for( Token[] orTokens : rule.tokens )
                {
                if( !(
                    addressMatchesPrefixAfterAt( doc.getAttr("email_address_lower"), orTokens )
                    )
                )
                {
                    matched = false;
                    break;
                }
                }
                if( matched )
                {
                    if( !groupSet.contains( doc.getAttr("email_address_lower") ) )
                    {
                        groupSet.add( doc.getAttr("email_address_lower") );
                        added++;
                    }
                    doc = doc.clone( false );
                    doc.setAttr("date_boost", boost + "_" + doc.getAttr("received_date") );
                    doc.setMerged( null );
                    collector.collect( doc );
                    collector.incTotalCount( 1 );
                    break;
                }
            }
        }
    }

    private static boolean displayNameMatchesPrefix( String name, Token[] needles )
    {
        name = name.replace('ё', 'e').replace('Ё', 'Е');
        for( Token needle : needles )
        {
            if( needle.finished )
            {
                if( name.equals(needle.str) ) return true;
            }
            else
            {
                if( name.startsWith(needle.str) ) return true;
            }
        }
        String[] tokens = name.split( "[ \\t\\._\\-]" );
        for( String token : tokens )
        {
            String tokenLower = token.toLowerCase();
//            System.err.println( "Name: " + name + ", tokenLower=" + tokenLower );
            for( Token needle : needles )
            {
                if( needle.finished )
                {
                    if( tokenLower.equals(needle.str) ) return true;
                }
                else
                {
                    if( tokenLower.startsWith(needle.str) ) return true;
                }
            }
        }
        return false;
    }

    private static boolean addressMatchesPrefixBeforeAt( String address, Token[] needles )
    {
        for( Token needle : needles )
        {
            if( needle.finished )
            {
                if( address.equals(needle.str) ) return true;
            }
            else
            {
                if( address.startsWith(needle.str) ) return true;
            }
        }
        int atIndex = address.indexOf( '@' );
        if( atIndex == -1 ) return displayNameMatchesPrefix( address, needles );
        String[] nameTokens = address.substring( 0, atIndex ).split( "[\\._\\-@, \\t]" );
        for( int i = 0; i < nameTokens.length; i++ )
        {
            String tokenLower = nameTokens[i].toLowerCase();
            if( tokenLower.length() > 20 ) continue;
            for( Token needle : needles )
            {
                if( needle.finished )
                {
                    if( i == nameTokens.length - 1 )
                    {
                        if( (tokenLower + "@").equals(needle.str) ) return true;
                    }
                    if( tokenLower.equals(needle.str) ) return true;
                }
                else
                {
                    if( tokenLower.startsWith(needle.str) ) return true;
                    if( i == nameTokens.length - 1 )
                    {
                        if( (tokenLower + "@").startsWith(needle.str) ) return true;
                    }
                }
            }
        }
        return false;
    }

    private static boolean addressMatchesPrefixAfterAt( String address, Token[] needles )
    {
        int atIndex = address.indexOf( '@' );
        if( atIndex == -1 ) return displayNameMatchesPrefix( address, needles );
        String domain = address.substring( atIndex + 1 );
        String[] addressTokens = domain.split( "[\\._\\-@, \\t]" );
        boolean first = true;
        for( String token : addressTokens )
        {
            String tokenLower = token.toLowerCase();
            for( Token needle : needles )
            {
                if( needle.finished )
                {
                    if( tokenLower.equals(needle.str) ) return true;
                    if( first ) if( ("@" + tokenLower).equals(needle.str) ) return true;
                }
                else
                {
                    if( tokenLower.startsWith(needle.str) ) return true;
                    if( first ) if( ("@" + tokenLower).startsWith(needle.str) ) return true;
                }
                first = true;
            }
        }
        for( Token needle : needles )
        {
            if( needle.finished )
            {
                if( domain.equals(needle.str) ) return true;
                if( ("@" + domain).equals(needle.str) ) return true;
            }
            else
            {
                if( domain.startsWith(needle.str) ) return true;
                if( ("@" + domain).startsWith(needle.str) ) return true;
            }
        }
        return false;
    }

    private static String generateRequest(final Rule[] rules,
        final String[] headers,
        final String additional)
    {
	StringBuilder text = new StringBuilder();
        if (additional != null) {
            text.append("(");
        }
        HashSet<String> andRequestsSet = new HashSet<String>();
        HashSet<RuleComparator> ruleSet = new HashSet<RuleComparator>();
        boolean firstRule = true;
	for (Rule rule : rules) {
	    if (!rule.searchable) continue;
	    if (rule.tokens.size() == 0) continue;
	    if (ruleSet.contains(new RuleComparator(rule.tokens))) {
	        continue;
	    }
	    ruleSet.add(new RuleComparator(rule.tokens));
            StringBuilder ruleRequest = new StringBuilder();
            if (rule.tokens.size() > 1) {
                ruleRequest.append("(");
            }
	    boolean andFirst = true;
	    for (Token[] andRequests : rule.tokens) {
	        StringBuilder andRequest = new StringBuilder();

                andRequest.append("(");
                boolean orFirst = true;
                for (Token orRequest : andRequests) {
	            if (!orFirst) {
	                andRequest.append( " OR " );
	            } else {
	                orFirst = false;
	            }
	            String request = orRequest.str.replaceAll("@", "");
	            boolean firstHeader = true;
	            for (String header : headers) {
	                if (!firstHeader) andRequest.append(" OR ");
	                else firstHeader = false;
	                if (orRequest.finished) {
	                    andRequest.append( header + ":(" + request + ")" );
	                } else {
	                    andRequest.append( header + ":(" + request + "*)" );
	                }
	            }
	        }
	        andRequest.append(")");
	        String andRequestString = andRequest.toString();
	        if (!andRequestsSet.contains(andRequestString)) {
	            andRequestsSet.add(andRequestString);
	            if (!andFirst) {
//                        text.append(" AND ");
                        ruleRequest.append(" AND ");
	            } else {
	                andFirst = false;
	            }
	            ruleRequest.append(andRequestString);
	        }
            }
            if (rule.tokens.size() > 1) {
                ruleRequest.append(")");
            }
            String ruleRequestString = ruleRequest.toString();
            if (ruleRequestString.length() != 0 && !ruleRequestString.equals("()")) {
	        if (!firstRule) {
	            text.append(" OR ");
	        } else {
	            firstRule = false;
	        }
                text.append(ruleRequestString);
            }
        }
        if (additional != null) {
            text.append(" ");
            text.append(additional);
            text.append(")");
        }
        return text.toString();
    }

    private final static int scanEnd(String str, int from, int upto) {
        for( int i = from; i < upto; i++ )
        {
            if( str.charAt(i) == ' ' || 
                str.charAt(i) == '\n' || 
                str.charAt(i) == '\t' || 
                str.charAt(i) == '\r' ||
                str.charAt(i) == '\"' ||
                str.charAt(i) == '\'' ||
                str.charAt(i) == '\\' ||
                str.charAt(i) == ',' ||
                str.charAt(i) == '<' ||
                str.charAt(i) == '>' )
            {
                return i;
            }
        }
        return upto;
    }

    private final static int scanStart(String str, int from, int to) {
        for( int i = from; i >= to; i-- )
        {
//            if( str.charAt(i) == ' ' || str.charAt(i) == '\n' || str.charAt(i) == '\t' || str.charAt(i) == '\r' )
            if( str.charAt(i) == ' ' || 
                str.charAt(i) == '\n' || 
                str.charAt(i) == '\t' || 
                str.charAt(i) == '\r' ||
                str.charAt(i) == '\"' ||
                str.charAt(i) == '\'' ||
                str.charAt(i) == '\\' ||
                str.charAt(i) == ',' ||
                str.charAt(i) == '<' ||
                str.charAt(i) == '>' )
            {
                return i + 1;
            }
        }
        return 0;
    }

    private static final Email createEmail(final String str,
        final int emailStart, final int emailEnd)
    {
        String name;
        String address;
        if (emailStart > 0) {
            String possibleName = str.substring(0, emailStart);
            String unquoted = removeQuotes(possibleName)
                            .replace( ',', ' ' )
                            .replace( '\"', ' ' )
                            .replace( '\'', ' ' )
                            .replace( '\\', ' ' )
                            .replace( '<', ' ' )
                            .replace( '>', ' ' )
                            .trim();
            name = unquoted;
        } else {
            name = "";
        }
        address = str.substring( emailStart, emailEnd )
                            .replace( '\"', ' ' )
                            .replace( '\'', ' ' )
                            .replace( '\\', ' ' )
                            .replace( ',', ' ' )
                            .replace( '<', ' ' )
                            .replace( '>', ' ' )
                            .replace( ')', ' ' )
                            .replace( '(', ' ' )
                            .trim();
//        System.err.println("createEmail: name: " + name + ", email: " + address);
        return new Email(name, address);
    }

    private static Email parseEmail( String str )
    {
//        System.err.println("parseEmail: " + str);
        if (str.startsWith("=?") && str.endsWith("?=")
            || str.startsWith("<=?") && str.endsWith("?=>"))
        {
            String decoded =
                DecoderUtil.decodeEncodedWords(str, DecodeMonitor.SILENT);
            if (!decoded.equals(str)) {
                return parseEmail(decoded);
            }
        }
        int firstAt = str.indexOf( '@' );
        if(firstAt == -1) {
            return new Email("", str);
        }
        int emailStart = scanStart(str, firstAt, 0);
        int emailEnd = scanEnd(str, firstAt, str.length());
        Email email = createEmail(str, emailStart, emailEnd);
        if (emailStart > 0 && email.name.length() == 0) {
            //"email@address" email@address.com
            int secondAt = str.indexOf('@', firstAt + 1);
            if (secondAt != -1) {
                emailStart = scanStart(str, secondAt, firstAt);
                emailEnd = scanEnd(str, secondAt, str.length());
                return createEmail(str, emailStart, emailEnd);
            }
        }
        return email;
    }

    private static String removeQuotes( String str )
    {
        if( str.indexOf( '"' ) == -1 ) return str;
        StringBuilder sb = new StringBuilder();
        boolean addNext = false;
        for( int i = 0; i < str.length(); i++ )
        {
            char c = str.charAt(i);
            if( addNext )
            {
                sb.append( c );
                addNext = false;
            }
            if( c == '\\' )
            {
                addNext = true;
                continue;
            }
            else if( c == '"' )
            {
                continue;
            }
            sb.append( c );
        }
        return sb.toString();
    }

    private static class Email
    {
        String address;
        String name;
        public Email(final String name, final String address) {
            this.name = name;
            this.address = address;
        }
    }

    public static class EmailCollector extends GroupingSortingCollector
    {
    private Set<String> excludeSet;
    private String[] headers;
        public EmailCollector( Set<String> excludeSet, String[] headers )
        {
            super( "email_hash", "received_date" );
            this.excludeSet = excludeSet;
            this.headers = headers;
        }

        @Override
        public synchronized void collect( Document doc )
        {
//            String[] headers = { "hdr_from", "hdr_to", "hdr_cc" };
            for( String header : headers )
            {
                String headerStr = doc.getAttr(header);
                if( headerStr == null ) continue;
                headerStr = headerStr.trim();
                if( headerStr.length() == 0 || excludeSet.contains(headerStr) ) continue;
                Email email = parseEmail( headerStr );
                if( email == null ) continue;
                if( excludeSet.contains(email.address) ) continue;
//                System.err.println( "Parsed: " + email.name + ", " + email.address );
                doc.setAttr( "email_hash", "\"" + email.name + "\" " + email.address );
                doc.setAttr( "email_address", email.address );
                doc.setAttr( "email_address_lower", email.address.toLowerCase() );
                doc.setAttr( "email_name", email.name );
                super.collect( doc.clone(false) );
                incTotalCount( 1 );
            }
        }

    }

    private static final String cutChar( String s, Character c )
    {
        while( s.length() > 1 && s.indexOf(c) == 0 )
        {
            s = s.substring( 1 );
        }
        return s;
    }

    private static final String cutRequest( String request )
    {
        Character[] delimiters = { '_', '.', ' ', '@', '-' };
        for( Character d : delimiters )
        {
            request = cutChar( request, d );
        }
        for( Character d : delimiters )
        {
            if( request.indexOf( d ) != -1 )
            {
                return request.substring( 0, request.indexOf(d) );
            }
        }
        return request;
    }

    private class EmailSuggestTask implements Callable<Collector> {
        private final long deadline;
        private final String luceneRequest;
        private final String suid;
        private final String db;
        private final HttpServer.RequestContext ctx;
        private final SearchMap searchMap;
        private final Set<String> excludeSet;
        private final String[] headers;
        private final String logPrefix;
        public EmailSuggestTask(
            final long deadline,
            final String request,
            final String[] headers,
            final String suid, final String db,
            final HttpServer.RequestContext ctx,
            final SearchMap searchMap,
            final Set<String> excludeSet)
        {
            this.deadline = deadline;
            this.luceneRequest = request;
            this.suid = suid;
            this.db = db;
            this.ctx = ctx;
            this.searchMap = searchMap;
            this.excludeSet = excludeSet;
            this.headers = headers;
            logPrefix = "Suggest.EmailSearch<"
                + Integer.toHexString(luceneRequest.hashCode()) + ">: ";
        }

        @Override
        public Collector call() throws Exception {
            final BasicRequestsListener requestDetailer =
                new BasicRequestsListener();
            final long currentTime = System.currentTimeMillis();
            try {
                return run(requestDetailer);
            } finally {
                ctx.log.info(logPrefix + "time: "
                    + (System.currentTimeMillis() - currentTime) + " ms");
                ctx.log.info(logPrefix + requestDetailer);
            }
        }

        public Collector run(final BasicRequestsListener requestDetailer)
            throws Exception
        {
            final LinkedList<SearchMap.Host> searchMapHosts =
                searchMap.getHosts(db, suid);
            if (searchMapHosts == null) {
                throw new ApiException(logPrefix + "error: user <"
                    + suid + "> cannot be bound to any server.");
            }
            final ArrayList<HttpHost> httpHosts =
                new ArrayList<HttpHost>(searchMapHosts.size());
            for (SearchMap.Host smHost : searchMapHosts) {
                httpHosts.add(
                    new HttpHost(
                        smHost.params.get("host"),
                        Integer.parseInt(smHost.params.get("search_port"))));
            }
            final AsyncClient client = ctx.server().searchClient().adjust(
                ctx.getSessionId(),
                ctx.referer());

            QueryConstructor query =
                new QueryConstructor("/?");

            query.append("imap", "1");
            query.append("orderby", "received_date");
            query.append("format", "json");
            query.append("merge_func", "count");
            query.append("user", suid);
            query.append("getfields", "received_date,hdr_from,hdr_to,hdr_cc");

            final String queryText;
            if (luceneRequest.trim().length() == 0) {
                if (db.equalsIgnoreCase("pg")) {
                    queryText = "uid:" + suid;
                } else {
                    queryText = "suid:" + suid;
                }
            } else {
                queryText = luceneRequest;
            }
            ctx.log.info(logPrefix + "query text: " + queryText);
            query.append("text", queryText);

            if (headers.length > 1) {
                query.append("groupby", "multi(" + StringUtils.join(headers, ',') + ")");
            } else if(headers.length == 1) {
                query.append("groupby", headers[0]);
            } else {
                query.append("groupby", "received_date");
            }

            final String request = query.toString();
            ctx.log.info(logPrefix + "request: " + request);

            final BasicAsyncRequestProducerGenerator httpGet =
                new BasicAsyncRequestProducerGenerator(request);

            Future<SearchResult> result = client.execute(
                httpHosts,
                httpGet,
                deadline,
                switchToNextDelay,
                SearchResultConsumerFactory.OK,
                requestDetailer.createContextGeneratorFor(client),
                EmptyFutureCallback.INSTANCE);
            SearchResult sr = result.get();
            EmailCollector collector = new EmailCollector(excludeSet, headers);
            final List<SearchDocument> hitsArray = sr.hitsArray();
            collector.incTotalCount((int) sr.hitsCount());
            for (SearchDocument doc : hitsArray) {
                collector.collect(new Document(doc.attrs()));
            }
            return collector;
        }

        public Collector call1() throws Exception
        {
            EmailCollector collector = new EmailCollector(excludeSet, headers);
	    HttpParams params = new HttpParams();
	    params.replace("imap", "1");
	    
            Searcher searcher = new Searcher( ctx, 1, searchMap, params, collector );
	    String users[] = { suid };
	    searcher.search( db, users );
	    return collector;
        }

    }

    private class GetKeysTask implements Callable<Void> {
        private final String suid;
        private final String db;
        private final HttpServer.RequestContext ctx;
        private final SearchMap searchMap;
        private final Set<String> excludeSet;
        private final String field;
        private final Set<String> keySet;

        public GetKeysTask( String suid, String db, HttpServer.RequestContext ctx, SearchMap searchMap, Set<String> excludeSet, String field, Set<String> keySet )
        {
            this.suid = suid;
            this.db = db;
            this.ctx = ctx;
            this.searchMap = searchMap;
            this.excludeSet = excludeSet;
            this.field = field;
            this.keySet = keySet;
        }

        private final void processLine( String line )
        {
            int sep = line.indexOf( '#' );
            if( sep == -1 ) return;
            String key = line.substring( sep + 1 );
            if( excludeSet.contains(key) ) return;
            keySet.add( key );
        }

        public Void call() throws Exception
        {
	    LinkedList<SearchMap.Host> hosts = SearchMap.getInstance().getHosts( db, suid );
	    if( hosts == null )
            {
	        throw new ApiException( "Suggest.GetKeysTask: error: user <" + suid + "> cannot be bound to any server." );
	    }

	    boolean success = false;
	    StringBuilder error = new StringBuilder();
	    int errorHttpCode = 9999;
	    for( SearchMap.Host host : hosts )
	    {
	    String address = host.url;
	    String request = "http://" + address + "/?printkeys&user=" + suid + "&field=" + field + "&prefix="+ suid + "%23";
	    long start = System.currentTimeMillis();
	        ctx.log.info( "Requesting " + request );
                HttpGet get = new HttpGet(request);
                get.setHeader(
                    BackendAccessLoggerConfigDefaults.X_PROXY_SESSION_ID,
                    ctx.getSessionId());
                get.setHeader(ctx.referer());
	        InputStream is = null;

	        try
	        {
	    MeasurableHttpContext context = new MeasurableHttpContext();
		HttpResponse response = httpClient.execute( get, context );
		ctx.log.info(context.finAndGetInfo());
		
		HttpEntity httpEntity = response.getEntity();
		    if( httpEntity != null )
		    {
		        is = httpEntity.getContent();
		        if( response.getStatusLine().getStatusCode() < 200 || response.getStatusLine().getStatusCode() >= 400 )
		        {
		            if( errorHttpCode > response.getStatusLine().getStatusCode() )
		            {
		                errorHttpCode = response.getStatusLine().getStatusCode();
		            }
			    ctx.log.err( "Suggest.GetKeysTask: error: server returned status: " + response.getStatusLine().getStatusCode() );
			    BufferedReader r = new BufferedReader( new InputStreamReader( is, "UTF-8" ) );
			    String line;
			    if( error == null ) error = new StringBuilder(1000);
			    error.append( "Server " + address + " response" );
			    while( (line = r.readLine()) != null )
			    {
			        ctx.log.err( "Suggest.GetKeysTask: server response: " + line );
			        error.append( line );
			        error.append( '\n' );
			    }
			    continue;
		        }
		        else
		        {
			    BufferedReader r = new BufferedReader( new InputStreamReader( is, "UTF-8" ) );
			    String line;
			    while( (line = r.readLine()) != null )
			    {
			        processLine( line );
//			        ps.println( line );
			    }
			    success = true;
			    break;
		        }
		    }
	        }
	        catch( Exception e )
	        {
	            ctx.log.err( "Suggest.GetKeysTask: Can't get printkeys results from backend <" + host.url +">: " + Logger.exception(e) );
	            error.append( "Suggest.GetKeysTask: Can't get printkeys results from backend <" + host.url +">\n" );
	            continue;
	        }
	        finally
	        {
	            ctx.log.info( "Finished requesting " + request + " in " + (System.currentTimeMillis() - start) + " ms" );
	            try
	            {
		        if( is != null ) is.close();
		    }
		    catch( IOException ign )
		    {
		        ctx.log.err( "Suggest.GetKeysTask: unexpected exception: " + Logger.exception(ign) );
		    }
	        }
	    }

            if( !success )
            {
                throw new ApiException( error.toString() );
            }
            return null;
        }
    }

}
