package ru.yandex.msearch.proxy.config;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.msearch.proxy.api.async.mail.relevance.search.MailSearchRelevanceConfig;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.MailSearchRelevanceConfigBuilder;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.MailSearchRelevanceConfigDefaults;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;

public abstract class AbstractRankingConfigBuilder<T> implements RankingConfig {
    private static final String SUUGEST_PREFIX = "suggest-relevance";
    private static final String MAIL_SEARCH_PREFIX = "mail-search-relevance";

    private List<MailSearchRelevanceConfigBuilder> mailSearch;
    private boolean rankEmailRequests;
    private Set<String> excludedExperiments;

    protected AbstractRankingConfigBuilder(final RankingConfig config) {
        mailSearch(config.mailSearch());
        rankEmailRequests(config.rankEmailRequests());
        excludedExperiments(config.excludedExperiments());
    }

    protected AbstractRankingConfigBuilder(
        final IniConfig config,
        final RankingConfig defaults)
        throws ConfigException
    {
        mailSearch = new ArrayList<>();
        for (Map.Entry<String, IniConfig> entry
                : config.sections().entrySet())
        {
            if (entry.getKey().startsWith(MAIL_SEARCH_PREFIX)) {
                mailSearch.add(
                    new MailSearchRelevanceConfigBuilder(
                        entry.getValue(),
                        MailSearchRelevanceConfigDefaults.INSTANCE));
            }
        }

        excludedExperiments = config.getAll(
            "excluded-experiments",
            defaults.excludedExperiments(),
            new CollectionParser<>(String::trim, HashSet::new));

        rankEmailRequests = config.getBoolean(
            "rank-email-requests",
            defaults.rankEmailRequests());
    }

    public abstract T self();

    @Override
    public List<? extends MailSearchRelevanceConfigBuilder> mailSearch() {
        return mailSearch;
    }

    @Override
    public boolean rankEmailRequests() {
        return rankEmailRequests;
    }

    public T rankEmailRequests(final boolean value) {
        this.rankEmailRequests = value;
        return self();
    }

    public T excludedExperiments(final Set<String> excluded) {
        this.excludedExperiments = new HashSet<>(excluded);
        return self();
    }

    public T mailSearch(final List<? extends MailSearchRelevanceConfig> list) {
        this.mailSearch = new ArrayList<>();
        if (list != null) {
            this.mailSearch = list.stream()
                .map(MailSearchRelevanceConfigBuilder::new)
                .collect(Collectors.toList());
        }

        return self();
    }

    @Override
    public Set<String> excludedExperiments() {
        return excludedExperiments;
    }
}
