package ru.yandex.msearch.proxy.config;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import ru.yandex.matrixnet.AbstractMatrixnetModelConfigBuilder;

import ru.yandex.msearch.proxy.api.async.mail.Product;
import ru.yandex.msearch.proxy.api.async.mail.Side;

import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractRelevanceConfigBuilder<T extends AbstractRelevanceConfigBuilder<T>>
    extends AbstractMatrixnetModelConfigBuilder<T>
    implements RelevanceConfig
{
    public static final String FACTORS = "factors";
    public static final String PRODUCTS = "products";
    public static final String SIDES = "sides";
    public static final String USAGE_STATUS = "usage-status";
    public static final String TEST_ID = "test-id";

    private Set<String> factors;
    private Set<Product> products;
    private Set<Side> sides;
    private String testId;
    private RelevanceUsageStatus usageStatus;

    protected AbstractRelevanceConfigBuilder() {
        this(RelevanceConfigDefaults.INSTANCE);
    }

    protected AbstractRelevanceConfigBuilder(
        final RelevanceConfig config)
    {
        super(config);

        factors(config.factors());
        testId(config.testId());
        products(config.products());
        sides(config.sides());
        usageStatus(config.usageStatus());
    }

    protected AbstractRelevanceConfigBuilder(
        final IniConfig config,
        final RelevanceConfig defaults)
        throws ConfigException
    {
        super(config, defaults);

        factors = config.getAll(
            FACTORS,
            defaults.factors(),
            new CollectionParser<>(String::trim, LinkedHashSet::new));

        products = config.getAll(
            PRODUCTS,
            defaults.products(),
            new CollectionParser<>(Product::parse, HashSet::new));

        sides = config.getAll(
            SIDES,
            defaults.sides(),
            new CollectionParser<>(Side::parse, HashSet::new));

        usageStatus = config.getEnum(RelevanceUsageStatus.class,
            USAGE_STATUS,
            defaults.usageStatus());

        testId = config.getString(TEST_ID, defaults.testId());
    }

    @Override
    public Set<String> factors() {
        return factors;
    }

    public T factors(final Set<String> factors) {
        this.factors = new HashSet<>(factors);
        return self();
    }

    @Override
    public String testId() {
        return testId;
    }

    public T testId(final String testId) {
        this.testId = testId;
        return self();
    }

    @Override
    public Set<Product> products() {
        return products;
    }

    public T products(final Set<Product> products) {
        this.products = new HashSet<>(products);
        return self();
    }

    @Override
    public Set<Side> sides() {
        return sides;
    }

    public T sides(final Set<Side> sides) {
        this.sides = new HashSet<>(sides);
        return self();
    }

    @Override
    public RelevanceUsageStatus usageStatus() {
        return usageStatus;
    }

    public T usageStatus(final RelevanceUsageStatus value) {
        this.usageStatus = value;
        return self();
    }

    protected abstract T self();

    public Map<String, String> toMap() {
        Map<String, String> result = new LinkedHashMap<>();
        result.put(AbstractMatrixnetModelConfigBuilder.NAME, name());
        result.put(
            AbstractMatrixnetModelConfigBuilder.MODEL_FILE, file().toString());

        if (factors() != null) {
            result.put(FACTORS, String.join(",", factors()));
        }

        if (products != null) {
            StringBuilder sb = new StringBuilder();
            for (Product product: products) {
                sb.append(product.name());
                sb.append(',');
            }

            sb.setLength(sb.length() - 1);
            result.put(PRODUCTS, sb.toString());
        }

        if (sides!= null) {
            StringBuilder sb = new StringBuilder();
            for (Side side: sides) {
                sb.append(side.name());
                sb.append(',');
            }

            sb.setLength(sb.length() - 1);
            result.put(SIDES, sb.toString());
        }

        result.put(USAGE_STATUS, String.valueOf(usageStatus()));
        result.put(TEST_ID, testId());

        return result;
    }
}
