package ru.yandex.msearch.proxy.config;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpTargetConfig;

import ru.yandex.msearch.proxy.MsearchProxyExperiment;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequestParams;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.api.async.suggest.united.TargetWeight;

import ru.yandex.parser.config.ConfigException;

public class ImmutableSuggestConfig implements SuggestConfig {
    private final ImmutableDnsConfig dnsConfig;
    private final ImmutableSubjectSuggestConfig subjectConfig;
    private final ImmutableContactSuggestConfig contactConfig;
    private final ImmutableHttpTargetConfig searchClientConfig;
    private final ImmutableMailSuggestConfig mailConfig;

    private final Map<Target, TargetWeight> weights;
    private final Map<MsearchProxyExperiment, Map<Target, TargetWeight>> expsWeights;

    private final int searchClientResolution;
    private final int limit;
    private final boolean twoStepsRequest;
    private final boolean highlight;
    private final long timeout;

    private final boolean historyToContacts;
    private final boolean contactDomain;
    private final boolean sendersReceivers;
    private final boolean translitOriginalMandatory;

    public ImmutableSuggestConfig(final SuggestConfig config)
        throws ConfigException
    {
        this.dnsConfig = new ImmutableDnsConfig(config.dnsConfig());
        this.subjectConfig =
            new ImmutableSubjectSuggestConfig(config.subjectConfig());
        this.contactConfig =
            new ImmutableContactSuggestConfig(config.contactConfig());
        this.searchClientConfig =
            new ImmutableHttpTargetConfig(config.searchClientConfig());

        this.searchClientResolution = config.searchClientResolution();

        this.limit = config.limit();
        this.twoStepsRequest = config.twoStepsRequest();
        this.timeout = config.timeout();
        this.highlight = config.highlight();

        this.historyToContacts = config.historyToContacts();
        this.contactDomain = config.contactDomainSuggest();
        this.sendersReceivers = config.sendersReceivers();
        this.translitOriginalMandatory = config.translitOriginalMandatory();
        this.mailConfig =
            new ImmutableMailSuggestConfig(config.mailConfig());

        this.weights =
            Collections.unmodifiableMap(new LinkedHashMap<>(config.weights()));

        final Map<MsearchProxyExperiment, Map<Target, TargetWeight>> map =
            new LinkedHashMap<>();
        config.expsWeights().forEach(
            (k, v) -> map.put(k, new LinkedHashMap<>(v)));
        this.expsWeights = Collections.unmodifiableMap(map);
    }

    @Override
    public ImmutableDnsConfig dnsConfig() {
        return dnsConfig;
    }

    @Override
    public ImmutableSubjectSuggestConfig subjectConfig() {
        return subjectConfig;
    }

    @Override
    public ImmutableContactSuggestConfig contactConfig() {
        return contactConfig;
    }

    @Override
    public int limit() {
        return limit;
    }

    @Override
    public boolean twoStepsRequest() {
        return twoStepsRequest;
    }

    @Override
    public long timeout() {
        return timeout;
    }

    @Override
    public boolean highlight() {
        return highlight;
    }

    @Override
    public boolean sendersReceivers() {
        return sendersReceivers;
    }

    @Override
    public boolean historyToContacts() {
        return historyToContacts;
    }

    @Override
    public boolean contactDomainSuggest() {
        return contactDomain;
    }

    @Override
    public ImmutableHttpTargetConfig searchClientConfig() {
        return searchClientConfig;
    }

    @Override
    public int searchClientResolution() {
        return searchClientResolution;
    }

    @Override
    public boolean translitOriginalMandatory() {
        return translitOriginalMandatory;
    }

    @Override
    public ImmutableMailSuggestConfig mailConfig() {
        return mailConfig;
    }

    @Override
    public Map<Target, TargetWeight> weights() {
        return weights;
    }

    @Override
    public Map<MsearchProxyExperiment, Map<Target, TargetWeight>> expsWeights() {
        return expsWeights;
    }

    public Map<Target, TargetWeight> weights(
        final SuggestRequest<?> request)
    {
        SuggestRequestParams params = request.requestParams();
        if (!params.experiments().isEmpty()) {
            for (MsearchProxyExperiment experiment: params.experiments()) {
                Map<Target, TargetWeight> weights = expsWeights.get(experiment);
                if (weights != null) {
                    request.logger().info(
                        "Suggest weights for experiment "
                            + experiment.name());
                    return weights;
                }
            }
        }

        return weights;
    }
}
