package ru.yandex.msearch.proxy.config;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import ru.yandex.http.config.AbstractHttpTargetConfigBuilder;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.email.types.MessageTypeToString;

import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonNegativeIntegerValidator;

import ru.yandex.search.proxy.SearchProxyParams;

public class SubscriptionsConfigBuilder
    extends AbstractHttpTargetConfigBuilder<SubscriptionsConfigBuilder>
    implements SubscriptionsConfig
{
    private int minMailsInSubcscriptions;
    private List<MessageType> messageTypes;
    private int timeout;
    private int minimalMailAge;
    private File pinnedDisplayNames;
    private long failoverDelay;

    public SubscriptionsConfigBuilder() {
        this(SubscriptionsConfigDefaults.INSTANCE);
    }

    public SubscriptionsConfigBuilder(final SubscriptionsConfig config) {
        super(config);

        this.minimalMailAge(config.minimalMailAge());
        this.minMailsInSubcscriptions(config.minMailsInSubcscriptions());
        this.messageTypes(config.messageTypes());
        this.timeout(config.timeout());
        this.pinnedDisplayNames(config.pinnedDisplayNames());
        this.failoverDelay(config.failoverDelay());
    }

    public SubscriptionsConfigBuilder(
        final IniConfig config)
        throws ConfigException
    {
        this(config, SubscriptionsConfigDefaults.INSTANCE);
    }

    public SubscriptionsConfigBuilder(
        final IniConfig config,
        final SubscriptionsConfig defaults)
        throws ConfigException
    {
        super(config, defaults);

        this.messageTypes =
            config.get(
                "message-type",
                defaults.messageTypes(),
                new CollectionParser<>(
                    MessageTypeToString.INSTANCE,
                    ArrayList::new));
        this.minMailsInSubcscriptions =
            config.get(
                "minimal-size",
                defaults.minMailsInSubcscriptions(),
                NonNegativeIntegerValidator.INSTANCE);
        this.timeout = config.getInt("timeout", defaults.timeout());
        this.minimalMailAge =
            config.getInt("minimal-age-days", defaults.minimalMailAge());
        this.pinnedDisplayNames =
            config.getInputFile(
                "pinned-display-names",
                defaults.pinnedDisplayNames());
        this.failoverDelay =
            config.getLongDuration(
                SearchProxyParams.FAILOVER_DELAY,
                defaults.failoverDelay());
    }

    @Override
    public List<MessageType> messageTypes() {
        return messageTypes;
    }

    public SubscriptionsConfigBuilder messageTypes(
        final List<MessageType> messageTypes)
    {
        this.messageTypes = new ArrayList<>(messageTypes);
        return this;
    }

    @Override
    public int minMailsInSubcscriptions() {
        return minMailsInSubcscriptions;
    }

    public SubscriptionsConfigBuilder minMailsInSubcscriptions(
        final int minMailsInSubcscriptions)
    {
        this.minMailsInSubcscriptions = minMailsInSubcscriptions;
        return this;
    }

    @Override
    public int timeout() {
        return timeout;
    }

    public SubscriptionsConfigBuilder timeout(final int timeout) {
        this.timeout = timeout;
        return this;
    }

    @Override
    public long failoverDelay() {
        return failoverDelay;
    }

    public SubscriptionsConfigBuilder failoverDelay(final long delay) {
        this.failoverDelay = delay;
        return this;
    }

    @Override
    public int minimalMailAge() {
        return minimalMailAge;
    }

    public SubscriptionsConfigBuilder minimalMailAge(final int minimalMailAge) {
        this.minimalMailAge = minimalMailAge;
        return this;
    }

    @Override
    public SubscriptionsConfigBuilder self() {
        return this;
    }

    @Override
    public File pinnedDisplayNames() {
        return pinnedDisplayNames;
    }

    public SubscriptionsConfigBuilder pinnedDisplayNames(
        final File pinnedDisplayNames)
    {
        this.pinnedDisplayNames = pinnedDisplayNames;
        return this;
    }
}
