package ru.yandex.msearch.proxy.config;

import java.util.AbstractMap;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;

import ru.yandex.http.config.DnsConfig;
import ru.yandex.http.config.DnsConfigBuilder;
import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;

import ru.yandex.msearch.proxy.MsearchProxyExperiment;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.api.async.suggest.united.TargetWeight;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class SuggestConfigBuilder implements SuggestConfig {
    private DnsConfigBuilder dnsConfig;
    private SubjectSuggestConfigBuilder subjectConfig;
    private ContactSuggestConfigBuilder contactConfig;
    private MailSuggestConfigBuilder mailConfig;
    private HttpTargetConfigBuilder searchClientConfig;

    private int searchClientResolution;
    private int limit;
    private boolean twoStepsRequest;
    private boolean highlight;
    private long timeout;

    private boolean historyToContacts;
    private boolean contactDomain;
    private boolean sendersReceivers;
    private boolean translitOriginalMandatory;

    private Map<Target, TargetWeight> weights;
    private Map<MsearchProxyExperiment, Map<Target, TargetWeight>> expsWeights;

    public SuggestConfigBuilder(final SuggestConfig config) {
        this.dnsConfig(config.dnsConfig());
        this.contactConfig(config.contactConfig());
        this.subjectConfig(config.subjectConfig());
        this.mailConfig(config.mailConfig());
        this.limit(config.limit());
        this.twoStepsRequest(config.twoStepsRequest());
        this.timeout(config.timeout());
        this.highlight(config.highlight());

        this.sendersReceivers(config.sendersReceivers());
        this.contactDomainSuggest(config.contactDomainSuggest());
        this.historyToContacts(config.historyToContacts());

        this.searchClientConfig(config.searchClientConfig());
        this.searchClientResolution(config.searchClientResolution());
        this.translitOriginalMandatory(config.translitOriginalMandatory());

        this.weights(config.weights());
        this.expsWeights(config.expsWeights());
    }

    public SuggestConfigBuilder(
        final IniConfig config,
        final SuggestConfig defaults)
        throws ConfigException
    {
        this.dnsConfig =
            new DnsConfigBuilder(config.section("dns"), defaults.dnsConfig());
        IniConfig contactSection = config.sectionOrNull("contact");
        if (contactSection == null) {
            this.contactConfig =
                new ContactSuggestConfigBuilder(defaults.contactConfig());
        } else {
            this.contactConfig =
                new ContactSuggestConfigBuilder(
                    contactSection,
                    defaults.contactConfig());
        }

        IniConfig subjectSection = config.sectionOrNull("subject");
        if (subjectSection == null) {
            this.subjectConfig =
                new SubjectSuggestConfigBuilder(defaults.subjectConfig());
        } else {
            this.subjectConfig =
                new SubjectSuggestConfigBuilder(
                    subjectSection,
                    defaults.subjectConfig());
        }

        IniConfig mailSection = config.sectionOrNull("mail");
        if (mailSection == null) {
            this.mailConfig =
                new MailSuggestConfigBuilder(defaults.mailConfig());
        } else {
            this.mailConfig =
                new MailSuggestConfigBuilder(
                    mailSection,
                    defaults.mailConfig());
        }

        IniConfig searchSection = config.sectionOrNull("search");
        if (searchSection != null) {
            this.searchClientConfig =
                new HttpTargetConfigBuilder(
                    searchSection,
                    defaults.searchClientConfig());
        } else {
            this.searchClientConfig =
                new HttpTargetConfigBuilder(defaults.searchClientConfig());
        }

        this.searchClientResolution =
            config.getInt(
                "timer.resolution",
                defaults.searchClientResolution());

        this.limit = config.getInt("limit", defaults.limit());
        this.twoStepsRequest =
            config.getBoolean("two-step-request", defaults.twoStepsRequest());

        this.timeout =
            config.getLong("timeout", defaults.timeout());

        this.highlight = config.getBoolean("highlight", defaults.highlight());

        this.historyToContacts =
            config.getBoolean("history-contact", defaults.historyToContacts());

        this.contactDomain =
            config.getBoolean(
                "contact-domain",
                defaults.contactDomainSuggest());

        this.sendersReceivers =
            config.getBoolean(
                "senders-receivers",
                defaults.sendersReceivers());

        this.translitOriginalMandatory =
            config.getBoolean(
                "translit-original-mandatory",
                defaults.translitOriginalMandatory());

        IniConfig weightsSection = config.section("weights");
        weights = new LinkedHashMap<>(defaults.weights());
        expsWeights = new LinkedHashMap<>(defaults.expsWeights());

        if (weightsSection != null) {
            for (Map.Entry<String, IniConfig> entry
                : weightsSection.sections().entrySet()) {
                Map.Entry<Target, TargetWeight> pair;
                try {
                    pair = parseWeights(entry);
                    weights.put(pair.getKey(), pair.getValue());
                } catch (ConfigException e) {
                    //assume it experiment
                    MsearchProxyExperiment experiment;
                    try {
                        experiment =
                            MsearchProxyExperiment.valueOf(
                                entry.getKey().toUpperCase(Locale.ENGLISH));
                    } catch (IllegalArgumentException iae) {
                        throw new ConfigException(
                            "Invalid experiment name " + entry.getKey()
                                + " expected one of "
                                + Arrays.toString(
                                    MsearchProxyExperiment.values()));
                    }
                    Map<Target, TargetWeight> expWeights
                        = new LinkedHashMap<>();

                    for (Map.Entry<String, IniConfig> expEntry
                        : entry.getValue().sections().entrySet()) {
                        pair = parseWeights(expEntry);
                        expWeights.put(pair.getKey(), pair.getValue());
                    }

                    expsWeights.put(experiment, expWeights);
                }
            }
        }
    }

    private Map.Entry<Target, TargetWeight> parseWeights(
        final Map.Entry<String, IniConfig> entry)
        throws ConfigException
    {
        String key = entry.getKey();
        try {
            Target target =
                Target.valueOf(
                    key.trim().toUpperCase(Locale.ENGLISH));
            int groupWeight = entry.getValue().getInt("group-weight");
            int weight = entry.getValue().getInt("target-weight");
            int limit = entry.getValue().getInt("limit", -1);

            return new AbstractMap.SimpleEntry<>(
                target,
                new TargetWeight(groupWeight, weight, limit));
        } catch (IllegalArgumentException iae) {
            throw new ConfigException(
                "Invalid target " + key,
                iae);
        }
    }

    @Override
    public DnsConfigBuilder dnsConfig() {
        return dnsConfig;
    }

    public SuggestConfigBuilder dnsConfig(final DnsConfig dnsConfig) {
        this.dnsConfig = new DnsConfigBuilder(dnsConfig);
        return this;
    }

    @Override
    public SubjectSuggestConfigBuilder subjectConfig() {
        return subjectConfig;
    }

    public SuggestConfigBuilder subjectConfig(
        final SubjectSuggestConfig config)
    {
        this.subjectConfig = new SubjectSuggestConfigBuilder(config);
        return this;
    }

    @Override
    public ContactSuggestConfigBuilder contactConfig() {
        return contactConfig;
    }

    public SuggestConfigBuilder contactConfig(
        final ContactSuggestConfig config)
    {
        this.contactConfig = new ContactSuggestConfigBuilder(config);
        return this;
    }

    @Override
    public int limit() {
        return limit;
    }

    public SuggestConfigBuilder limit(final int limit) {
        this.limit = limit;
        return this;
    }

    @Override
    public boolean twoStepsRequest() {
        return twoStepsRequest;
    }

    public SuggestConfigBuilder twoStepsRequest(final boolean tsr) {
        this.twoStepsRequest = tsr;
        return this;
    }

    @Override
    public long timeout() {
        return timeout;
    }

    public SuggestConfigBuilder timeout(final long timeout) {
        this.timeout = timeout;
        return this;
    }

    @Override
    public boolean highlight() {
        return highlight;
    }

    public SuggestConfigBuilder highlight(final boolean value) {
        this.highlight = value;
        return this;
    }

    @Override
    public boolean sendersReceivers() {
        return sendersReceivers;
    }

    public SuggestConfigBuilder sendersReceivers(final boolean v) {
        this.sendersReceivers = v;
        return this;
    }

    @Override
    public boolean historyToContacts() {
        return historyToContacts;
    }

    public SuggestConfigBuilder historyToContacts(final boolean v) {
        this.historyToContacts = v;
        return this;
    }

    @Override
    public boolean contactDomainSuggest() {
        return contactDomain;
    }

    public SuggestConfigBuilder contactDomainSuggest(final boolean v) {
        this.contactDomain = v;
        return this;
    }

    @Override
    public HttpTargetConfigBuilder searchClientConfig() {
        return searchClientConfig;
    }

    public SuggestConfigBuilder searchClientConfig(
        final HttpTargetConfig searchClientConfig)
    {
        this.searchClientConfig =
            new HttpTargetConfigBuilder(searchClientConfig);
        return this;
    }

    @Override
    public int searchClientResolution() {
        return searchClientResolution;
    }

    public SuggestConfigBuilder searchClientResolution(
        final int searchClientResolution)
    {
        this.searchClientResolution = searchClientResolution;
        return this;
    }

    public boolean translitOriginalMandatory() {
        return translitOriginalMandatory;
    }

    public SuggestConfigBuilder translitOriginalMandatory(
        final boolean translitOriginalMandatory)
    {
        this.translitOriginalMandatory = translitOriginalMandatory;
        return this;
    }

    @Override
    public MailSuggestConfigBuilder mailConfig() {
        return mailConfig;
    }

    public SuggestConfigBuilder mailConfig(
        final MailSuggestConfig config)
    {
        this.mailConfig = new MailSuggestConfigBuilder(config);
        return this;
    }

    @Override
    public Map<Target, TargetWeight> weights() {
        return weights;
    }

    public SuggestConfigBuilder weights(
        final Map<Target, TargetWeight> weights)
    {
        this.weights = new LinkedHashMap<>(weights);
        return this;
    }

    @Override
    public Map<MsearchProxyExperiment, Map<Target, TargetWeight>> expsWeights() {
        return expsWeights;
    }

    public SuggestConfigBuilder expsWeights(
        final Map<MsearchProxyExperiment, Map<Target, TargetWeight>> map)
    {
        this.expsWeights = new LinkedHashMap<>();
        map.forEach((k, v) -> expsWeights.put(k, new LinkedHashMap<>(v)));

        return this;
    }
}
