package ru.yandex.msearch.proxy.dispatcher;

import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;

import org.apache.http.HttpHost;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.httpclient.CommonHttpClient;
import ru.yandex.msearch.proxy.searchmap.SearchMap;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

import ru.yandex.search.prefix.Prefix;

public class DispatcherFactory
{
public static int LUCENE_JSON_PORT;
public static int LUCENE_SEARCH_PORT;
private static int pingInterval;
private static int getErrorThreshold;
private static double onErrorTimeKoef;
private static int avgDepth;
private static final String lucenePingPath = "/ping";
private static final String zoolooserPingPath =
    "/_status?allow_cached=true&prefix=100500&service=";

private static DispatcherFactory instance = null;
public static DispatcherFactory getInstance() {
    return instance;
}

    enum HostType {
        LUCENE {
            public String getPingUri(final String service) {
                return lucenePingPath;
            }
        },
        ZOOLOOSER {
            public String getPingUri(final String service) {
                return zoolooserPingPath + service;
            }
        };
        public abstract String getPingUri(final String service);
    };

    public static void init(
        final IniConfig config,
        final ImmutableMsearchProxyConfig proxyConfig)
        throws ConfigException
    {
        CommonHttpClient.init(config);
        pingInterval = config.getInt("ping_interval", 500);
        getErrorThreshold = config.getInt("get_error_threshold", 10);
        onErrorTimeKoef = config.getDouble("on_error_time_koef", 1d);
        avgDepth = config.getInt("avg_depth", 5);
        LUCENE_JSON_PORT = config.getInt("lucene.json.port", 8082);
        LUCENE_SEARCH_PORT = config.getInt("lucene.search.port", 8088);
        ZooLooserDispatcher.init(config.section("zoolooser"));
        PreciseHttpRequestDispatcher.init(config, proxyConfig);
        instance = new DispatcherFactory();
    }

    public static Dispatcher create( Prefix prefix, String db, SearchMap searchMap, boolean precise, HttpServer.RequestContext ctx ) throws DispatcherException
    {
        LinkedList<SearchMap.Host> hosts = searchMap.getHosts( db, prefix );
        if( hosts == null )
        {
            throw new DispatcherException.NotFoundException( "Dispatcher error: user <" + prefix + "> cannot be bound to any server." );
        }

        for( SearchMap.Host host : hosts )
        {
            if( host.params.get("zk") != null )
            {
                return new ZooLooserDispatcher( instance, prefix, db, searchMap, precise, ctx );
            }
        }
        return new SynchronousDispatcher( instance, prefix, db, searchMap, precise, ctx );
    }

    public DispatcherHttpHost createZooLooserHost(
        final HttpServer.RequestContext ctx,
        final HttpHost host,
        final String service)
    {
        return DispatcherHttpHost.create(
            ctx,
            instance,
            host,
            service,
            HostType.ZOOLOOSER);
    }

    public DispatcherHttpHost createZooLooserHost(
        final HttpServer.RequestContext ctx,
        final String address,
        final int port,
        final String service)
    {
        return DispatcherHttpHost.create(
            ctx,
            instance,
            address,
            port,
            service,
            HostType.ZOOLOOSER);
    }

    public DispatcherHttpHost createLuceneHost(
        final HttpServer.RequestContext ctx,
        final HttpHost host,
        final String service)
    {
        return DispatcherHttpHost.create(
            ctx,
            instance,
            host,
            service,
            HostType.LUCENE);
    }

    public DispatcherHttpHost createLuceneHost(
        final HttpServer.RequestContext ctx,
        final String address,
        final int port,
        final String service)
    {
        return DispatcherHttpHost.create(
            ctx,
            instance,
            address,
            port,
            service,
            HostType.LUCENE);
    }

    public static List<DispatcherHttpHost> sortHostsByAVGTime(
        final List<DispatcherHttpHost> hosts)
    {
        Collections.sort(
            hosts,
            new Comparator<DispatcherHttpHost>() {
                @Override
                public int compare(
                    final DispatcherHttpHost h1,
                    final DispatcherHttpHost h2)
                {
                    return Double.compare(
                        h1.getAccessTime(),
                        h2.getAccessTime());
                }
            }
        );
        return hosts;
    }

    protected int avgDepth()
    {
        return avgDepth;
    }

    protected double onErrorTimeKoef()
    {
        return onErrorTimeKoef;
    }

    protected int getErrorThreshold()
    {
        return getErrorThreshold;
    }
    
    protected int pingInterval()
    {
        return pingInterval;
    }
}

