package ru.yandex.msearch.proxy.dispatcher;

import java.io.IOException;
import java.util.LinkedList;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;


import ru.yandex.http.util.client.measurable.MeasurableHttpContext;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.httpclient.CommonHttpClient;
import ru.yandex.msearch.proxy.logger.Logger;

public class DispatcherHttpHost implements Runnable {
    private static final ConcurrentHashMap<DispatcherHttpHost, DispatcherHttpHost> hostsMap = new ConcurrentHashMap<>();
    private static final ScheduledThreadPoolExecutor pingScheduler =
        new ScheduledThreadPoolExecutor(1);
    private static final int DEFAULT_REQUEST_TIMEOUT = 30000;

    private final AtomicInteger errors = new AtomicInteger(0);
    private final AtomicLong lastRequestTime = new AtomicLong(0);
    private final LinkedList<Double> timesList = new LinkedList<>();

    private final HttpServer.RequestContext ctx;
    private final DispatcherFactory factory;
    private final HttpHost host;
    private final int avgDepth;
    private final String pingUri;
    private final BasicAsyncRequestProducerGenerator ping;
    private final FutureCallback<HttpResponse> pingCallback;
    private long pingStartTime;

    private volatile boolean unreachable = false;
    private double sum = 0d;

    private DispatcherHttpHost(
        final HttpServer.RequestContext ctx,
        final DispatcherFactory factory,
        final HttpHost host,
        final String service,
        final DispatcherFactory.HostType hostType)
    {
        this.ctx = ctx;
        this.factory = factory;
        this.host = host;
        avgDepth = factory.avgDepth();
        pingUri = hostType.getPingUri(service);
        ping = new BasicAsyncRequestProducerGenerator(pingUri);
        pingCallback = new FutureCallback<HttpResponse>() {
            @Override
            public void completed(final HttpResponse response) {
                final int code = response.getStatusLine().getStatusCode();
                if (code >= 400 || code < 200) {
                    Logger.err("DispatcherHttpHost<"
                        + host + ">: ping error. Invalid http response code: "
                        + code);
                    unreachable = true;
                    if (response.getEntity() != null) {
                        try {
                            final String reply =
                                EntityUtils.toString(response.getEntity());
                            if (reply.length() != 0) {
                                Logger.err("DispatcherHttpHost<"
                                    + host + ">: server reply: " + reply);
                            }
                        } catch (IOException ign) {
                        }
                    }
                } else {
                    final long pingTime = System.nanoTime() - pingStartTime;
                    double fTime = pingTime / 1000000d;
                    updateAccessTime(fTime);
                    errors.set(0);
                    if (unreachable) {
                        unreachable = false;
                        synchronized (timesList) { //Take over old values
                            double last = timesList.getLast();
                            timesList.clear();
                            sum = last;
                            timesList.add( last );
                        }
                    }
                }
                schedulePing();
            }

            @Override
            public void failed(final Exception e) {
                Logger.err("DispatcherHttpHost<" + host + ">: ping error: "
                    + e.getMessage());
                final long pingTime = System.nanoTime() - pingStartTime;
                double fTime = pingTime / 1000000d;
                fTime *= factory.onErrorTimeKoef();
                updateAccessTime(fTime);
                if(!(e instanceof
                    org.apache.http.conn.ConnectionPoolTimeoutException))
                {
                    if (errors.incrementAndGet() > factory.getErrorThreshold()) {
                        unreachable = true;
                    }
                }
                schedulePing();
            }

            @Override
            public void cancelled() {
            }
        };
    }

    public static DispatcherHttpHost create(
        final HttpServer.RequestContext ctx,
        final DispatcherFactory factory,
        final HttpHost host,
        final String service,
        final DispatcherFactory.HostType hostType)
    {
        return getHost(ctx, factory, host, service, hostType);
    }

    public static DispatcherHttpHost create(
        final HttpServer.RequestContext ctx,
        final DispatcherFactory factory,
        final String hostname,
        final int port,
        final String service,
        final DispatcherFactory.HostType hostType)
    {
        return getHost(ctx, factory, new HttpHost(hostname, port), service, hostType);
    }

    public HttpHost httpHost() {
        return host;
    }

    public void ping() {
        final AsyncClient client = ctx.server().searchClient();
        pingStartTime = System.nanoTime();
        client.execute(host, ping, pingCallback);
    }

    public void schedulePing() {
        pingScheduler.schedule(
            this,
            factory.pingInterval(),
            TimeUnit.MILLISECONDS);
    }

    @Override
    public void run() {
        ping();
    }

    public CloseableHttpResponse execute( CloseableHttpClient client, HttpRequestBase request, int timeout, HttpServer.RequestContext ctx ) throws IOException
    {
        if( unreachable ) throw new IOException( "HttpHost <" + host + "> is unreachable" );
        try
        {
            final int connTimeout = Math.max(timeout >> 3, 1000);
    	    RequestConfig config = RequestConfig.custom()
    	        		    .setSocketTimeout( timeout )
    		        	    .setConnectTimeout( timeout )
    			            .setConnectionRequestTimeout( timeout )
    			            .setExpectContinueEnabled( false )
    			            .setStaleConnectionCheckEnabled( false )
    			        .build();
            request.setConfig( config );

            MeasurableHttpContext context = new MeasurableHttpContext();
            CloseableHttpResponse response = client.execute( host, request, context );
            ctx.log.info(context.finAndGetInfo());
            return response;
        }
        catch( Exception e )
        {
            synchronized(this)
            {
                notify();
            }
            throw e;
        }
    }

    public CloseableHttpResponse timingExecute( CloseableHttpClient client, HttpRequestBase request, int timeout ) throws IOException
    {
        lastRequestTime.set( System.currentTimeMillis() );
        long time = System.nanoTime();
        try
        {
    	    RequestConfig config = RequestConfig.custom()
    				    .setSocketTimeout( timeout )
    				    .setConnectTimeout( timeout )
    				    .setConnectionRequestTimeout( timeout )
    				    .setExpectContinueEnabled( false )
    				    .setStaleConnectionCheckEnabled( false )
    				    .build();
    	    request.setConfig( config );

            CloseableHttpResponse response = client.execute( host, request );

            time = System.nanoTime() - time;
            double fTime = time / 1000000d;
            updateAccessTime(fTime);
            return response;
        }
        catch( Exception e )
        {
            double millis = (System.nanoTime() - time) / 1000000d;
            millis *= factory.onErrorTimeKoef();
            updateAccessTime( millis );
            if( !(e instanceof org.apache.http.conn.ConnectionPoolTimeoutException) )
            {
        	if( errors.incrementAndGet() > factory.getErrorThreshold() )
        	{
            	    unreachable = true;
        	}
    	    }
    	    throw( e );
        }
    }

    public CloseableHttpResponse execute( HttpRequestBase request, int timeout, HttpServer.RequestContext ctx ) throws IOException
    {
        return execute( CommonHttpClient.httpClient, request, timeout, ctx );
    }

    public String toString()
    {
        return host.toString() + ", pingUri=" + pingUri + ", AVGAccessTime=" + getAccessTime();
    }

    public String addressString()
    {
        return host.toString();
    }

    public int hashCode()
    {
        return host.hashCode() ^ pingUri.hashCode();
    }

    public boolean equals(final Object o) {
        DispatcherHttpHost other = (DispatcherHttpHost) o;
        return other.host.equals(host) && other.pingUri.equals(pingUri);
    }

    public void updateAccessTime(final double milliseconds) {
        synchronized (timesList) {
            sum += milliseconds;
            timesList.add(milliseconds);
            if (timesList.size() > avgDepth) {
                sum -= timesList.pollFirst();
            }
        }
    }

    public double getAccessTime() {
        synchronized (timesList) {
            return sum / timesList.size();
        }
    }

    private static DispatcherHttpHost getHost(
        final HttpServer.RequestContext ctx,
        final DispatcherFactory factory,
        final HttpHost host,
        final String service,
        final DispatcherFactory.HostType hostType)
    {
        DispatcherHttpHost newHost =
            new DispatcherHttpHost(ctx, factory, host, service, hostType);
        DispatcherHttpHost dispHost = hostsMap.get(newHost);
        if (dispHost == null) {
            dispHost = hostsMap.putIfAbsent(newHost, newHost);
            if (dispHost == null) {
                dispHost = newHost;
                newHost.ping();
            }
        }
        return dispHost;
    }
}
