package ru.yandex.msearch.proxy.dispatcher;

import java.io.IOException;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.util.EntityUtils;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.msearch.proxy.logger.Logger;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class PreciseHttpRequestDispatcher
{
private final HttpServer.RequestContext ctx;

private CloseableHttpResponse response;
private final Object responseLock = new Object();
private volatile int runningHosts = 0;
private volatile boolean failed;
private List<Exception> exceptions = null;

protected static ThreadPoolExecutor executor = null;
private static final Object initLock = new Object();

    public PreciseHttpRequestDispatcher( HttpServer.RequestContext ctx )
    {
        this.ctx = ctx;
    }

    public CloseableHttpResponse dispatch( List<DispatcherHttpHost> hosts, HttpRequestBase request, int timeout ) throws DispatcherException
    {
	if( timeout <= 0 ) throw new DispatcherException( "PreciseHttpRequestDispatcher.dispatch: can't satisfy requested timeout: " + timeout );
        long startTime = System.currentTimeMillis();

        response = null;
        failed = false;

        DispatcherHttpHost firstHost = hosts.remove(0);

        queryHost( firstHost, request, timeout );

        try
        {
            synchronized( responseLock )
            {
                if( response != null )
                {
                    return response;
                }
                responseLock.wait( timeout / 4 > 0 ? timeout / 4 : timeout );
                if( response != null )
                {
                    return response;
                }

                //We have past timeout / 2 time since first request start and has no response to be ready.
                if( runningHosts > 0 )
                {
                    //First host request is still running. This may be caused by a dropped TCP packet. We should try to restart request.
//                    hosts.add( firstHost ); TODO actualy this is overkill
                }

                if( hosts.size() > 0 )
                {
                    int timeLeft = timeout - (int)(System.currentTimeMillis() - startTime);
                    for( DispatcherHttpHost host : hosts )
                    {
                        queryHost( host, request, timeLeft );
                    }

                    while( (timeLeft = timeout - (int)(System.currentTimeMillis() - startTime)) > 0 )
                    {
                        responseLock.wait( timeLeft );
                        if( response != null ) return response;
                        if( runningHosts == 0 ) break;
                    }
                }

		failed = true;

                DispatcherException e;
                if( runningHosts == 0 )
                {
                    e = new DispatcherException( "PreciseHttpRequestDispatcher.dispatch: can't dispatch search request" );
                }
                else
                {
                    e = new DispatcherException( "PreciseHttpRequestDispatcher.dispatch: can't satisfy requested timeout" );
                }
                if( exceptions != null )
                {
                    for( Exception s : exceptions )
                    {
                        e.addSuppressed( s );
                    }
                }
                throw e;
            }
        }
        catch( InterruptedException e )
        {
            ctx.log.err( "PreciseHttpRequestDispatcher.dispatch: thread was interrupted: " + Logger.exception(e) );
            synchronized( responseLock )
            {
        	if( response != null ) return response;
    		failed = true;
    	    }
            throw new DispatcherException( e );
        }
        catch( Throwable eae )
        {
    	    ctx.log.err( "PreciseHttpRequestDispatcher.dispatch: unhandled exception: " + Logger.exception(eae) );
            synchronized( responseLock )
            {
        	if( response != null ) return response;
    		failed = true;
    	    }
    	    throw new DispatcherException( eae );
        }

    }

    private final void queryHost( DispatcherHttpHost host, HttpRequestBase request, int timeout ) throws DispatcherException
    {
        try
        {
            ctx.log.err( "PreciseHttpRequestDispatcher.queryHost: sending request " + request + " to host <" + host + ">" );
//            runningHosts.add( host );
	    runningHosts++;
            executor.submit( new QueryHostTask( host, request, timeout ) );
        }
        catch( Exception e )
        {
            ctx.log.err( "PreciseHttpRequestDispatcher.queryHost: <" + host + ">: exception: " + Logger.exception(e) );
            synchronized( responseLock )
            {
                if( exceptions == null )
                {
                    exceptions = new LinkedList<Exception>();
                }
                exceptions.add( e );
//                runningHosts.remove( host );
		runningHosts--;
                responseLock.notify();
            }
        }
    }

    public static void init(
        final IniConfig config,
        final ImmutableMsearchProxyConfig proxyConfig)
        throws ConfigException
    {
	int maxThreads =
	    config.getInt("precise_http_request_dispatcher_threads", -1);
	if (maxThreads <= 0) {
	    maxThreads = proxyConfig.syncServerThreads() * 2;
	}
        executor = new ThreadPoolExecutor(
            1,
            maxThreads,
            30,
            TimeUnit.SECONDS,
            new SynchronousQueue<Runnable>());
        executor.setRejectedExecutionHandler(
            new ThreadPoolExecutor.AbortPolicy());
    }

    private class QueryHostTask implements Callable<Void>
    {
    private final DispatcherHttpHost host;
    private final HttpRequestBase request;
    private final int timeout;

        public QueryHostTask( DispatcherHttpHost host, HttpRequestBase request, int timeout )
        {
            this.host = host;
            this.request = request;
            this.timeout = timeout;
        }

        public Void call()
        {
	    CloseableHttpResponse response = null;
            try
            {
                response = host.execute( request, timeout, ctx );
                int responseCode = response.getStatusLine().getStatusCode();
	        if( responseCode < 200 || responseCode >= 500 )
	        {
	            String responseBody = "empty response body";
		    if( response.getEntity() != null )
		    {
		        responseBody = EntityUtils.toString( response.getEntity() );
		    }
		    response.close();
		    ctx.log.err( "PreciseHttpRequestDispatcher.queryHost: <" + host.addressString() + ">: bad response: code = " + responseCode + ", body = " + responseBody );
	            throw new DispatcherException.BadResponseException( responseCode, host.addressString(), responseBody );
	        }
	        synchronized( responseLock )
	        {
	            if( PreciseHttpRequestDispatcher.this.response != null || failed )
	            {
	        	response.close();
	                //we are too late.
//	                EntityUtils.consumeQuietly( response.getEntity() );
	            }
	            else
	            {
	                PreciseHttpRequestDispatcher.this.response = response;
	            }
//                    runningHosts.remove( host );
		    runningHosts--;
	            responseLock.notify();
	        }
            }
            catch( Exception e )
            {
		ctx.log.err( "PreciseHttpRequestDispatcher.queryHost: <" + host.addressString() + ">: exception: " + Logger.exception(e) );
		try
		{
		    if( response != null ) response.close();
		}
		catch( IOException e2 )
		{
		    ctx.log.err( "PreciseHttpRequestDispatcher.queryHost: <" + host.addressString() + ">: response close exception: " + Logger.exception(e2) );
		}
                synchronized( responseLock )
                {
                    if( exceptions == null )
                    {
                        exceptions = new LinkedList<Exception>();
                    }
                    exceptions.add( e );
		    runningHosts--;
//                    runningHosts.remove( host );
                    responseLock.notify();
                }
            }
            return null;
        }

    }
}
