package ru.yandex.msearch.proxy.dispatcher;

import java.io.IOException;

import java.net.URI;
import java.net.URISyntaxException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.util.EntityUtils;

import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.searchmap.SearchMap;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

import ru.yandex.search.prefix.Prefix;

public class ZooLooserDispatcher extends AbstractDispatcher
{
private static Object initLock = new Object();
private static int ZOOLOOSER_PORT;
private static int ZOOLOOSER_STATUS_PORT;
private static int ZOOLOOSER_INDEX_TIMEOUT;

    public static void init(final IniConfig config) throws ConfigException {
        ZOOLOOSER_PORT = config.getInt("port", 889);
        ZOOLOOSER_STATUS_PORT = config.getInt("status_port", 890);
        ZOOLOOSER_INDEX_TIMEOUT = config.getInt("index_timeout", 30000);
    }

    public ZooLooserDispatcher( DispatcherFactory factory, Prefix prefix, String db, SearchMap searchMap, boolean precise, HttpServer.RequestContext ctx )
    {
        super(factory, prefix, db, searchMap, precise, ctx);
    }

    public static final String convertAddress( String searchMapAddress )
    {
        String address = searchMapAddress.split(":")[0];
        return address + ":" + ZOOLOOSER_PORT;
    }

    @Override
    public List<DispatcherHttpHost> getSearchHosts() throws DispatcherException
    {
	return getSearchHosts( 30000 );
    }

    public List<DispatcherHttpHost> getSearchHosts( int timeout ) throws DispatcherException
    {
        LinkedList<SearchMap.Host> hosts = searchMap.getHosts( db, prefix );
        String zk = null;

        HashMap<String, Integer> searchPortsMap = new HashMap<>();
        for( SearchMap.Host host : hosts )
        {
            final String hostZk = host.params.get("zk");
            if (hostZk != null) {
                searchPortsMap.put(hostName(host), newSearchPort(host));
                zk = hostZk;
            }
        }

        ArrayList<DispatcherHttpHost> zooKeepers = new ArrayList<DispatcherHttpHost>( 3 );
        for( String zooKeeper : zk.split("\\|") )
        {
            zooKeepers.add( factory.createZooLooserHost(ctx, zooKeeper.split(":")[0], ZOOLOOSER_STATUS_PORT, db) );
        }
        factory.sortHostsByAVGTime( zooKeepers );

	long time = System.currentTimeMillis();

        DispatcherAndReader dispatcher;
        if( precise )
        {
	    dispatcher = new PreciseHttpRequestDispatcherAndReader( ctx );
	}
	else
	{
	    dispatcher = new SequentialHttpRequestDispatcherAndReader( ctx );
	}
        String content = dispatcher.dispatch( zooKeepers, new HttpGet( "/_status?allow_cached=true&prefix=" + prefix.hash() + "&service=" + db ), timeout);

        String[] lines = content.split("\n");
        List<DispatcherHttpHost> ret = null;
        for( String l : lines )
        {
            if( ret == null ) ret = new ArrayList<DispatcherHttpHost>();
            int semicolon = l.indexOf(':');
            final String hostName;
            if (semicolon != -1){
                hostName = l.substring(0,semicolon + 1);
            } else {
                hostName = l;
            }
            Integer port = searchPortsMap.get(hostName);
            if (port == null) {
                port = DispatcherFactory.LUCENE_SEARCH_PORT;
            }
            ret.add( factory.createLuceneHost(ctx, hostName, port, db) );
        }

        factory.sortHostsByAVGTime( ret );

        ctx.log.info( "ZooLooserDispatcher.getHosts: hostlist get time: " + (System.currentTimeMillis() - time) );
        for( DispatcherHttpHost host : ret )
        {
            ctx.log.info( "ZooLooserDispatcher.getHosts.hostList: " + host );
        }
        return ret;
    }

    @Override
    public CloseableHttpResponse searchDispatch( HttpRequestBase request, int timeout ) throws DispatcherException
    {
        long start = System.currentTimeMillis();
	List<DispatcherHttpHost> hosts = getSearchHosts( timeout );

        request = addSessionId(request);
        if( precise )
        {
            PreciseHttpRequestDispatcher httpDispatcher = new PreciseHttpRequestDispatcher( ctx );
	    return httpDispatcher.dispatch( hosts, request, timeout - (int)(System.currentTimeMillis() - start) );
	}
	else
	{
            SequentialHttpRequestDispatcher httpDispatcher = new SequentialHttpRequestDispatcher( ctx );
	    return httpDispatcher.dispatch( hosts, request, timeout - (int)(System.currentTimeMillis() - start) );
	}
    }

    @Override
    public void indexDispatch( HttpRequestBase request, int timeout ) throws DispatcherException
    {
        LinkedList<SearchMap.Host> hosts = searchMap.getHosts( db, prefix );
        String zk = null;

        for( SearchMap.Host host : hosts )
        {
            zk = host.params.get("zk");
            if( zk != null ) break;
        }

        StringBuilder error = new StringBuilder();

        ArrayList<DispatcherHttpHost> zooKeepers = new ArrayList<DispatcherHttpHost>();
        for( String zooKeeper : zk.split("\\|") )
        {
            String address = zooKeeper.split(":")[0];
            zooKeepers.add( factory.createZooLooserHost(ctx, address, ZOOLOOSER_PORT, db) );
        }
        factory.sortHostsByAVGTime( zooKeepers );

        request = addUserInfo(request);
    	for( DispatcherHttpHost zooKeeper : zooKeepers )
    	{
            long startTime = System.currentTimeMillis();
            CloseableHttpResponse response = null;
    	    try
    	    {
	        ctx.log.info( "ZooLooserDispatcher.dispatch: trying zooKeeper: " + zooKeeper );
	        response = zooKeeper.execute( request, timeout, ctx );
	        HttpEntity httpEntity = response.getEntity();
	        int responseCode = response.getStatusLine().getStatusCode();
	        if( httpEntity == null )
	        {
	            ctx.log.err( "Empty response with code <" + responseCode + "> from: " + zooKeeper.addressString() );
	            error.append( "Empty response with code <" + responseCode + "> from: " + zooKeeper.addressString() + "\n" );
	            continue;
	        }
	        else
	        {
	            String content = EntityUtils.toString( httpEntity );
	            if( responseCode >= 200 && responseCode < 400 )
	            {
	                ctx.log.debug( "ZooLooserDispatcher.dispatch: success" );
	                return;
	            }
	            else if( responseCode != HttpStatus.SC_GATEWAY_TIMEOUT && responseCode != HttpStatus.SC_SERVICE_UNAVAILABLE )
	            {
	                throw new DispatcherException.BadResponseException( responseCode, zooKeeper.addressString(), content );
	            }
	            else
	            {
	                ctx.log.err( "Invalid response with code <" + responseCode + "> from: " + zooKeeper.addressString() + ": " + content );
	                error.append( "Invalid response with code <" + responseCode + "> from: " + zooKeeper.addressString() + ": " + content + "\n" );
	                continue;
	            }
	        }
	    }
	    catch( DispatcherException.BadResponseException e )
	    {
	        throw e;
	    }
	    catch( IOException e )
	    {
	        error.append( "IOException while handling address <" + zooKeeper.addressString() + ">" );
	        error.append( Logger.exception(e) );
	        ctx.log.err( "IOException while handling address <" + zooKeeper.addressString() + ">: " + Logger.exception(e) );
	    }
	    finally
	    {
	        long execTime = System.currentTimeMillis() - startTime;
	        ctx.log.info( "ZooLooserDispatcher.dispatch: <" + zooKeeper.addressString() + "> execution time: " + execTime );
	        try
	        {
	    	    if( response != null ) response.close();
	    	}
	    	catch( IOException e )
	    	{
	    	    ctx.log.info( "ZooLooserDispatcher.dispatch: <" + zooKeeper.addressString() + "> http response close exception: " + Logger.exception(e) );
	    	}
	    }
        }
        if( error.length() > 0 )
        {
            throw new DispatcherException( error.toString() );
        }
        else
        {
            ctx.log.debug( "Noerror" );
        }
    }

}
