package ru.yandex.msearch.proxy.document;

import java.io.IOException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.InputStreamBody;

import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.NByteArrayEntityAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;

public class DiskDocumentHandler implements ProxyRequestHandler {
    private final HttpProxy proxy;
    private final AsyncClient client;
    private final AsyncClient mulcaClient;
    private final ImmutableHttpHostConfig diskConfig;
    private final ImmutableHttpHostConfig mulcaConfig;

    public DiskDocumentHandler(
        final HttpProxy proxy,
        final ImmutableMsearchProxyConfig config) throws ConfigException {
        this.proxy = proxy;

        client = proxy.client("DiskDocumentSearch", config.multisearchConfig().disk());
        diskConfig = config.multisearchConfig().disk();
        mulcaConfig =
            new HttpHostConfigBuilder()
                .host(new HttpHost("storage.mail.yandex.net", 10010))
                .connections(5).timeout(60000).build();
        mulcaClient = proxy.client("Mulca", mulcaConfig);
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        Long userId = session.params().getLong("uid");
        String path = session.params().getString("path");
        QueryConstructor qc = new QueryConstructor("/sequential/search?");
        qc.append("prefix", userId);
        qc.append("service", "disk_queue");
        qc.append("source", "documents_proxy");
        qc.append("text", "mediatype:9 AND key:" + SearchRequestText.fullEscape(path, false));
        qc.append("get", "id,stid,mimetype,preview_stid,name");

        BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(qc.toString());
        //SaveCallback saveCallback = new SaveCallback(context.session());
        AsyncClient client = this.client.adjust(session.context());
        session.logger().info("Request " + diskConfig.host() + " " + qc.toString());
        client.execute(
            diskConfig.host(),
            generator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            new DiskCallback(session, userId, path));
    }

    private class DiskCallback extends AbstractProxySessionCallback<JsonObject> {
        private final ProxySession session;
        private final String path;
        private final Long uid;
        public DiskCallback(
            final ProxySession session,
            final Long uid,
            final String path)
        {
            super(session);

            this.session = session;
            this.path = path;
            this.uid = uid;
        }

        @Override
        public void completed(final JsonObject jsonObj) {
            //ProxySession session = context.session();
            BasicAsyncRequestProducerGenerator generator;
            String mimetype;
            String name;
            try {
                JsonMap map = jsonObj.asMap();
                JsonList list = map.getList("hitsArray");
                if (list.size() != 1) {
                    failed(new NotFoundException("Document not found"));
                    return;
                }
                map = list.get(0).asMap();
                String stid = map.getString("stid");
                name = map.getString("name");
                mimetype = map.getString("mimetype");
                generator = new BasicAsyncRequestProducerGenerator("/gate/get/" + stid);
            } catch (JsonException je) {
                failed(je);
                return;
            }

            AsyncClient client = mulcaClient.adjust(session.context());
            client.execute(
                mulcaConfig.host(),
                generator,
                NByteArrayEntityAsyncConsumerFactory.OK,
                new DownloadCallback(session, mimetype, name));
        }
    }

    private class DownloadCallback extends AbstractProxySessionCallback<HttpEntity> {
        private final String contentType;
        private final String name;

        public DownloadCallback(final ProxySession session, final String contentType, final String name) {
            super(session);
            this.contentType = contentType;
            this.name = name;
        }

        @Override
        public void completed(final HttpEntity entity) {
            session.getResponse().addHeader("name", name);
            session.getResponse().addHeader("source-content-type", contentType);
            session.response(HttpStatus.SC_OK, entity);
        }
    }
}
