package ru.yandex.msearch.proxy.document;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;

public class Document implements Comparable<Document> {
    public static final String SERVER_RELEVANCE_KEY = "_relevant_";

    private static final Set<String> INT_FIELDS =
        new HashSet<>(
            Arrays.asList(
                SERVER_RELEVANCE_KEY,
                "received_date",
                "mid",
                "attachsize_b"));
    private static final AtomicLong UNIQUE_ID = new AtomicLong();

    protected final Set<String> intFields;
    protected Map<String,String> attrs;
    protected String orderField;
    protected String groupField;
    protected long orderFieldInt;
    protected long groupFieldInt;
    protected boolean intCompare = false;
    protected boolean intGroup = false;
    private List<Document> merged = null;
    private volatile long uniqueId = 0L;

    public Document()
    {
        this(INT_FIELDS);
    }

    public Document( Set<String> intFields )
    {
        this(intFields, new HashMap<String, String>());
    }

    public Document( Map<String,String> attrs )
    {
        this(INT_FIELDS, attrs);
    }

    public Document( Set<String> intFields, Map<String,String> attrs )
    {
        this.intFields = intFields;
        this.attrs = attrs;
    }

    public void setAttr( String key, String value )
    {
	attrs.put(key, value);
    }

    public void setAttrs( Document other )
    {
	attrs.putAll(other.attrs);
    }

    public String getAttr( String key )
    {
	return attrs.get(key);
    }

    public Map<String,String> getAttrs()
    {
	return attrs;
    }

    public void setOrderField( String field )
    {
	orderField = attrs.get(field);
	if( intFields.contains(field) )
	{
	    try
	    {
		orderFieldInt = Long.parseLong(orderField);
		intCompare = true;
	    }
	    catch( Exception e )
	    {
	    }
	}
	else
	{
	    intCompare = false;
	}
    }

    public void setGroupField( String field )
    {
	groupField = attrs.get(field);
	if( intFields.contains(field) )
	{
	    try
	    {
		groupFieldInt = Long.parseLong(groupField);
		intGroup = true;
	    }
	    catch( Exception e )
	    {
	    }
	}
	else
	{
	    intGroup = false;
	}
    }

    public long uniqueId() {
        if (uniqueId == 0L) {
            synchronized (this) {
                if (uniqueId == 0L) {
                    uniqueId = UNIQUE_ID.incrementAndGet();
                }
            }
        }
        return uniqueId;
    }

    @Override
    public int hashCode()
    {
        if( groupField == null ) return 0;
        return groupField.hashCode();
    }

    @Override
    public boolean equals( Object o )
    {
	Document other = (Document)o;
	if( groupField == null ) return false;
	if( other.groupField == null ) return false;
	if( intGroup && other.intGroup ) {
            return groupFieldInt == other.groupFieldInt;
        }
	return groupField.equals(other.groupField);
    }

    protected int doCompareTo(final Document other) {
	if (orderField == null) {
            if (other.orderField == null) {
                return 0;
            } else {
                return -1;
            }
        } else if (other.orderField == null) {
            return 1;
        } else if (intCompare && other.intCompare) {
            return Long.compare(orderFieldInt, other.orderFieldInt);
        } else {
            return orderField.compareTo(other.orderField);
        }
    }

    @Override
    public int compareTo(final Document other) {
        int cmp = doCompareTo(other);
        if (cmp == 0) {
            cmp = Long.compare(uniqueId(), other.uniqueId());
        }
        return cmp;
    }

    public void addMerged( Document other )
    {
	if( merged == null ) merged = new LinkedList<Document>();
	merged.add( other );
    }

    public int getMergedCount()
    {
	if( merged == null ) return 0;
	return merged.size();
    }

    public Document getMerged( int i )
    {
	if( merged == null ) return null;
	return merged.get(i);
    }

    public List<Document> getMerged()
    {
	return merged;
    }

    public void setMerged( List<Document> merged )
    {
	this.merged = merged;
    }

    public String toString()
    {
	return this.toString( "" );
    }

    public String toString( String prefix )
    {
	StringBuilder sb = new StringBuilder( 50 );
	sb.append( prefix );
	sb.append( "<document>\n" );
	for( Map.Entry<String,String> entry : attrs.entrySet() )
	{
	    sb.append( prefix );
	    sb.append( "\t<" );
	    sb.append( entry.getKey() );
	    sb.append( ">" );
	    sb.append( '=' );
	    sb.append( entry.getValue() );
	    sb.append( '\n' );
	}
	if( merged != null )
	{
	    sb.append( prefix );
	    sb.append( "\t<merged_docs>\n" );
	    for( Document sub : merged )
	    {
		sb.append( sub.toString( prefix + "\t\t" ) );
	    }
	    sb.append( prefix );
	    sb.append( "\t</merged_docs>\n" );
	}
	sb.append( prefix );
	sb.append( "</document>\n" );
	return sb.toString();
    }

    public Document clone( boolean deep )
    {
        Document doc = new Document();
        doc.attrs.putAll( attrs );
        doc.groupField = groupField;
        doc.orderField = orderField;
        doc.orderFieldInt = orderFieldInt;
        doc.groupFieldInt = groupFieldInt;
        doc.intCompare = intCompare;
        doc.intGroup = intGroup;
        if( merged == null ) return doc;
        if( deep )
        {
            for( Document m : merged )
            {
                doc.addMerged( m.clone(deep) );
            }
        }
        doc.setMerged( new LinkedList<Document>(merged) );
        return doc;
    }
}

