package ru.yandex.msearch.proxy.highlight;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import org.apache.commons.lang3.StringEscapeUtils;

import org.apache.commons.lang3.text.translate.CharSequenceTranslator;

import ru.yandex.collection.IntInterval;

import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;

import ru.yandex.util.string.StringUtils;

public enum HtmlHighlighter implements Highlighter {
    INSTANCE;

    public static final CharSequenceTranslator HTML_TRANSLATOR
        = StringEscapeUtils.ESCAPE_HTML4;

    private static final String HL_PREFIX = "<span class=\"msearch-highlight\">";
    private static final String HL_POSTFIX = "</span>";

    private static final int HL_LENGTH =
        HL_PREFIX.length() + HL_POSTFIX.length();

    public void highlightString(
        final StringBuilder sb,
        final String s,
        final int start,
        final int end)
    {
        sb.append(HL_PREFIX);
        sb.append(HTML_TRANSLATOR.translate(s.substring(start, end)));
        sb.append(HL_POSTFIX);
    }

    public String highlightString(
        final String s,
        final int start,
        final int end)
    {
        if (end <= start || start < 0 || end > s.length()) {
            return HTML_TRANSLATOR.translate(s);
        }

        StringBuilder sb  = new StringBuilder(s.length() + HL_LENGTH);
        sb.append(HTML_TRANSLATOR.translate(s.substring(0, start)));
        highlightString(sb, s, start, end);
        sb.append(HTML_TRANSLATOR.translate(s.substring(end, s.length())));
        return sb.toString();
    }

    public String highlight(
        final String request,
        final String suggest)
    {
        return highlight(Collections.singletonList(request), suggest, true);
    }

    public String highlightOrNull(
        final String s,
        final String preparedRequest,
        final boolean morpho)
    {
        if (s == null || s.isEmpty()) {
            return null;
        }

        return highlightOrNull(s, prepare(s), preparedRequest, morpho);
    }

    /**
     * @param s               - not null string, which we will try to highlight
     * @param preparedS       - s lowered and prefixed with space
     * @param preparedRequest - not null lowered request prefixed with space
     * @param morpho          - use morpho
     * @return highlighted string or null
     */
    public String highlightOrNull(
        final String s,
        final String preparedS,
        final String preparedRequest,
        final boolean morpho)
    {
        int index = preparedS.indexOf(preparedRequest);
        if (index != -1) {
            return highlightString(
                s,
                index,
                index + preparedRequest.length() - 1);
        }

        if (morpho) {
            if (preparedRequest.length() < MORPHO_THRSH) {
                return null;
            }

            for (int i = 1; i < MORPHO_REDUCE + 1; i++) {
                index = indexOf(
                    preparedS,
                    0,
                    preparedRequest,
                    0,
                    preparedRequest.length() - 1 - i);
                if (index != -1) {
                    return highlightString(
                        s,
                        index,
                        index + preparedRequest.length() - 1);
                }
            }
        }

        return null;
    }

    public String highlight(
        final Collection<String> requests,
        final String s,
        final boolean morpho)
    {
        if (s == null || s.isEmpty()) {
            return s;
        }

        String preparedString = prepare(s);

        Set<String> preparedRequests = new LinkedHashSet<>(requests.size());

        for (String request: requests) {
            if (request == null) {
                continue;
            }

            String preparedRequest =
                StringUtils.concat(
                    ' ',
                    request.trim().toLowerCase(Locale.ROOT));

            if (preparedRequest.isEmpty()) {
                continue;
            }

            String hl =
                highlightOrNull(s, preparedString, preparedRequest, false);
            if (hl != null) {
                return hl;
            }

            preparedRequests.add(preparedRequest);
        }

        // ok fail, trying morpho
        if (morpho) {
            for (String request : preparedRequests) {
                String hl =
                    highlightOrNull(s, preparedString, request, true);
                if (hl != null) {
                    return hl;
                }
            }
        }

        return HTML_TRANSLATOR.translate(s);
    }


    @Override
    public JsonObject highlight(
        final String s, final int start, final int end)
    {
        return new JsonString(highlightString(s, start, end));
    }

    @Override
    public JsonObject highlight(
        final String s,
        final List<String> preparedRequestWords,
        final boolean startWord,
        final boolean morpho)
    {
        IntervalRequestMatchesConsumer intervals =
            new IntervalRequestMatchesConsumer();
            RequestMatcher.INSTANCE.match(
                s,
                preparedRequestWords,
                intervals,
                startWord,
                morpho);

        try {
            if (intervals.size() == 0) {
                return null;
            } else {
                StringBuilder sb =
                    new StringBuilder(
                        HTML_TRANSLATOR.translate(
                            s.substring(0, intervals.get(0).min())));
                for (int i = 0; i < intervals.size(); i++) {
                    if (i > 0) {
                        sb.append(
                            HTML_TRANSLATOR.translate(
                                s.substring(
                                    intervals.get(i-1).max(),
                                    intervals.get(i).min())));
                    }
                    sb.append(HL_PREFIX);
                    sb.append(
                        HTML_TRANSLATOR.translate(
                            s.substring(
                                intervals.get(i).min(),
                                intervals.get(i).max())));
                    sb.append(HL_POSTFIX);
                }

                sb.append(
                    HTML_TRANSLATOR.translate(
                        s.substring(
                            intervals.get(intervals.size() - 1).max(),
                            s.length())));
                return new JsonString(sb.toString());
            }
        } catch (Exception e) {
            return new JsonString("Failed to highlight:" + s);
        }
    }
}
