package ru.yandex.msearch.proxy.highlight;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.collection.IntInterval;

import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;

import ru.yandex.json.dom.JsonObject;

public enum IntervalHighlighter implements Highlighter {
    INTANCE;

    public List<IntInterval> highlightIntervals(
        final String s,
        final List<String> preparedRequestWords,
        final boolean morpho)
    {
        return highlightIntervals(s, preparedRequestWords, true, morpho);
    }

    public List<IntInterval> highlightIntervals(
        final String s,
        final List<String> preparedRequestWords,
        final boolean wordsStart,
        final boolean morpho)
    {
        if (s == null || s.isEmpty()) {
            return Collections.emptyList();
        }

        LinkedList<IntInterval> intervals = new LinkedList<>();
        String preparedS = s.toLowerCase(Locale.ROOT);
        int lastIndex = 0;
        for (String rWord: preparedRequestWords) {
            IntInterval interval;
            if (wordsStart) {
                interval =
                    highlightOrNullInterval(preparedS, rWord, lastIndex, morpho);
            } else {
                interval =
                    highlightOrNullFull(preparedS, rWord, lastIndex, morpho);
            }

            if (interval != null) {
                intervals.add(interval);
                lastIndex = interval.max();
            } else if (lastIndex > 0) {
                if (wordsStart) {
                    interval =
                        highlightOrNullInterval(preparedS, rWord, 0, morpho);
                } else {
                    interval =
                        highlightOrNullFull(preparedS, rWord, 0, morpho);
                }

                if (interval != null) {
                    // we started search from beginning,
                    // so interval could be appeared somewhere in middle
                    IntInterval item = null;
                    ListIterator<IntInterval> intervalIt
                        = intervals.listIterator();
                    while (intervalIt.hasNext()) {
                        item = intervals.get(intervalIt.nextIndex());
                        if (item.min() >= interval.max()) {
                            break;
                        }

                        intervalIt.next();
                        if (interval.min() >= item.max()) {
                            continue;
                        }

                        IntInterval merged = interval.mergeIfOverlaps(item);
                        if (merged != null) {
                            intervalIt.remove();
                            interval = merged;
                            break;
                        }
                    }

                    while (intervalIt.hasNext()) {
                        IntInterval next =
                            intervals.get(intervalIt.nextIndex());
                        if (next.min() > interval.max()) {
                            intervalIt.add(interval);
                            interval = null;
                            break;
                        }

                        IntInterval merged = interval.mergeIfOverlaps(next);
                        if (merged == null) {
                            // very strange
                            intervalIt.add(interval);
                            interval = null;
                            break;
                        } else {
                            intervalIt.next();
                            intervalIt.remove();
                            interval = merged;
                        }
                    }

                    if (interval != null) {
                        intervals.add(interval);
                    }
                }
            }
        }

        return intervals;
    }

    /**
     *  Highlights without any restrictions, good for short texts, like
     *  email adresses and so on
     * @param lowS - text where to find
     * @param preparedRequestWord - word to find
     * @param startIndex - start index in lowS
     * @param morpho - use morpho
     * @return
     */
    public IntInterval highlightOrNullFull(
        final String lowS,
        final String preparedRequestWord,
        final int startIndex,
        boolean morpho)
    {
        int index = lowS.indexOf(preparedRequestWord, startIndex);
        if (index >= 0) {
            return new IntInterval(index, index + preparedRequestWord.length());
        }

        if (morpho && preparedRequestWord.length() >= MORPHO_THRSH) {
            return highlightUsingMorpho(lowS, preparedRequestWord, startIndex);
        }

        return null;
    }

    protected IntInterval highlightUsingMorpho(
        final String lowS,
        final String preparedRequestWord,
        final int startIndex)
    {
        String requestWord = ' ' + preparedRequestWord;
        for (int i = 1; i < MORPHO_REDUCE + 1; i++) {
            char c =
                requestWord.charAt(requestWord.length() - i);
            if (!Character.isLetter(c)) {
                continue;
            }

            int index = indexOf(
                lowS,
                startIndex,
                requestWord,
                0,
                requestWord.length() - 1 - i);
            if (index > 0) {
                int start = index + requestWord.length() - 1 - i;
                for (int j = start;
                     j < Math.min(start + MORPHO_THRSH, lowS.length());
                     j++)
                {
                    c = lowS.charAt(j);
                    if (Character.isISOControl(c)
                        || Character.isSpaceChar(c)
                        || Character.isWhitespace(c))
                    {
                        return new IntInterval(index + 1, j);
                    }
                }

                return new IntInterval(
                    index + 1,
                    index + requestWord.length());
            }
        }

        return null;
    }

    public IntInterval highlightOrNullInterval(
        final String lowS,
        final String preparedRequestWord,
        final int startIndex,
        boolean morpho)
    {
        StringBuilder pb = new StringBuilder("\\b[^\\w]*(");
        pb.append(Pattern.quote(preparedRequestWord));
        pb.append(')');
        if (preparedRequestWord.length() <= Highlighter.WORD_MATCH_THRESHOLD) {
            pb.append("\\b");
        }

        Matcher matcher =
            Pattern.compile(
                pb.toString(),
                Pattern.UNICODE_CHARACTER_CLASS)
                .matcher(lowS);
        //int index = lowS.indexOf(preparedRequestWord, startIndex);

        if (matcher.find(startIndex)) {
            return new IntInterval(
                matcher.start(1),
                matcher.start(1) + preparedRequestWord.length());
        }

        if (morpho && preparedRequestWord.length() >= MORPHO_THRSH) {
            return highlightUsingMorpho(lowS, preparedRequestWord, startIndex);
        }

        return null;
    }

    @Override
    public JsonObject highlight(
        final String s, final int start, final int end)
    {
        JsonList interval = new JsonList(BasicContainerFactory.INSTANCE);
        interval.add(new JsonLong(start));
        interval.add(new JsonLong(end));
        return interval;
    }

    @Override
    public JsonObject highlight(
        final String s,
        final List<String> preparedRequestWords,
        final boolean wordsStart,
        final boolean morpho)
    {
        JsonList intervals = new JsonList(BasicContainerFactory.INSTANCE);
        for (IntInterval interval
            : highlightIntervals(s, preparedRequestWords, wordsStart, morpho))
        {
            JsonList jsonInterval =
                new JsonList(BasicContainerFactory.INSTANCE);
            jsonInterval.add(new JsonLong(interval.min()));
            jsonInterval.add(new JsonLong(interval.max()));
            intervals.add(jsonInterval);
        }

        return intervals;
    }
}
