package ru.yandex.msearch.proxy.highlight;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.collection.IntInterval;
import ru.yandex.util.string.StringUtils;


public enum RequestMatcher {
    INSTANCE;

    int MORPHO_THRSH = 6;
    int MORPHO_REDUCE= 2;
    // maximal length of word, to match it vs exact word in text
    // instead of matching with start of the word in text
    int WORD_MATCH_THRESHOLD= 2;

    public boolean match(
        final String s,
        final String request,
        final RequestMatchesConsumer consumer)
    {
        return match(s, request, consumer, true);
    }

    public boolean match(
        final String s,
        final List<String> requestWords,
        final RequestMatchesConsumer consumer)
    {
        return match(s, requestWords, consumer, true, true);
    }

    public boolean match(
        final String s,
        final String request,
        final RequestMatchesConsumer consumer,
        final boolean morpho)
    {
        return match(s, prepareRequest(request), consumer, morpho, true);
    }

    public List<IntInterval> match(
        final String s,
        final List<String> words,
        final boolean morpho)
    {
        return match(s, words, true, morpho);
    }

    public List<IntInterval> match(
        final String s,
        final List<String> words,
        final boolean wordStart,
        final boolean morpho)
    {
        IntervalRequestMatchesConsumer consumer =
            new IntervalRequestMatchesConsumer();
        match(s, words, consumer, wordStart, morpho);
        return consumer;
    }


    /**
     *  Highlights without any restrictions, good for short texts, like
     *  email adresses and so on
     * @param lowS - text where to find
     * @param preparedRequestWord - word to find
     * @param startIndex - start index in lowS
     * @param morpho - use morpho
     * @return
     */
    public int matchEntire(
        final String lowS,
        final String preparedRequestWord,
        final RequestMatchesConsumer consumer,
        final int startIndex,
        boolean morpho)
    {
        int index = lowS.indexOf(preparedRequestWord, startIndex);
        if (index >= 0) {
            int max = index + preparedRequestWord.length();
            consumer.apply(index, max);
            return max;
        }

        if (morpho && preparedRequestWord.length() >= MORPHO_THRSH) {
            return matchWordWithMorpho(
                consumer,
                lowS,
                preparedRequestWord,
                startIndex);
        }

        return -1;
    }

    public boolean match(
        final String s,
        final List<String> preparedRequestWords,
        final RequestMatchesConsumer consumer,
        final boolean wordsStart,
        final boolean morpho)
    {
        if (s == null || s.isEmpty()) {
            return false;
        }

        String preparedS = s.toLowerCase(Locale.ROOT);
        int lastIndex = 0;
        for (String rWord: preparedRequestWords) {
            int index;
            if (wordsStart) {
                index =
                    matchWord(preparedS, rWord, consumer, lastIndex, morpho);
            } else {
                index =
                    matchEntire(preparedS, rWord, consumer, lastIndex, morpho);
            }

            if (index >= 0) {
                lastIndex = index;
            } else if (lastIndex > 0) {
                if (wordsStart) {
                    index =
                        matchWord(preparedS, rWord, consumer, 0, morpho);
                } else {
                    index =
                        matchEntire(preparedS, rWord, consumer, 0, morpho);
                }

                if (index >= 0) {
                    lastIndex = index;
                }
            }
        }

        return lastIndex > 0;
    }

    public int matchWord(
        final String lowS,
        final String preparedRequestWord,
        final RequestMatchesConsumer consumer,
        final int startIndex,
        boolean morpho)
    {
        StringBuilder pb = new StringBuilder("\\b[^\\w]*(");
        pb.append(Pattern.quote(preparedRequestWord));
        pb.append(')');
        if (preparedRequestWord.length() <= Highlighter.WORD_MATCH_THRESHOLD) {
            pb.append("\\b");
        }

        Matcher matcher =
            Pattern.compile(
                pb.toString(),
                Pattern.UNICODE_CHARACTER_CLASS)
                .matcher(lowS);
        //int index = lowS.indexOf(preparedRequestWord, startIndex);

        if (matcher.find(startIndex)) {
            int max = matcher.start(1) + preparedRequestWord.length();
            consumer.apply(matcher.start(1), max);
            return max;
        }

        if (morpho && preparedRequestWord.length() >= MORPHO_THRSH) {
            return matchWordWithMorpho(
                consumer,
                lowS,
                preparedRequestWord,
                startIndex);
        }

        return -1;
    }

    protected int matchWordWithMorpho(
        final RequestMatchesConsumer consumer,
        final String lowS,
        final String preparedRequestWord,
        final int startIndex)
    {
        String requestWord = ' ' + preparedRequestWord;
        for (int i = 1; i < MORPHO_REDUCE + 1; i++) {
            char c =
                requestWord.charAt(requestWord.length() - i);
            if (!Character.isLetter(c)) {
                continue;
            }

            int index = indexOf(
                lowS,
                startIndex,
                requestWord,
                0,
                requestWord.length() - 1 - i);
            if (index > 0) {
                int start = index + requestWord.length() - 1 - i;
                for (int j = start;
                     j < Math.min(start + MORPHO_THRSH, lowS.length());
                     j++)
                {
                    c = lowS.charAt(j);
                    if (Character.isISOControl(c)
                        || Character.isSpaceChar(c)
                        || Character.isWhitespace(c))
                    {
                        consumer.apply(index + 1, j);
                        return j;
                    }
                }

                int max = index + requestWord.length();
                consumer.apply(index + 1, max);
                return max;
            }
        }

        return -1;
    }

    private int indexOf(
        final String source,
        final int sourceOffset,
        final String target,
        final int targetOffset,
        final int targetCount)
    {
        if (targetCount == 0) {
            return 0;
        }

        char first = target.charAt(targetOffset);
        int max = (source.length() - targetCount);

        for (int i = sourceOffset; i <= max; i++) {
            if (source.charAt(i) != first) {
                while (++i <= max && source.charAt(i) != first) {
                    ;
                }
            }

            if (i <= max) {
                int j = i + 1;
                int end = j + targetCount - 1;
                for (int k = targetOffset + 1; j < end && source.charAt(j)
                    == target.charAt(k); j++, k++) {
                    ;
                }

                if (j == end) {
                    return i;
                }
            }
        }
        return -1;
    }

    public String prepare(final String s) {
        return StringUtils.concat(' ', s.trim().toLowerCase(Locale.ROOT));
    }

    public List<String> prepareRequest(final String request) {
        List<String> words = new ArrayList<>();
        boolean space = true;
        StringBuilder sb = null;
        for (int i = 0; i < request.length(); i++) {
            char c = request.charAt(i);
            if (Character.isISOControl(c)
                || Character.isSpaceChar(c)
                || Character.isWhitespace(c))
            {
                if (!space) {
                    words.add(sb.toString());
                    sb = null;
                }

                space = true;
                continue;
            } else if (space) {
                space = false;
                sb = new StringBuilder();
            }

            sb.append(Character.toLowerCase(c));
        }

        if (sb != null) {
            words.add(sb.toString());
        }

        return words;
    }
}
