package ru.yandex.msearch.proxy.logger;

import java.io.IOException;

import java.util.Map;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.http.util.server.HttpServer;

import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;

import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;

import ru.yandex.parser.config.ConfigException;

import ru.yandex.tskv.ImmutableTskvFormatConfig;
import ru.yandex.tskv.PrefixedTskvLogger;
import ru.yandex.tskv.TskvFormatConfig;
import ru.yandex.tskv.TskvFormatConfigBuilder;
import ru.yandex.tskv.TskvString;

public class ProxyTskvLogger extends PrefixedTskvLogger<ProxyTskvLogger> {

    private final static String TSKV_FORMAT_KEY = "tskv_format";
    private final static String TSKV_FORMAT_VALUE = "mail-msproxy-factors-log";
    private final static String REFERER = "referer";
    private final static String MODULE_FIELD = "module";
    private final static String SESSION_FIELD = "session";
    private final static String HOST = "host";
    private final static String URI = "uri";

    private static java.util.logging.Logger logger(
        final ImmutableMsearchProxyConfig config)
        throws ConfigException
    {
        return config.tskvLogConfig().build(config.loggers().handlersManager());
    }

    public ProxyTskvLogger prefix(final String session, final String module) {
        TskvString record = record();
        record.append(MODULE_FIELD, module);
        record.append(SESSION_FIELD, session);
        TskvFormatConfigBuilder builder = new TskvFormatConfigBuilder(config);
        builder.linePrefix(record.toString());
        return new ProxyTskvLogger(logger, builder);
    }

    protected ProxyTskvLogger(java.util.logging.Logger logger) {
        super(logger);
    }

    public void log(final HttpCoreContext context, final String... pairs) {
        log(record(context, pairs));
    }

    public void log(final JsonMap doc, String... pairs) throws IOException {
        TskvString record = record();
        try {
            for (Map.Entry<String, JsonObject> entry: doc.entrySet()) {
                record.append(entry.getKey(), entry.getValue().asString());
            }
        } catch (JsonBadCastException jbe) {
            throw new IOException(jbe);
        }

        record.append(pairs);
        log(record);
    }

    protected ProxyTskvLogger(
        final Logger logger,
        final TskvFormatConfig config)
    {
        super(logger, config);
    }

    public ProxyTskvLogger(
        final ImmutableMsearchProxyConfig config)
        throws ConfigException
    {
        this(logger(config), new ImmutableTskvFormatConfig());
    }

    public TskvString record(final HttpCoreContext context) {
        HttpRequest request =
            (HttpRequest) context.getAttribute(HttpCoreContext.HTTP_REQUEST);
        String session = (String) context.getAttribute(HttpServer.SESSION_ID);
        String host = context.getTargetHost() == null ? "": context.getTargetHost().getHostName();

        TskvString record = record();
        record.append(TSKV_FORMAT_KEY, TSKV_FORMAT_VALUE);
        record.append(HOST, host);
        record.append(SESSION_FIELD, session);
        return record;
    }

    protected TskvString record(
        final HttpCoreContext context,
        final String... pairs)
    {
        HttpRequest request =
            (HttpRequest) context.getAttribute(HttpCoreContext.HTTP_REQUEST);
        String session = (String) context.getAttribute(HttpServer.SESSION_ID);
        Header refererHeader = request.getFirstHeader(HttpHeaders.REFERER);
        String referer = refererHeader == null ? "": refererHeader.getValue();
        String host = context.getTargetHost() == null ? "": context.getTargetHost().getHostName();

        TskvString record = record().append(pairs);
        record.append(TSKV_FORMAT_KEY, TSKV_FORMAT_VALUE);
        record.append(SESSION_FIELD, session);
        record.append(REFERER, referer);
        record.append(HOST, host);
        record.append(URI, request.getRequestLine().getUri());
        return record;
    }

    public ProxyTskvLogger prefix(final HttpCoreContext context) {
        return prefix(record(context));
    }

    public ProxyTskvLogger prefix(
        final HttpCoreContext context,
        final String... pairs)
    {
        return prefix(record(context, pairs));
    }

    @Override
    public TskvString record() {
        return new TskvString(config);
    }

    @Override
    protected ProxyTskvLogger create(
        final Logger logger, final TskvFormatConfig config)
    {
        return new ProxyTskvLogger(logger, config);
    }
}
