package ru.yandex.msearch.proxy.search;

import java.io.PrintStream;
import java.io.IOException;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.LinkedList;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.TreeMap;

import ru.yandex.blackbox.BlackboxUserinfo;

import ru.yandex.msearch.proxy.searchmap.SearchMap;
import ru.yandex.msearch.proxy.collector.Collector;
import ru.yandex.msearch.proxy.collector.SortingCollector;
import ru.yandex.msearch.proxy.collector.LinkedHashCollector;
import ru.yandex.msearch.proxy.logger.Logger;
import ru.yandex.msearch.proxy.ora.WmiFilterSearchClient;
import ru.yandex.msearch.proxy.ora.FilterSearch;
import ru.yandex.msearch.proxy.OutputPrinter;
import ru.yandex.msearch.proxy.document.Document;
import ru.yandex.msearch.proxy.HttpServer;
import ru.yandex.msearch.proxy.HttpServer.HttpParams;
import ru.yandex.msearch.proxy.MsearchProxyException;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class MailSearcher
{
private final String mailDb;
private final String suid;
private final String db;
private String order;
private String serverOrder;
private boolean desc = true;
String scope;
String searchFolders;
String folderSet = "";
List<String> fids;
List<String> lids;
PrintStream ps;
SearchMap searchMap;
private int found;
private boolean imapSearch;
private boolean resolveFids = false;
private boolean unread = false;
private boolean excludeSpam = false;
private boolean excludeTrash = false;
private int from = -1;
private int to = -1;
private String getFields;
private HttpServer.RequestContext ctx;
private int usersPerSearch = USERS_PER_SEARCH;
private boolean attachmentsSearch = false;
private Map<String,String> requestParams = null;

private static int SEARCH_PREFETCH_WINDOW;
private static int ORACLE_BATCH_SIZE;
private static int FAST_FILTER_ORACLE_BATCH_SIZE;
private static String FAST_FILTER_ORACLE_BATCH_SIZE_OVERRIDE;
public static final int MIN_ORACLE_BATCH_SIZE = 100;
public static final int MAX_THREADS = 1;
public static final int USERS_PER_SEARCH = 100;

    public static void init(final IniConfig config) throws ConfigException {
        SEARCH_PREFETCH_WINDOW = config.getInt("search_prefetch_window", 10000);
        ORACLE_BATCH_SIZE = config.getInt("oracle_batch_size", 300);
        FAST_FILTER_ORACLE_BATCH_SIZE =
            config.getInt("fast_filter_oracle_batch_size", 30000);
        FAST_FILTER_ORACLE_BATCH_SIZE_OVERRIDE = config.getString(
            "fast_filter_oracle_batch_size_override", "mdb303:2000");
    }

    public MailSearcher( HttpServer.RequestContext ctx, String mailDb, String suid ) throws MsearchProxyException
    {
	this.searchMap = SearchMap.getInstance();
	this.mailDb = mailDb;
	this.suid = suid;
        if (BlackboxUserinfo.corp(Long.parseLong(suid))) {
            db = ctx.server().config().pgCorpQueue();
        } else {
            db = ctx.server().config().pgQueue();
        }
	this.order = "default";
	this.searchFolders = "search_folders_private";
	this.fids = new LinkedList<String>();
	this.lids = new LinkedList<String>();
	this.imapSearch = false;
	this.getFields = "mid,received_date";
	this.ctx = ctx;
    }

    public void setUsersPerSearch( int ups )
    {
	usersPerSearch = ups;
    }

    public void setFids( List<String> fids )
    {
	this.fids = fids;
    }

    public void setLids( List<String> lids )
    {
	this.lids = lids;
    }

    public void setImapSearch( boolean is )
    {
	this.imapSearch = is;
    }

    public void setFields( String fields )
    {
	this.getFields = fields;
    }

    public void setSearchScope( String scope )
    {
	this.scope = scope;
    }

    public void setSearchFolders( String searchFolders )
    {
	this.searchFolders = searchFolders;
    }

    public void setFolderSet( String folderSet )
    {
	this.folderSet = folderSet;
    }

    public void setResolveFids( boolean resolveFids )
    {
	this.resolveFids = resolveFids;
    }

    public void setUnread( boolean unread )
    {
	this.unread = unread;
    }

    public void setExcludeSpam(final boolean excludeSpam) {
	this.excludeSpam = excludeSpam;
    }

    public void setExcludeTrash(final boolean excludeTrash) {
	this.excludeTrash = excludeTrash;
    }

    public void setFromDate( int from )
    {
        this.from = from;
    }

    public void setToDate( int to )
    {
        this.to = to;
    }

    public void setServerOrder( String serverOrder )
    {
        this.serverOrder = serverOrder;
    }

    public void setDesc(final boolean desc) {
        this.desc = desc;
    }

    public void setAttachmentsSearch( boolean attachments )
    {
        this.attachmentsSearch = attachments;
    }

    public void setRequestParam(final String param, final String value) {
        if (requestParams == null) {
            requestParams = new HashMap<String,String>();
        }
        requestParams.put(param, value);
    }

    public void search(
        final Collector collector,
        final String request,
        final int toFind)
        throws MsearchProxyException
    {
        checkParams();
        List<String> outFids = null;
        if (resolveFids) {
            outFids = new LinkedList<String>();
        }
        String[] users = new String[]{suid};
        if (resolveFids) {
            setFids(outFids);
        }
        HttpParams searchParams = buildParams(request);

        int lastTotalCount = 0;
        int searchLength = SEARCH_PREFETCH_WINDOW;
        if( searchLength < toFind ) searchLength = toFind + toFind / 4;
        int found = 0;
        int searchOffset = 0;
        do
        {
            ctx.checkAbort();
            Collector tempCollector = new LinkedHashCollector( "mid" );
            boolean hasNext = requestSearchServers( ctx, db, searchParams, tempCollector, users, searchOffset, searchLength );
            lastTotalCount = tempCollector.getTotalCount();
            if( tempCollector.getCount() <= searchOffset )
            {
                ctx.log.info( "no more results from search servers" );
                break;
            }

            ctx.log.info( "filtering in oracle" );
            found += filterSearch( tempCollector, collector, order, searchOffset, toFind - found );
            ctx.log.info( found + " found after filtering in oracle" );
            if( found >= toFind )
            {
                ctx.log.info( "found needed results: " + found + "/" + toFind );
                break;
            }
            searchOffset += searchLength;
            if( !hasNext )
            {
                ctx.log.info( "no more results from search servers" );
    	        break;
            }
        }
        while( true );

        ctx.log.info( "found " + found + " documents" );
        ctx.log.info( "found " + found + " (" + collector.getCount() + ") documents" );
        if (attachmentsSearch) {
            found = lastTotalCount;
        }
        collector.incTotalCount( found );
    }

    private HttpParams buildParams(final String request) {
        HttpParams params = new HttpParams();
        if (imapSearch) {
            params.add("imap", "1");
        }
        if (attachmentsSearch) {
            params.add("merge_func", "none");
        }
        params.add("user", suid);
        params.add("text", request);
        params.add("scope", scope);
        params.add("getfields", getFields);
        params.add("mdb", db);
        params.add("orderby", serverOrder);
        params.add("asc", Boolean.toString(!desc));
        if (from != -1) {
            params.add("from_day", Integer.toString(from % 100));
            params.add("from_month", Integer.toString((from / 100) % 100 - 1));
            params.add("from_year", Integer.toString(from / 10000));
        }
        if (to != -1)
        {
            params.add("to_day", Integer.toString(to % 100));
            params.add("to_month", Integer.toString((to / 100) % 100 - 1));
            params.add("to_year", Integer.toString(to / 10000));
        }
        if (from != -1 || to != -1) {
            params.add("dated", "1");
        }
        if (requestParams != null) {
            params.addAll(requestParams);
        }
        return params;
    }

    private boolean requestSearchServers( HttpServer.RequestContext ctx, String db, HttpParams params, Collector collector, String[] users, int offset, int length ) throws MsearchProxyException
    {
        ctx.log.info( "querying search servers with offset/length: " + offset + "/" + length );
        params.replace( "offset", Integer.toString( offset ) );
        params.replace( "length", Integer.toString( length ) );
        Searcher searcher = new Searcher( ctx, usersPerSearch, searchMap, params, collector  );
        ctx.log.info( "before search" );
        searcher.search( db, users );
        ctx.log.info( "after search" );
        ctx.log.info( "servers returned " + collector.getTotalCount() + " documents" );
        return collector.getCount() >= offset + length;
    }

    private static int getFastFilterBatchSize( String db )
    {
	int size = FAST_FILTER_ORACLE_BATCH_SIZE;
	try
	{
	    String[] overrides = FAST_FILTER_ORACLE_BATCH_SIZE_OVERRIDE.split( " " );
	    for( int i = 0; i < overrides.length; i++ )
	    {
		if( overrides[i].startsWith( db + ":" ) )
		{
		    size = Integer.parseInt( overrides[i].substring( overrides[i].indexOf( ':' ) + 1 ) );
		    break;
		}
	    }
	}
	catch( Exception e )
	{
	}
	return size;
    }

    private int filterSearch( Collector input, Collector output, String order, int searchOffset, int count ) throws MsearchProxyException
    {
	final int maxBatchSize = ORACLE_BATCH_SIZE;
	int batchSize = count > maxBatchSize + maxBatchSize / 2 ? maxBatchSize : 
			(count / MAX_THREADS < MIN_ORACLE_BATCH_SIZE ? MIN_ORACLE_BATCH_SIZE : count / MAX_THREADS);
	Map<Document, Document> inputHits = input.hits();
	Iterator<Document> inputIter = inputHits.keySet().iterator();

	for( int i = 0; i < searchOffset && inputIter.hasNext(); i++ ) inputIter.next();

	FilterSearch filter = new WmiFilterSearchClient( ctx.getSessionId(), ctx );

	int found = 0;
	while( inputIter.hasNext() )
	{
	    Map<String, Document> toFilter = new LinkedHashMap<>();
	    filter.setMailDb( mailDb );
	    filter.setSuid( suid );
	    filter.setOrder( order );
	    filter.setFids( fids );
	    filter.setLids( lids );
	    filter.setUnread( unread );
	    filter.setFolderSet( folderSet );
            filter.setExcludeSpam(excludeSpam);
            filter.setExcludeTrash(excludeTrash);
	    for( int i = 0; i < batchSize && inputIter.hasNext(); i++ )
	    {
		Document d = inputIter.next();
		toFilter.put( d.getAttr("mid"), d );
		inputIter.remove();
	    }
	    if( toFilter.size() > 0 )
	    {
		ctx.checkAbort();
		ctx.log.info( "filtering in oracle: " + toFilter.size() );
                long start = System.currentTimeMillis();
                try
                {
                    found += filter.filter( toFilter, output );
                }
                catch( Exception e )
                {
                    ctx.log.err( "filter call error: " + e.getMessage() + "\n" + Logger.exception(e) );
                    throw new MsearchProxyException("/filter_search request failed", e);
                }
                ctx.log.debug( "filter call finished in: " + (System.currentTimeMillis() - start) + " ms" );
	    }
	    if( found >= count ) break;
	}
	return found;
    }

    private void checkParams() throws MsearchProxyException
    {
	if( mailDb == null )
	{
	    ctx.log.err( "MailSearcher: mailDb parameter is empty" );
	    throw new MsearchProxyException( "mailDb parameter is empty" );
	}
	
	if( suid == null )
	{
	    ctx.log.err( "MailSearcher: suid parameter is empty" );
	    throw new MsearchProxyException( "suid parameter is empty" );
	}
    }

    public int status()
    {
	return found;
    }
}
