package ru.yandex.search.mail.indexer.pg;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.function.Supplier;

import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.StepBackPrimitiveIterator;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySessionCallback;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.writer.JsonType;

public class PgIndexer
    extends HttpProxy<ImmutablePgIndexerConfig>
    implements ProxyRequestHandler
{
    private static final String BATCH_SIZE = "batch-size=";
    private static final String MIDS = "&mids=";

    private final AsyncClient msalClient;
    private final AsyncClient producerClient;
    private final ContentType producerContentType;

    public PgIndexer(final ImmutablePgIndexerConfig config)
        throws IOException
    {
        super(config);
        msalClient = client("MSAL", config.msalConfig());
        producerClient = client("Producer", config.producerConfig());
        producerContentType = ContentType.APPLICATION_JSON.withCharset(
            producerClient.requestCharset());
        register(
            new Pattern<>("/reindex", false),
            this,
            RequestHandlerMapper.GET);
    }

    @Override
    public void handle(final ProxySession session) throws BadRequestException {
        handle(new ReindexContext(session));
    }

    private void handle(final ReindexContext context) {
        AsyncClient client = msalClient.adjust(context.session().context());
        client.execute(
            config.msalConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                "/get-user-revision?json-type=dollar&uid=" + context.uid()),
            AsyncStringConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(client),
            new RevisionCallback(context));
    }

    private static String changeTypeToString(final ChangeType type) {
        return type.name().toLowerCase(Locale.ROOT).replace('_', '-');
    }

    private class RevisionCallback
        extends AbstractProxySessionCallback<String>
    {
        private final ReindexContext context;

        RevisionCallback(final ReindexContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final String result) {
            long revision;
            try {
                revision = Long.parseLong(result);
            } catch (NumberFormatException e) {
                failed(e);
                return;
            }
            context.session().logger().info(
                "Revision retrieved for uid " + context.uid()
                + " is " + revision);
            AsyncClient client =
                msalClient.adjust(context.session().context());
            client.execute(
                config.msalConfig().host(),
                new BasicAsyncRequestProducerGenerator(
                    "/user-mids?json-type=dollar&uid=" + context.uid()),
                MidsConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(client),
                new MidsCallback(context, revision));
        }
    }

    private class MidsCallback extends AbstractProxySessionCallback<Mids> {
        private final ReindexContext context;
        private final long revision;
        private final AsyncClient client;
        private final Supplier<? extends HttpClientContext> contextGenerator;

        MidsCallback(final ReindexContext context, final long revision) {
            super(context.session());
            this.context = context;
            this.revision = revision;
            client = producerClient.adjust(context.session().context());
            contextGenerator =
                context.session().listener().createContextGeneratorFor(client);
        }

        private String prepareUri(final String action, final String suffix) {
            return "/notify?mdb=pg&uid=" + context.uid()
                + "&revision=" + revision
                + "&change-type=" + action
                + '&' + suffix;
        }

        private String prepareRequestBody(
            final String action,
            final String pgShard,
            final long... mids)
        {
            StringBuilder sb = new StringBuilder(
                "{\"operation_id\":\"0\",\"fresh_count\":\"0\","
                + "\"useful_new_messages\":\"0\",\"uid\":\"");
            sb.append(context.uid());
            sb.append("\",\"lcn\":\"");
            sb.append(revision);
            sb.append("\",\"change_type\":\"");
            sb.append(action);
            sb.append("\",\"operation_date\":\"");
            sb.append(System.currentTimeMillis());
            sb.insert(sb.length() - 2 - 1, '.');
            sb.append("\",\"pgshard\":");
            sb.append(pgShard);
            sb.append(",\"changed\":[");
            for (int i = 0; i < mids.length; ++i) {
                if (i != 0) {
                    sb.append(',');
                }
                sb.append("{\"mid\":");
                sb.append(mids[i]);
                sb.append('}');
            }
            sb.append(']');
            sb.append('}');
            return new String(sb);
        }

        // CSOFF: ParameterNumber
        private BasicAsyncRequestProducerGenerator prepareCleanupRequest(
            final String uriSuffix,
            final long midFirst,
            final long midLast,
            final String pgShard)
        {
            String action = changeTypeToString(ChangeType.SEARCH_MIDS_CLEANUP);
            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(
                    prepareUri(
                        action,
                        uriSuffix
                        + "&batch-size=1&mids=" + midFirst + '-' + midLast),
                    prepareRequestBody(action, pgShard, midFirst, midLast),
                    producerContentType);
            producerGenerator.addHeader(
                YandexHeaders.SERVICE,
                config.serviceName());
            producerGenerator.addHeader(
                YandexHeaders.ZOO_SHARD_ID,
                Long.toString(context.shard()));
            return producerGenerator;
        }
        // CSON: ParameterNumber

        private void indexBatches(
            final List<long[]> batches,
            final MultiFutureCallback<Void> callback,
            final String pgShard)
            throws IOException
        {
            String action = changeTypeToString(ChangeType.SEARCH_UPDATE);
            int size = batches.size();
            long[] lastBatch = batches.get(size - 1);
            String uri = prepareUri(
                action,
                BATCH_SIZE + batches.size()
                + MIDS + batches.get(0)[0]
                + '-' + lastBatch[lastBatch.length - 1]);
            BasicAsyncRequestProducerGenerator producerGenerator;
            if (batches.size() == 1) {
                producerGenerator = new BasicAsyncRequestProducerGenerator(
                    uri,
                    prepareRequestBody(action, pgShard, lastBatch),
                    producerContentType);
                producerGenerator.addHeader(
                    YandexHeaders.ZOO_SHARD_ID,
                    Long.toString(context.shard()));
            } else {
                MultipartEntityBuilder builder =
                    MultipartEntityBuilder.create();
                builder.setMimeSubtype("mixed");
                for (long[] batch: batches) {
                    builder.addPart(
                        FormBodyPartBuilder
                            .create()
                            .addField(
                                YandexHeaders.ZOO_SHARD_ID,
                                Long.toString(context.shard()))
                            .addField(
                                YandexHeaders.URI,
                                prepareUri(
                                    action,
                                    BATCH_SIZE + batch.length
                                    + MIDS + batch[0]
                                    + '-' + batch[batch.length - 1]))
                            .setBody(
                                new StringBody(
                                    prepareRequestBody(action, pgShard, batch),
                                    producerContentType))
                            .setName("envelope.json")
                            .build());
                }
                producerGenerator = new BasicAsyncRequestProducerGenerator(
                    uri,
                    builder.build());
            }
            producerGenerator.addHeader(
                YandexHeaders.SERVICE,
                config.serviceName());
            client.execute(
                config.producerConfig().host(),
                producerGenerator,
                EmptyAsyncConsumerFactory.OK,
                contextGenerator,
                callback.newCallback());
        }

        @Override
        public void completed(final Mids mids) {
            String pgShard = JsonType.NORMAL.toString(mids.pgShard());
            StepBackPrimitiveIterator.OfLong iter = mids.iterator();
            if (!iter.hasNext()) {
                context.incrementIteration();
                int retries = config.midsRetrievalRetries();
                if (context.iteration() <= retries) {
                    context.session().logger().warning(
                        "At iteration " + context.iteration()
                         + " of " + retries
                         + " not mids returned from database, retrying");
                    handle(context);
                } else {
                    context.session().logger().warning(
                        "No mids retrieved for user after "
                        + context.iteration()
                        + " iterations. Cleaning index for user "
                        + context.uid());
                    client.execute(
                        config.producerConfig().host(),
                        prepareCleanupRequest("empty-user", 2L, 1L, pgShard),
                        EmptyAsyncConsumerFactory.OK,
                        contextGenerator,
                        new BasicProxySessionCallback(context.session()));
                }
            } else {
                long[] midsBatch =
                    new long[Math.max(2, config.midsPerRequest())];
                List<long[]> midsBatches = new ArrayList<>();
                int length = 0;
                long lastMid = 0L;
                while (true) {
                    int batchSize = 0;
                    while (batchSize < midsBatch.length && iter.hasNext()) {
                        lastMid = iter.nextLong();
                        midsBatch[batchSize++] = lastMid;
                        ++length;
                    }
                    midsBatches.add(Arrays.copyOf(midsBatch, batchSize));
                    if (iter.hasNext()) {
                        // step back to create overlap
                        iter.stepBack();
                    } else {
                        break;
                    }
                }
                context.session().logger().info(
                    length + " mids splitted to "
                    + midsBatches.size() + " batches");
                MultiFutureCallback<Void> callback = new MultiFutureCallback<>(
                    new BasicProxySessionCallback(context.session()));
                client.execute(
                    config.producerConfig().host(),
                    prepareCleanupRequest(
                        "cleanup-user",
                        mids.iterator().nextLong(),
                        lastMid,
                        pgShard),
                    EmptyAsyncConsumerFactory.OK,
                    contextGenerator,
                    callback.newCallback());
                mids.clear();
                int pos = 0;
                List<long[]> producerBatch = new ArrayList<>();
                int subrequests = 0;
                int batchSize = config.producerBatchSize();
                while (true) {
                    producerBatch.clear();
                    while (pos < midsBatches.size()
                        && producerBatch.size() < batchSize)
                    {
                        producerBatch.add(midsBatches.get(pos++));
                    }
                    try {
                        indexBatches(producerBatch, callback, pgShard);
                    } catch (IOException e) {
                        failed(e);
                        return;
                    }
                    ++subrequests;
                    if (pos == midsBatches.size()) {
                        break;
                    }
                }
                callback.done();
                context.session().logger()
                    .info(subrequests + " subrequests sent to producer");
            }
        }
    }
}

