package ru.yandex.search.msal.pool;

import java.io.IOException;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public class DBConnectionPool implements JsonValue {
    private static final int SESSION_KILLED = 28;
    private static final int CONNECTION_POOLING_LIMIT_REACHED = 12602;

    private final Properties properties = new Properties();
    private final String url;
    private final ImmutablePoolConfig config;
    private final Logger logger;
    private final Deque<Connection> freeConn;
    private int allowedToCreate;
    private boolean closed = false;

    public DBConnectionPool(
        final String url,
        final ImmutablePoolConfig config,
        final Logger logger)
    {
        this.url = url;
        this.config = config;
        this.logger = logger;
        properties.put("user", config.user());
        properties.put("password", config.password());
        properties.putAll(config.properties());
        freeConn = new ArrayDeque<>(config.poolSize());
        allowedToCreate = config.poolSize();
        logger.info("Creating connection pool to: " + url);
        try {
            Driver driver =
                (Driver) Class.forName(config.driverName())
                    .getDeclaredConstructor().newInstance();
            DriverManager.registerDriver(driver);
        } catch (Throwable t) {
            logger.log(
                Level.SEVERE,
                "Can't register driver " + config.driverName(),
                t);
            throw new RuntimeException(t);
        }
        logger.info("Registered driver " + config.driverName());
    }

    private boolean isClosed(final Connection conn) throws SQLException {
        if (conn.isClosed()) {
            return true;
        }
        try (Statement stmt = conn.createStatement()) {
            stmt.execute(config.pingQuery());
        }
        return false;
    }

    private synchronized boolean isClosedWithAccount(
        final Connection conn,
        final Logger logger)
    {
        try {
            if (!isClosed(conn)) {
                return false;
            }
            logger.info("Connection is closed: " + conn);
        } catch (SQLException e) {
            closeConnection(conn, logger);
        }
        allowedToCreate++;
        return true;
    }

    private synchronized void checkClosed() throws SQLException {
        if (closed) {
            throw new SQLException(
                "Pool closed",
                null,
                SESSION_KILLED,
                null);
        }
    }

    public Connection getConnection(final Logger logger)
        throws SQLException
    {
        checkClosed();
        long deadline = System.currentTimeMillis() + config.timeout();
        boolean tryToCreate = false;
        Connection reuse = null;
        synchronized (this) {
            while (freeConn.isEmpty() && allowedToCreate <= 0) {
                long current = System.currentTimeMillis();
                if (current >= deadline) {
                    break;
                }
                try {
                    this.wait(deadline - current);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
            checkClosed();
            while (!freeConn.isEmpty()) {
                Connection conn = freeConn.pollLast();
                if (!isClosedWithAccount(conn, logger)) {
                    reuse = conn;
                    break;
                }
            }
            if (reuse == null && allowedToCreate > 0) {
                tryToCreate = true;
                allowedToCreate--;
            }
        }
        if (reuse != null) {
            logger.info("Reusing connection to " + url + ':' + ' ' + reuse);
            return newPoolConnection(reuse, logger);
        }
        if (tryToCreate) {
            try {
                Connection conn = newConnection();
                logger.info(
                    "Created new connection to " + url + ':' + ' ' + conn);
                return newPoolConnection(conn, logger);
            } catch (SQLException e) {
                logger.log(
                    Level.WARNING,
                    "Failed to create connection to " + url,
                    e);
                synchronized (this) {
                    allowedToCreate++;
                }
                throw e;
            }
        }
        logger.severe(url + ": Can't get connection");
        throw new SQLException(
            "Can't get connection",
            null,
            CONNECTION_POOLING_LIMIT_REACHED,
            null);
    }

    // put connection to pool & notify
    synchronized void freeConnection(
        final Connection conn,
        final Logger logger)
    {
        logger.info("Connection returned to pool " + url + ':' + ' ' + conn);
        if (conn == null) {
            logger.severe(url + ": Passed NULL connection to free");
        } else if (closed) {
            closeConnection(conn, logger);
        } else if (freeConn.size() + allowedToCreate + 1 > config.poolSize()) {
            closeConnection(conn, logger);
        } else {
            freeConn.add(conn);
            this.notifyAll();
        }
    }

    // create conection WITHOUT checking/decreasing allowedToCreate
    private Connection newConnection() throws SQLException {
        return DriverManager.getConnection(url, properties);
    }

    protected Connection newPoolConnection(
        final Connection conn,
        final Logger logger)
    {
        return new PoolConnection(conn, this, logger);
    }

    // close connection WITHOUT incresing allowedToCreate
    private void closeConnection(final Connection conn, final Logger logger) {
        logger.info("Connection is closing: " + conn);
        try {
            conn.close();
        } catch (SQLException e) {
            logger.log(Level.SEVERE, url + ": Can't close connection", e);
        }
    }

    public synchronized void close() {
        if (!closed) {
            closed = true;
            allowedToCreate = 0;
            for (Connection conn: freeConn) {
                closeConnection(conn, logger);
            }
            freeConn.clear();
        } else {
            logger.severe(url + ": Pool already closed");
        }
    }

    public int prefetchSize() {
        return config.prefetchSize();
    }

    public Map<String, Object> status() {
        int allowedToCreate;
        int available;
        synchronized (this) {
            allowedToCreate = this.allowedToCreate;
            available = freeConn.size();
        }
        Map<String, Object> status = new LinkedHashMap<>();
        status.put("url", url);
        status.put(
            "active-connections",
            config.poolSize() - available - allowedToCreate);
        status.put("available-connections", available);
        status.put("max-connections", config.poolSize());
        return status;
    }

    @Override
    public String toString() {
        return url;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.value(status());
    }
}

