package ru.yandex.search.msal.pool;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.function.GenericFunction;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;

public class PoolConfigBuilder implements PoolConfig {
    private static final int ATTRS_COUNT = 5;
    private static final int USERNAME = 3;
    private static final int PASSWORD = 4;
    private static final String ANY = "*";

    private String driverName;
    private String user;
    private String password;
    private String pingQuery;
    private int poolSize;
    private int timeout;
    private int prefetchSize;
    private Map<String, String> properties;

    public PoolConfigBuilder() {
        this(PoolConfigDefaults.INSTANCE);
    }

    public PoolConfigBuilder(final PoolConfig config) {
        driverName = config.driverName();
        user = config.user();
        password = config.password();
        pingQuery = config.pingQuery();
        poolSize = config.poolSize();
        timeout = config.timeout();
        prefetchSize = config.prefetchSize();
        properties = new HashMap<>(config.properties());
    }

    public PoolConfigBuilder(final IniConfig config)
        throws ConfigException
    {
        this(config, PoolConfigDefaults.INSTANCE);
    }

    public PoolConfigBuilder(final IniConfig config, final PoolConfig defaults)
        throws ConfigException
    {
        driverName = config.get(
            "driver",
            defaults.driverName(),
            NonEmptyValidator.INSTANCE);
        user = config.get(
            "user",
            defaults.user(),
            NonEmptyValidator.INSTANCE);
        password = config.get(
            "password",
            null,
            NonEmptyValidator.INSTANCE);
        if (password == null && user != null) {
            String pgpassFilename = config.getString("pgpass", "");
            File pgpass = new File(pgpassFilename);
            if (pgpass.exists()) {
                List<String> lines;
                try {
                    lines = Files.readAllLines(
                        pgpass.toPath(),
                        StandardCharsets.UTF_8);
                } catch (IOException e) {
                    throw new ConfigException(
                        "Failed to read pgpass from " + pgpass,
                        e);
                }

                AuthParser parser =
                    config.getEnum(
                        AuthParser.class,
                        "auth-parser",
                        AuthParser.DEFAULT);

                String env = config.getString(
                    "enviroment",
                    "prod");

                for (String line: lines) {
                    if (!line.isEmpty()) {
                        AuthAttrs attrs = parser.apply(line);
                        if (attrs == null) {
                            continue;
                        }

                        if ((ANY.equalsIgnoreCase(attrs.enviroment)
                            || env.equalsIgnoreCase(attrs.enviroment))
                            && (attrs.user.equals(user)
                            || attrs.user.equals(ANY)))
                        {
                            password = attrs.password;
                            break;
                        }
                    }
                }
            }
        }
        if (password == null) {
            password = defaults.password();
        }
        pingQuery = config.get(
            "ping-query",
            defaults.pingQuery(),
            NonEmptyValidator.INSTANCE);
        poolSize = config.get(
            "size",
            defaults.poolSize(),
            PositiveIntegerValidator.INSTANCE);
        timeout = config.get(
            "timeout",
            defaults.timeout(),
            PositiveIntegerValidator.INSTANCE);
        prefetchSize = config.get(
            "prefetch",
            defaults.prefetchSize(),
            NonNegativeIntegerValidator.INSTANCE);

        IniConfig properties = config.sectionOrNull("properties");
        if (properties == null) {
            this.properties = new HashMap<>(defaults.properties());
        } else {
            Set<String> keys = properties.keys();
            this.properties = new HashMap<>(keys.size() << 1);
            for (String key: keys) {
                this.properties.put(key, properties.getString(key));
            }
        }
    }

    @Override
    public String driverName() {
        return driverName;
    }

    public PoolConfigBuilder driverName(final String driverName) {
        this.driverName = driverName;
        return this;
    }

    @Override
    public String user() {
        return user;
    }

    public PoolConfigBuilder user(final String user) {
        this.user = user;
        return this;
    }

    @Override
    public String password() {
        return password;
    }

    public PoolConfigBuilder password(final String password) {
        this.password = password;
        return this;
    }

    @Override
    public String pingQuery() {
        return pingQuery;
    }

    public PoolConfigBuilder pingQuery(final String pingQuery) {
        this.pingQuery = pingQuery;
        return this;
    }

    @Override
    public int poolSize() {
        return poolSize;
    }

    public PoolConfigBuilder poolSize(final int poolSize) {
        this.poolSize = poolSize;
        return this;
    }

    @Override
    public int timeout() {
        return timeout;
    }

    public PoolConfigBuilder timeout(final int timeout) {
        this.timeout = timeout;
        return this;
    }

    @Override
    public int prefetchSize() {
        return prefetchSize;
    }

    public PoolConfigBuilder prefetchSize(final int prefetchSize) {
        this.prefetchSize = prefetchSize;
        return this;
    }

    @Override
    public Map<String, String> properties() {
        return properties;
    }

    public PoolConfigBuilder properties(final Map<String, String> properties) {
        this.properties = new HashMap<>(properties);
        return this;
    }

    public ImmutablePoolConfig build() throws ConfigException {
        return new ImmutablePoolConfig(this);
    }

    private static final class AuthAttrs {
        private final String user;
        private final String password;
        private final String enviroment;

        private AuthAttrs(
            final String user,
            final String password,
            final String enviroment)
        {
            this.user = user;
            this.password = password;
            this.enviroment = enviroment;
        }
    }

    private enum AuthParser
        implements GenericFunction<String, AuthAttrs, ConfigException>
    {
        DEFAULT {
            @Override
            public AuthAttrs apply(final String s) throws ConfigException {
                CollectionParser<String, List<String>, ConfigException> parser;
                if (!s.isEmpty()) {
                    parser =
                        new CollectionParser<>(x -> x, ArrayList::new, ':');
                    List<String> attrs = parser.apply(s);
                    if (attrs.size() == ATTRS_COUNT) {
                        String username = attrs.get(USERNAME);
                        String password = attrs.get(PASSWORD);

                        return new AuthAttrs(username, password, ANY);
                    }
                }

                return null;
            }
        },
        ACEVENTURA {
            @Override
            public AuthAttrs apply(final String s) throws ConfigException {
                String[] split = s.split(":");
                if (split.length == 2) {
                    String enviroment = split[0].trim();
                    String password = split[1].trim();
                    if (enviroment.isEmpty() || password.isEmpty()) {
                        throw new ConfigException(
                            "Bad formatted line, empty attr " + s);
                    }

                    return new AuthAttrs(ANY, password, enviroment);
                } else {
                    throw new ConfigException("Bad formatted line " + s);
                }
            }
        };
    }
}

