package ru.yandex.search.msal.pool;

import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.ClientInfoStatus;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.util.Collections;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Executor;

public class ProxyConnection implements Connection {
    private static final String CONNECTION_IS_CLOSED =
        "Connection is closed.";
    private static final Map<String, ClientInfoStatus> EMPTY_MAP =
        Collections.emptyMap();

    protected volatile Connection conn;
    private final int prefetchSize;

    public ProxyConnection(final Connection conn, final int prefetchSize) {
        this.conn = conn;
        this.prefetchSize = prefetchSize;
    }

    @Override
    public void close() {
        conn = null;
    }

    @Override
    public boolean isClosed() throws SQLException {
        return conn == null || conn.isClosed();
    }

    @Override
    @SuppressWarnings("deprecation")
    protected void finalize() throws Throwable {
        try {
            close();
        } finally {
            super.finalize();
        }
    }

    protected void checkClosed() throws SQLException {
        if (conn == null) {
            throw new SQLException(CONNECTION_IS_CLOSED);
        }
    }

    private <S extends Statement> S prefetched(final S stmt)
        throws SQLException
    {
        try (StatementHolder<S> holder = new StatementHolder<>(stmt)) {
            holder.get().setFetchSize(prefetchSize);
            return holder.release();
        }
    }

    // Generated code

    @Override
    public Statement createStatement() throws SQLException {
        checkClosed();
        return prefetched(conn.createStatement());
    }

    @Override
    public Statement createStatement(
        final int resultSetType,
        final int resultSetConcurrency)
        throws SQLException
    {
        checkClosed();
        return prefetched(
            conn.createStatement(resultSetType, resultSetConcurrency));
    }

    @Override
    public Statement createStatement(
        final int resultSetType,
        final int resultSetConcurrency,
        final int resultSetHoldability)
        throws SQLException
    {
        checkClosed();
        return prefetched(
            conn.createStatement(
                resultSetType,
                resultSetConcurrency,
                resultSetHoldability));
    }

    @Override
    public PreparedStatement prepareStatement(final String sql)
        throws SQLException
    {
        checkClosed();
        return prefetched(conn.prepareStatement(sql));
    }

    @Override
    public PreparedStatement prepareStatement(
        final String sql,
        final int resultSetType,
        final int resultSetConcurrency)
        throws SQLException
    {
        checkClosed();
        return prefetched(
            conn.prepareStatement(sql, resultSetType, resultSetConcurrency));
    }

    @Override
    public PreparedStatement prepareStatement(
        final String sql,
        final int autoGeneratedKeys)
        throws SQLException
    {
        checkClosed();
        return prefetched(conn.prepareStatement(sql, autoGeneratedKeys));
    }

    @Override
    public PreparedStatement prepareStatement(
        final String sql,
        final int[] columnIndexes)
        throws SQLException
    {
        checkClosed();
        return prefetched(conn.prepareStatement(sql, columnIndexes));
    }

    @Override
    public PreparedStatement prepareStatement(
        final String sql,
        final String[] columnNames)
        throws SQLException
    {
        checkClosed();
        return prefetched(conn.prepareStatement(sql, columnNames));
    }

    // CSOFF: ParameterNumber
    @Override
    public PreparedStatement prepareStatement(
        final String sql,
        final int resultSetType,
        final int resultSetConcurrency,
        final int resultSetHoldability)
        throws SQLException
    {
        checkClosed();
        return prefetched(
            conn.prepareStatement(
                sql,
                resultSetType,
                resultSetConcurrency,
                resultSetHoldability));
    }
    // CSON: ParameterNumber

    @Override
    public CallableStatement prepareCall(final String sql) throws SQLException {
        checkClosed();
        return prefetched(conn.prepareCall(sql));
    }

    @Override
    public CallableStatement prepareCall(
        final String sql,
        final int resultSetType,
        final int resultSetConcurrency)
        throws SQLException
    {
        checkClosed();
        return prefetched(
            conn.prepareCall(sql, resultSetType, resultSetConcurrency));
    }

    // CSOFF: ParameterNumber
    @Override
    public CallableStatement prepareCall(
        final String sql,
        final int resultSetType,
        final int resultSetConcurrency,
        final int resultSetHoldability)
        throws SQLException
    {
        checkClosed();
        return prefetched(
            conn.prepareCall(
                sql,
                resultSetType,
                resultSetConcurrency,
                resultSetHoldability));
    }
    // CSON: ParameterNumber

    @Override
    public String nativeSQL(final String sql) throws SQLException {
        checkClosed();
        return conn.nativeSQL(sql);
    }

    @Override
    public void setAutoCommit(final boolean autoCommit) throws SQLException {
        checkClosed();
        conn.setAutoCommit(autoCommit);
    }

    @Override
    public boolean getAutoCommit() throws SQLException {
        checkClosed();
        return conn.getAutoCommit();
    }

    @Override
    public void commit() throws SQLException {
        checkClosed();
        conn.commit();
    }

    @Override
    public void rollback() throws SQLException {
        checkClosed();
        conn.rollback();
    }

    @Override
    public void rollback(final Savepoint savepoint) throws SQLException {
        checkClosed();
        conn.rollback(savepoint);
    }

    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        checkClosed();
        return conn.getMetaData();
    }

    @Override
    public void setReadOnly(final boolean readOnly) throws SQLException {
        checkClosed();
        conn.setReadOnly(readOnly);
    }

    @Override
    public boolean isReadOnly() throws SQLException {
        checkClosed();
        return conn.isReadOnly();
    }

    @Override
    public void setCatalog(final String catalog) throws SQLException {
        checkClosed();
        conn.setCatalog(catalog);
    }

    @Override
    public String getCatalog() throws SQLException {
        checkClosed();
        return conn.getCatalog();
    }

    @Override
    public void setTransactionIsolation(final int level) throws SQLException {
        checkClosed();
        conn.setTransactionIsolation(level);
    }

    @Override
    public int getTransactionIsolation() throws SQLException {
        checkClosed();
        return conn.getTransactionIsolation();
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        checkClosed();
        return conn.getWarnings();
    }

    @Override
    public void clearWarnings() throws SQLException {
        checkClosed();
        conn.clearWarnings();
    }

    @Override
    public Map<String, Class<?>> getTypeMap() throws SQLException {
        checkClosed();
        return conn.getTypeMap();
    }

    @Override
    public void setTypeMap(final Map<String, Class<?>> map)
        throws SQLException
    {
        checkClosed();
        conn.setTypeMap(map);
    }

    @Override
    public void setHoldability(final int holdability) throws SQLException {
        checkClosed();
        conn.setHoldability(holdability);
    }

    @Override
    public int getHoldability() throws SQLException {
        checkClosed();
        return conn.getHoldability();
    }

    @Override
    public Savepoint setSavepoint() throws SQLException {
        checkClosed();
        return conn.setSavepoint();
    }

    @Override
    public Savepoint setSavepoint(final String name) throws SQLException {
        checkClosed();
        return conn.setSavepoint(name);
    }

    @Override
    public void releaseSavepoint(final Savepoint savepoint)
        throws SQLException
    {
        checkClosed();
        conn.releaseSavepoint(savepoint);
    }

    @Override
    public Clob createClob() throws SQLException {
        checkClosed();
        return conn.createClob();
    }

    @Override
    public Blob createBlob() throws SQLException {
        checkClosed();
        return conn.createBlob();
    }

    @Override
    public NClob createNClob() throws SQLException {
        checkClosed();
        return conn.createNClob();
    }

    @Override
    public SQLXML createSQLXML() throws SQLException {
        checkClosed();
        return conn.createSQLXML();
    }

    @Override
    public boolean isValid(final int timeout) throws SQLException {
        checkClosed();
        return conn.isValid(timeout);
    }

    @Override
    public void setClientInfo(final String name, final String value)
        throws SQLClientInfoException
    {
        if (conn == null) {
            throw new SQLClientInfoException(CONNECTION_IS_CLOSED, EMPTY_MAP);
        }
        conn.setClientInfo(name, value);
    }

    @Override
    public void setClientInfo(final Properties properties)
        throws SQLClientInfoException
    {
        if (conn == null) {
            throw new SQLClientInfoException(CONNECTION_IS_CLOSED, EMPTY_MAP);
        }
        conn.setClientInfo(properties);
    }

    @Override
    public String getClientInfo(final String name) throws SQLException {
        checkClosed();
        return conn.getClientInfo(name);
    }

    @Override
    public Properties getClientInfo() throws SQLException {
        checkClosed();
        return conn.getClientInfo();
    }

    @Override
    public Array createArrayOf(final String typeName, final Object[] elements)
        throws SQLException
    {
        checkClosed();
        return conn.createArrayOf(typeName, elements);
    }

    @Override
    public Struct createStruct(
        final String typeName,
        final Object[] attributes)
        throws SQLException
    {
        checkClosed();
        return conn.createStruct(typeName, attributes);
    }

    @Override
    public void setSchema(final String schema) throws SQLException {
        checkClosed();
        conn.setSchema(schema);
    }

    @Override
    public String getSchema() throws SQLException {
        checkClosed();
        return conn.getSchema();
    }

    @Override
    public void abort(final Executor executor) throws SQLException {
        checkClosed();
        conn.abort(executor);
    }

    @Override
    public void setNetworkTimeout(
        final Executor executor,
        final int milliseconds)
        throws SQLException
    {
        checkClosed();
        conn.setNetworkTimeout(executor, milliseconds);
    }

    @Override
    public int getNetworkTimeout() throws SQLException {
        checkClosed();
        return conn.getNetworkTimeout();
    }

    @Override
    public <T> T unwrap(final Class<T> iface) throws SQLException {
        checkClosed();
        return conn.unwrap(iface);
    }

    @Override
    public boolean isWrapperFor(final Class<?> iface) throws SQLException {
        checkClosed();
        return conn.isWrapperFor(iface);
    }
}

