package ru.yandex.search.salo;

import java.io.IOException;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.client.Timings;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.logger.PrefixedLogger;

public class HttpLockHolder {
    private static final String PRODUCER_NAME = "&producer-name=";
    private static final String SALO_WORKER = "&salo-worker=";
    private static final int SUFFIX_LENGTH =
        "&salo-worker=mdb3020:15&transfer-timestamp=1234567890123".length();
    private static final String BATCH_SIZE1 = "&batch-size=1";
    private static final int SINGLE_SUFFIX_LENGTH =
        SUFFIX_LENGTH + BATCH_SIZE1.length();

    private final PrefixedLogger logger;
    private final String name;
    private final String dbId;
    private final CloseableHttpClient client;
    private final HttpHost host;
    private final String service;
    private final String lockRequest;
    private final String getRequest;
    private final int workersPerMdb;
    private final int workersLookahead;
    private final ContentType requestContentType;

    public HttpLockHolder(
        final PrefixedLogger logger,
        final Salo salo,
        final Mdb mdb)
    {
        this.logger = logger;
        name = mdb.name();
        dbId = mdb.dbId();
        client = salo.zoolooserClient();
        host = salo.config().zoolooserConfig().host();
        service = mdb.service();
        lockRequest =
            "/_producer_lock?service=" + service
            + "&session-timeout=" + salo.config().sessionTimeout()
            + PRODUCER_NAME + name;
        getRequest =
            "/_producer_position?service=" + service + PRODUCER_NAME + name;
        workersPerMdb = salo.config().workersPerMdb();
        workersLookahead = salo.config().workersLookahead();
        requestContentType = ContentType.APPLICATION_JSON.withCharset(
            salo.config().zoolooserConfig().requestCharset());
    }

    private long operationId(final int workerId)
        throws HttpException, IOException
    {
        HttpGet request = new HttpGet(getRequest + ':' + workerId);
        try (CloseableHttpResponse response = client.execute(host, request)) {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            }
            String body = CharsetUtils.toString(response.getEntity());
            try {
                long operationId = Long.parseLong(body);
                logger.info(
                    "Retrieved operation id for worker #" + workerId
                    + ':' + ' ' + operationId);
                return operationId;
            } catch (NumberFormatException e) {
                BadResponseException ex =
                    new BadResponseException(request, response, body);
                ex.initCause(e);
                throw ex;
            }
        }
    }

    public Token tryLock() {
        HttpGet request = new HttpGet(lockRequest);
        try (CloseableHttpResponse response =
                client.execute(host, new HttpGet(lockRequest)))
        {
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_FORBIDDEN) {
                logger.fine("Failed to obtain lock");
            } else if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            } else {
                String token = CharsetUtils.toString(response.getEntity());
                logger.info("Token received: " + token);
                long minOperationId = Long.MAX_VALUE;
                for (int i = 0; i < workersPerMdb; ++i) {
                    long operationId = operationId(i);
                    if (operationId != -1L && operationId < minOperationId) {
                        minOperationId = operationId;
                    }
                }
                boolean lookahead = true;
                for (int i = 0; i < workersLookahead && lookahead; ++i) {
                    long operationId = operationId(workersPerMdb + i);
                    if (operationId != -1L) {
                        logger.warning(
                            "For worker #" + (workersPerMdb + i)
                            + " operation id expected to be -1 but was: "
                            + operationId
                            + ". Lock request failed");
                        lookahead = false;
                    }
                }
                if (lookahead) {
                    if (minOperationId == Long.MAX_VALUE) {
                        minOperationId = -1L;
                    }
                    return new Token(token, minOperationId);
                }
            }
        } catch (HttpException | IOException | NumberFormatException e) {
            logger.log(Level.WARNING, "Lock request failed", e);
        }
        return null;
    }

    public Timings storeEnvelopes(
        final List<Envelope> envelopes,
        final String workerName)
        throws HttpException, IOException, TokenException
    {
        HttpRequest request = makeSetRequest(envelopes, workerName);
        Timings timings = new Timings();
        try (CloseableHttpResponse response = client.execute(
                host,
                request,
                timings.createContext()))
        {
            String body = CharsetUtils.toString(response.getEntity());
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_FORBIDDEN) {
                throw new TokenException(
                    "Token rejected by zoolooser with message: " + body,
                    workerName,
                    envelopes.get(0).tokenString());
            } else if (status == HttpStatus.SC_CONFLICT) {
                logger.info(
                    "Skipping envelopes " + envelopes
                    + ", because of conflict " + body);
            } else if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response, body);
            } else {
                logger.fine(
                    "Saved operations: "
                    + envelopes.get(0).operationId()
                    + '-' + envelopes.get(envelopes.size() - 1).operationId()
                    + ':' + ' ' + body);
            }
        }
        return timings;
    }

    private HttpRequest makeSetRequest(
        final List<Envelope> envelopes,
        final String workerName)
    {
        return makeSetRequest(envelopes, workerName, true);
    }

    public HttpRequest makeSetRequest(
        final List<Envelope> envelopes,
        final String workerName,
        final boolean passPosition)
    {
        HttpRequest request;
        Envelope first = envelopes.get(0);
        StringBuilder suffix = new StringBuilder(SUFFIX_LENGTH);
        suffix.append(SALO_WORKER);
        suffix.append(workerName);
        suffix.append("&transfer-timestamp=");
        suffix.append(System.currentTimeMillis());
        if (envelopes.size() == 1 && !first.optional()) {
            StringBuilder sb = first.copyUri(SINGLE_SUFFIX_LENGTH);
            sb.append(BATCH_SIZE1);
            sb.append(suffix);

            if (RequestHandlerMapper.GET.equals(first.method())) {
                request = new HttpGet(new String(sb));
            } else {
                HttpPost post = new HttpPost(new String(sb));
                post.setEntity(first.createEntity(requestContentType));
                request = post;
            }

            request.addHeader(YandexHeaders.LOCKID, first.tokenString());
            request.addHeader(YandexHeaders.SERVICE, service);
            request.addHeader(
                YandexHeaders.ZOO_SHARD_ID,
                Integer.toString(first.shard()));

            if (passPosition) {
                request.addHeader(YandexHeaders.PRODUCER_NAME, workerName);
                request.addHeader(
                    YandexHeaders.PRODUCER_POSITION,
                    first.operationId().toPlainString());
            }
        } else {
            HttpPost postRequest;
            StringBuilder uri = new StringBuilder("/notify?");
            uri.append(dbId);
            uri.append(SALO_WORKER);
            uri.append(workerName);
            uri.append("&operation-ids=");
            uri.append(first.operationId());
            uri.append('-');
            uri.append(envelopes.get(envelopes.size() - 1).operationId());
            uri.append("&first-operation-date=");
            uri.append((long) first.operationDate());
            if (first.optional()) {
                uri.append("&optional");
            }
            uri.append("&batch-size=");
            uri.append(envelopes.size());

            postRequest = new HttpPost(new String(uri));
            postRequest.addHeader(YandexHeaders.LOCKID, first.tokenString());
            postRequest.addHeader(YandexHeaders.SERVICE, service);
            if (passPosition) {
                postRequest.addHeader(YandexHeaders.PRODUCER_NAME, workerName);
            }

            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMimeSubtype("mixed");
            for (Envelope envelope: envelopes) {
                StringBuilder sb = envelope.copyUri(SUFFIX_LENGTH);
                sb.append(suffix);
                FormBodyPartBuilder partBuilder =
                    FormBodyPartBuilder
                        .create()
                        .addField(YandexHeaders.ZOO_SHARD_ID, Integer.toString(envelope.shard()))
                        .addField(YandexHeaders.URI, new String(sb))
                        .addField(YandexHeaders.ZOO_HTTP_METHOD, envelope.method())
                        .setBody(envelope.createBody(requestContentType))
                        .setName("envelope.json");

                if (passPosition) {
                    partBuilder.addField(YandexHeaders.PRODUCER_POSITION, envelope.operationId().toPlainString());
                }

                builder.addPart(partBuilder.build());
            }

            postRequest.setEntity(builder.build());
            request = postRequest;
        }

        return request;
    }
}

