package ru.yandex.search.salo;

import java.io.IOException;
import java.nio.charset.CharacterCodingException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.charset.Encoder;
import ru.yandex.function.GenericBiConsumer;
import ru.yandex.function.StringBuilderVoidProcessor;
import ru.yandex.http.util.client.Timings;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.salo.config.ImmutableSaloConfig;

public class Mdb extends Thread {
    public static final long SHARDS = 65534L;

    private static final String MS = " ms";
    private static final String TO = " to ";
    private static final String LAST_OPERATION_DATE = "last_operation_date";
    private static final String LAST_TRANSFER_LAG = "last_transfer_lag";
    private static final String TRANSFER_LAG = "transfer_lag";
    private static final String FETCH_LAG = "fetch_lag";

    private final ReadWriteLock rwlock = new ReentrantReadWriteLock();
    private final Lock readLock = rwlock.readLock();
    private final Lock writeLock = rwlock.writeLock();
    private final MdbsContext mdbsContext;
    private final String service;
    private final PrefixedLogger logger;
    private final String name;
    private final String dbId;
    private final EnvelopeFactory envelopeFactory;
    private final ThreadGroup workersThreadGroup;
    private final ImmutableSaloConfig config;
    private final HttpLockHolder holder;
    private final int shardsPerWorker;
    private final MdbWorker[] workers;
    private final StringBuilderVoidProcessor<byte[], CharacterCodingException>
        encoder;
    private final MsalClient client;
    private final int midsLimit;
    private volatile long tokenVersion = 0L;
    private volatile boolean running = true;
    private volatile boolean locked = true;
    private volatile Token lastToken;
    private volatile long fetchLag = 0L;
    // Last worker received envelope
    private int lastWorker;

    public Mdb(
        final MdbsContext mdbsContext,
        final DatabaseContext databaseContext,
        final String service)
    {
        super(mdbsContext.threadGroup(), databaseContext.name());
        this.mdbsContext = mdbsContext;
        this.service = service;
        Salo salo = mdbsContext.salo();
        logger = salo.logger();
        name = databaseContext.name();
        dbId = databaseContext.dbId();
        envelopeFactory = databaseContext.envelopeFactory();
        workersThreadGroup = new ThreadGroup(mdbsContext.threadGroup(), name);
        config = mdbsContext.config();
        holder = new HttpLockHolder(logger.addPrefix("lock"), salo, this);
        workers = new MdbWorker[config.workersPerMdb()];
        shardsPerWorker =
            (int) ((SHARDS + config.workersPerMdb() - 1L)
                / config.workersPerMdb());
        for (int i = 0; i < config.workersPerMdb(); ++i) {
            workers[i] = new MdbWorker(i, this);
        }
        encoder = new StringBuilderVoidProcessor<>(
            new Encoder(config.zoolooserConfig().requestCharset()));
        client = new MsalClient(this);
        midsLimit = config.midsLimit();
    }

    public MdbsContext context() {
        return mdbsContext;
    }

    public String service() {
        return service;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public String name() {
        return name;
    }

    public String dbId() {
        return dbId;
    }

    public EnvelopeFactory envelopeFactory() {
        return envelopeFactory;
    }

    public ThreadGroup workersThreadGroup() {
        return workersThreadGroup;
    }

    public ImmutableSaloConfig config() {
        return config;
    }

    public int workersCount() {
        return workers.length;
    }

    public int shardsPerWorker() {
        return shardsPerWorker;
    }

    public int midsLimit() {
        return midsLimit;
    }

    public StringBuilderVoidProcessor<byte[], CharacterCodingException>
        encoder()
    {
        return encoder;
    }

    public boolean locked() {
        return locked;
    }

    public boolean pinhole() {
        return client.pinhole();
    }

    public long lastTransferLag() {
        long max = 0L;
        for (MdbWorker worker: workers) {
            long lag = worker.lastTransferLag();
            if (lag > max) {
                max = lag;
            }
        }

        return max;
    }

    public long transferLag() {
        long max = 0L;
        for (MdbWorker worker: workers) {
            long lag = worker.transferLag();
            if (lag > max) {
                max = lag;
            }
        }

        return max;
    }

    public void fetchLag(final long lag) {
        this.fetchLag = lag;
    }

    public long fetchLag() {
        return fetchLag;
    }

    public void markActive() {
        mdbsContext.markShardActive(name);
    }

    public MsalClient client() {
        return client;
    }

    public Map<String, Object> status() {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put("locked", locked);
        double lastOperationDate = 0d;
        long lastTransferLag = 0L;
        long transferLag = 0L;
        for (MdbWorker worker: workers) {
            double workerOperationDate = worker.lastOperationDate();
            if (workerOperationDate != 0d) {
                if (workerOperationDate > lastOperationDate) {
                    lastOperationDate = workerOperationDate;
                }
                long workerLastTransferLag = worker.lastTransferLag();
                if (workerLastTransferLag > lastTransferLag) {
                    lastTransferLag = workerLastTransferLag;
                }

                long workerTransferLag = worker.transferLag();
                if (workerTransferLag > transferLag) {
                    transferLag = workerTransferLag;
                }
                Map<String, Object> workerStatus = new LinkedHashMap<>();
                workerStatus.put("queue_size", worker.queue().size());
                workerStatus.put(LAST_OPERATION_DATE, workerOperationDate);
                workerStatus.put(LAST_TRANSFER_LAG, workerLastTransferLag);
                workerStatus.put(TRANSFER_LAG, workerTransferLag);
                status.put(worker.getName(), workerStatus);
            }
        }
        if (lastOperationDate != 0d) {
            status.put(LAST_OPERATION_DATE, lastOperationDate);
            status.put(LAST_TRANSFER_LAG, lastTransferLag);
            status.put(TRANSFER_LAG, transferLag);
            status.put(FETCH_LAG, fetchLag);
        }
        return status;
    }

    @Override
    public void run() {
        logger.info("Starting");
        for (MdbWorker worker: workers) {
            worker.start();
        }
        while (running) {
            // CSOFF: EmptyBlock
            try {
                processOperationQueue();
            } catch (InterruptedException e) {
            }
            // CSON: EmptyBlock
        }
        for (MdbWorker worker: workers) {
            try {
                worker.join();
            } catch (InterruptedException e) {
                break;
            }
        }
        logger.info("Stopped");
    }

    public void close() {
        running = false;
        interrupt();
        for (MdbWorker worker: workers) {
            worker.interrupt();
        }
    }

    private boolean dispatchEnvelope(final Envelope envelope)
        throws InterruptedException
    {
        int worker = envelope.shard() / shardsPerWorker;
        BlockingQueue<Envelope> queue = workers[worker].queue();
        boolean result = false;
        do {
            long currentTokenVersion = this.tokenVersion;
            if (envelope.tokenVersion() != currentTokenVersion) {
                logger.warning(
                    "While processing envelope with operation id"
                    + envelope.operationId()
                    + " token version changed from " + envelope.tokenVersion()
                    + TO + currentTokenVersion + ". Ignoring other envelopes");
                return false;
            }
            result = queue.offer(
                envelope,
                config.envelopesCheckInterval(),
                TimeUnit.MILLISECONDS);
        } while (!result && running);
        return result;
    }

    private void processOperationQueue() throws InterruptedException {
        Token token = holder.tryLock();
        locked = token != null;
        if (!locked) {
            lastToken = null;
            logger.info(
                "Can't obtain lock for " + name
                + ", sleep for: " + config.lockCheckInterval() + MS);
            sleep(config.lockCheckInterval());
            return;
        }

        lastToken = token;
        long tokenTimestamp = System.currentTimeMillis();
        String tokenString = token.tokenString();
        long lastOperationId = token.minOperationId();
        long tokenVersion = this.tokenVersion;
        logger.fine(
            "For token " + tokenString
            + " last operation id is " + lastOperationId
            + " and tokenVersion is " + tokenVersion);
        client.setLastOperationId(lastOperationId);
        EnvelopesContext context =
            new EnvelopesContext(tokenString, tokenVersion, this);
        while (running
            && tokenVersion == this.tokenVersion
            && !interrupted())
        {
            List<Envelope> envelopes;
            try {
                envelopes = client.next(context);
            } catch (HttpException | IOException e) {
                logger.log(
                    Level.WARNING,
                    "Failed to retrieve data from operations queue",
                    e);
                envelopes = null;
            }
            int size;
            if (envelopes == null) {
                size = 0;
            } else {
                size = envelopes.size();
            }
            if (size == 0) {
                long sinceLastLease =
                    System.currentTimeMillis() - tokenTimestamp;
                if (sinceLastLease > config.sessionTimeout()) {
                    logger.info(
                        sinceLastLease
                        + " ms passed since last token lease. "
                        + "It's time to renew it");
                    break;
                } else {
                    logger.info(
                        "No envelopes to process. Sleep for: "
                        + config.envelopesCheckInterval()
                        + MS);
                    sleep(config.envelopesCheckInterval());
                }
            } else {
                long start = System.currentTimeMillis();
                for (int i = 0; i < size; ++i) {
                    if (!dispatchEnvelope(envelopes.get(i))) {
                        return;
                    }
                }
                lastOperationId =
                    envelopes.get(size - 1).operationId().longValue();
                long shardOffset = lastOperationId % shardsPerWorker;
                lastWorker = (lastWorker + 1) % workers.length;
                Envelope envelope = new PingEnvelope(
                    lastOperationId,
                    context,
                    (int) Math.min(
                        SHARDS - 1,
                        workers[lastWorker].firstShard() + shardOffset));
                if (!dispatchEnvelope(envelope)) {
                    return;
                }
                tokenTimestamp = System.currentTimeMillis();
                logger.info(
                    "All envelopes added to processing queue in "
                    + (tokenTimestamp - start) + MS);
            }
        }
    }

    @SuppressWarnings("NonAtomicVolatileUpdate")
    private void incrementTokenVersion() {
        long tokenVersion;
        writeLock.lock();
        try {
            tokenVersion = ++this.tokenVersion;
        } finally {
            writeLock.unlock();
        }
        logger.info("Token version incremented to " + tokenVersion);
    }

    public void exectueWithLock(
        final GenericBiConsumer<HttpLockHolder, EnvelopesContext, Exception> consumer)
        throws Exception
    {
        Token token = lastToken;
        String tokenString = null;
        long tokenVersion = -1L;
        if (token != null) {
            tokenString = token.tokenString();
            tokenVersion = token.minOperationId();
        }
        readLock.lock();
        try {
            consumer.accept(holder, new EnvelopesContext(tokenString, tokenVersion, this));
        } finally {
            readLock.unlock();
        }
    }

    public Timings processEnvelopes(
        final List<Envelope> envelopes,
        final String workerName)
    {
        TokenException e = null;
        long tokenVersion = -1L;
        long envelopesTokenVersion = envelopes.get(0).tokenVersion();
        while (running) {
            readLock.lock();
            try {
                tokenVersion = this.tokenVersion;
                if (envelopesTokenVersion == tokenVersion) {
                    try {
                        return holder.storeEnvelopes(envelopes, workerName);
                    } catch (TokenException ex) {
                        e = ex;
                        break;
                    } catch (HttpException | IOException ex) {
                        logger.log(
                            Level.WARNING,
                            "Failed to process envelopes " + envelopes,
                            ex);
                    }
                } else {
                    break;
                }
            } finally {
                readLock.unlock();
            }
        }
        if (isInterrupted()) {
            logger.info("Thread interrupted while processing " + envelopes);
        } else if (e == null) {
            logger.warning(
                "Token version changed from " + envelopesTokenVersion
                + TO + tokenVersion + ". Ignoring envelopes " + envelopes);
        } else {
            logger.log(
                Level.WARNING,
                "Token invalidated for envelopes " + envelopes,
                e);
            incrementTokenVersion();
        }
        return null;
    }
}

