package ru.yandex.search.salo;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;

import ru.yandex.http.util.client.Timings;

public class MdbWorker extends Thread {
    private static final int INITIAL_LOG_RECORD_CAPACITY = 256;
    private static final double MILLIS = 1000d;

    private final int id;
    private final Mdb mdb;
    private final String name;
    private final BlockingQueue<Envelope> queue;
    private final int requestsBatchSize;
    private final List<Envelope> envelopes;
    private volatile double lastOperationDate = 0d;
    private volatile long lastTransferLag = 0L;
    private volatile long transferLag = 0L;

    public MdbWorker(final int id, final Mdb mdb) {
        super(
            mdb.workersThreadGroup(),
            mdb.name() + ':' + id + ':' + (id * mdb.shardsPerWorker()) + '-'
            + (((id + 1) * mdb.shardsPerWorker()) - 1));
        this.id = id;
        this.mdb = mdb;
        name = mdb.name() + ':' + id;
        queue = new ArrayBlockingQueue<>(mdb.config().workerQueueLength());
        requestsBatchSize = mdb.config().requestsBatchSize();
        envelopes = new ArrayList<>(requestsBatchSize);
    }

    public int firstShard() {
        return id * mdb.shardsPerWorker();
    }

    public BlockingQueue<Envelope> queue() {
        return queue;
    }

    public double lastOperationDate() {
        return lastOperationDate;
    }

    public long lastTransferLag() {
        return lastTransferLag;
    }

    public long transferLag() {
        return transferLag;
    }

    @Override
    public void run() {
        mdb.logger().info("Started");
        Envelope envelope = null;
        while (!isInterrupted()) {
            try {
                if (envelope == null) {
                    envelope = queue.poll(
                        mdb.config().envelopesCheckInterval(),
                        TimeUnit.MILLISECONDS);
                }
                if (envelope != null) {
                    // Skip all optional envelopes if there are other envelopes
                    // after them
                    while (envelope.optional() && !isInterrupted()) {
                        Envelope next = queue.poll(
                            mdb.config().envelopesCheckInterval(),
                            TimeUnit.MILLISECONDS);
                        if (next == null) {
                            break;
                        } else {
                            envelope = next;
                        }
                    }
                    envelopes.clear();
                    envelopes.add(envelope);
                    int envelopesCount;
                    int midsCount = envelope.midsCount();
                    int size = envelope.size();
                    if (envelope.optional()) {
                        // Non-optional envelope may be saved between loop
                        // cycles. But there no need for optional envelope
                        envelope = null;
                        // Doesn't count this envelope for something useful
                        envelopesCount = 0;
                    } else {
                        envelopesCount = 1;
                        long tokenVersion = envelope.tokenVersion();
                        Envelope prev = envelope;
                        envelope = null;
                        while (envelopes.size() < requestsBatchSize) {
                            envelope = queue.poll();
                            if (envelope == null
                                || envelope.tokenVersion() != tokenVersion)
                            {
                                break;
                            } else {
                                if (!envelope.optional()) {
                                    ++envelopesCount;
                                }
                                midsCount += envelope.midsCount();
                                size += envelope.size();
                                if (prev.optional()) {
                                    envelopes.set(
                                        envelopes.size() - 1,
                                        envelope);
                                } else {
                                    envelopes.add(envelope);
                                }
                                prev = envelope;
                                envelope = null;
                            }
                        }
                    }
                    long start = System.currentTimeMillis();
                    Timings timings = mdb.processEnvelopes(envelopes, name);
                    if (timings != null) {
                        StringBuilder sb =
                            new StringBuilder(INITIAL_LOG_RECORD_CAPACITY);
                        sb.append(envelopes.size());
                        sb.append(" envelopes processed in ");
                        sb.append(System.currentTimeMillis() - start);
                        sb.append(" ms. Transfer lag ");
                        long lag = 0L;
                        for (int i = envelopes.size(); i-- > 0;) {
                            Envelope last = envelopes.get(i);
                            if (!last.optional()) {
                                lastOperationDate = last.operationDate();
                                long date =
                                    (long) (lastOperationDate * MILLIS);
                                lag = start - date;
                                mdb.context().transferred(
                                    mdb.name(),
                                    new TransferStats(
                                        envelopesCount,
                                        midsCount,
                                        size),
                                    start,
                                    date);
                                break;
                            }
                        }
                        sb.append(lag);
                        lastTransferLag = lag;
                        transferLag = lag;
                        sb.append(" ms. Processing started after ");
                        sb.append(timings.startTime() - start);
                        sb.append(" ms and took ");
                        timings.toStringBuilder(sb);
                        sb.append(". Envelopes processed: [");
                        envelopes.get(0).toStringBuilder(sb);
                        for (int i = 1; i < envelopes.size(); ++i) {
                            sb.append(',');
                            envelopes.get(i).toStringBuilder(sb);
                        }
                        sb.append(']');
                        mdb.logger().fine(new String(sb));
                    }
                } else {
                    transferLag = 0;
                }
            } catch (InterruptedException e) {
                break;
            }
        }
        mdb.logger().info("Stopped");
    }
}

