package ru.yandex.search.salo;

import java.util.Set;
import java.util.function.Supplier;

import org.apache.http.HttpStatus;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.collection.IdentityHashSet;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.salo.config.ImmutableSaloConfig;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.stater.StaterConfigBuilder;
import ru.yandex.stater.UniqueKeysStater;
import ru.yandex.util.timesource.TimeSource;

public class MdbsContext {
    private static final String AMMM = "_ammm";
    private static final int EXPECTED_SHARDS = 64;
    private static final Supplier<Set<String>> SHARDS_SET_SUPPLIER =
        () -> new IdentityHashSet<>(EXPECTED_SHARDS);

    private final ImmutableSaloConfig config;
    private final PrefixedLogger logger;
    private final Salo salo;
    private final ThreadGroup threadGroup;
    private final RequestsStater queueStater;
    private final RequestsStater minTransactionStater;
    private final RequestsStater transferLagStater;
    private final TimeFrameQueue<Integer> receivedEnvelopesStats;
    private final TimeFrameQueue<Long> pinholeTimeStats;
    private final TimeFrameQueue<Long> totalPinholeTimeStats;
    private final TimeFrameQueue<TransferStats> transferStats;
    private final TimeFrameQueue<String> msalResponsesStats;
    private final TimeFrameQueue<String> activeShards;

    public MdbsContext(
        final Salo salo,
        final ThreadGroup threadGroup,
        final String statsPrefix)
    {
        this.config = salo.config();
        this.threadGroup = threadGroup;
        this.salo = salo;
        this.logger = salo.logger();
        String suffix = '-' + statsPrefix;

        queueStater =
            new StaterConfigBuilder(config.msalResponsesStaterConfig())
                .prefix(config.msalResponsesStaterConfig().prefix() + suffix)
                .build()
                .build();
        salo.registerStater(queueStater);

        minTransactionStater =
            new StaterConfigBuilder(config.minTransactionStaterConfig())
                .prefix(config.minTransactionStaterConfig().prefix() + suffix)
                .build()
                .build();
        salo.registerStater(minTransactionStater);

        transferLagStater =
            new StaterConfigBuilder(config.transferLagStaterConfig())
                .prefix(config.transferLagStaterConfig().prefix() + suffix)
                .build()
                .build();
        salo.registerStater(transferLagStater);

        receivedEnvelopesStats =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                receivedEnvelopesStats,
                new NamedStatsAggregatorFactory<>(
                    "envelopes-received" + suffix + AMMM,
                    IntegralSumAggregatorFactory.INSTANCE)));

        pinholeTimeStats = new TimeFrameQueue<>(config.metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                pinholeTimeStats,
                new NamedStatsAggregatorFactory<>(
                    "pinhole-time" + suffix + AMMM,
                    IntegralSumAggregatorFactory.INSTANCE)));

        totalPinholeTimeStats =
            new TimeFrameQueue<>(config.metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                totalPinholeTimeStats,
                new NamedStatsAggregatorFactory<>(
                    "total-pinhole-time" + suffix + "_ammt",
                    new MaxAggregatorFactory(0L))));

        transferStats = new TimeFrameQueue<>(config.metricsTimeFrame());
        salo.registerStater(new TransferStater(transferStats, statsPrefix));
        msalResponsesStats = new TimeFrameQueue<>(config.metricsTimeFrame());
        salo.registerStater(
            new MsalResponsesStater(msalResponsesStats, statsPrefix));
        activeShards = new TimeFrameQueue<>(config.activeShardsTimeFrame());
        salo.registerStater(
            new UniqueKeysStater<>(
                activeShards,
                SHARDS_SET_SUPPLIER,
                statsPrefix + "-active-shards_ammv"));
    }

    public ImmutableSaloConfig config() {
        return config;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public Salo salo() {
        return salo;
    }

    public ThreadGroup threadGroup() {
        return threadGroup;
    }

    public void queueResponse(final int status, final long startTime) {
        queueStater.accept(
            new RequestInfo(
                TimeSource.INSTANCE.currentTimeMillis(),
                status,
                startTime,
                startTime,
                0L,
                0L));
    }

    public void minTransactionResponse(
        final int status,
        final long startTime)
    {
        minTransactionStater.accept(
            new RequestInfo(
                TimeSource.INSTANCE.currentTimeMillis(),
                status,
                startTime,
                startTime,
                0L,
                0L));
    }

    public void envelopesReceived(final String mdb, final int envelopesCount) {
        msalResponsesStats.accept(mdb);
        receivedEnvelopesStats.accept(envelopesCount);
    }

    public void pinhole(final long pinholeTime, final long totalPinholeTime) {
        pinholeTimeStats.accept(pinholeTime);
        totalPinholeTimeStats.accept(totalPinholeTime);
    }

    public void markShardActive(final String shard) {
        activeShards.accept(shard);
    }

    public void transferred(
        final String shard,
        final TransferStats stat,
        final long transferDate,
        final long lastOperationDate)
    {
        activeShards.accept(shard);
        transferStats.accept(stat);
        transferLagStater.accept(
            new RequestInfo(
                transferDate,
                HttpStatus.SC_OK,
                lastOperationDate,
                lastOperationDate,
                0L,
                0L));
    }

    public CloseableHttpClient msalClient() {
        return salo.msalClient();
    }

    public ImmutableHttpHostConfig msalConfig() {
        return salo.config().msalConfig();
    }
}

