package ru.yandex.search.salo;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.collection.DoublePair;
import ru.yandex.dbfields.PgFields;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.function.CharArrayProcessable;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.KeyInterningStringCollectorsFactory;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.search.salo.config.ImmutableSaloConfig;

public class MsalClient {
    private static final String MS = " ms";

    private final StringBuilder sb = new StringBuilder();
    private final Mdb mdb;
    private final CloseableHttpClient client;
    private final Logger logger;
    private final HttpHost host;
    private final long envelopesCheckInterval;
    private final int selectLength;
    private final int requestPrefixLength;
    private final List<Envelope> envelopes;
    private final String minTransactionDateUri;
    private final ContainerFactory jsonContainerFactory;
    private long operationId = 0L;
    private volatile boolean pinhole = false;

    public MsalClient(final Mdb mdb) {
        this.mdb = mdb;
        this.jsonContainerFactory =
            mdb.envelopeFactory().jsonContainerFactory();
        ImmutableSaloConfig config = mdb.context().config();
        client = mdb.context().msalClient();
        logger = mdb.context().logger();
        host = mdb.context().msalConfig().host();
        envelopesCheckInterval = config.envelopesCheckInterval();
        selectLength = config.selectLength();
        sb.append(
            "/operations-queue-envelopes?json-type=dollar"
            + "&namespace=operations_queue&");
        sb.append(mdb.dbId());
        String scope = mdb.envelopeFactory().scope();
        if (scope != null) {
            sb.append("&scope=");
            sb.append(scope);
        }
        sb.append("&length=");
        requestPrefixLength = sb.length();
        envelopes = new ArrayList<>(selectLength);
        minTransactionDateUri =
            "/get-min-transaction-date?json-type=dollar&" + mdb.dbId();
    }

    public boolean pinhole() {
        return pinhole;
    }

    public void setLastOperationId(final long lastOperationId) {
        operationId = lastOperationId + 1L;
        logger.fine("Advancing operation id to " + operationId);
    }

    private void requestEnvelopes(
        final EnvelopesContext context,
        final int count,
        final boolean master)
        throws IOException, HttpException
    {
        sb.setLength(requestPrefixLength);
        sb.append(count);
        sb.append("&op-id=");
        sb.append(operationId);
        if (master) {
            sb.append("&no-ro");
        }
        String requestUri = sb.toString();
        logger.fine("Requesting MSAL for " + requestUri);
        int status = YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST;
        long start = System.currentTimeMillis();
        try (CloseableHttpResponse response = client.execute(
                host,
                new BasicHttpRequest(HttpGet.METHOD_NAME, requestUri)))
        {
            status = response.getStatusLine().getStatusCode();
            if (status != YandexHttpStatus.SC_OK) {
                throw new BadResponseException(requestUri, response);
            }
            CharArrayProcessable data =
                CharsetUtils.toDecoder(response.getEntity());
            try {
                BasicGenericConsumer<JsonObject, JsonException> consumer =
                    new BasicGenericConsumer<>();
                data.processWith(
                    new JsonParser(
                        new StackContentHandler(
                            new TypesafeValueContentHandler(
                                consumer,
                                KeyInterningStringCollectorsFactory.INSTANCE
                                    .apply(data.length()),
                                jsonContainerFactory))));
                final JsonMap value = consumer.get().asMap();
                JsonList rows = value.get("rows").asList();
                envelopes.clear();
                int rowsSize = rows.size();
                mdb.envelopeFactory().process(context, rows, envelopes);
//                int rowsSize = rows.size();
//                for (int i = 0; i < rowsSize; ++i) {
//                    JsonMap row = rows.get(i).asMap();
//                    mdb.envelopeFactory().process(context, row, envelopes);
//                }
                StringBuilder sb = new StringBuilder("MSAL returned ");
                sb.append(rowsSize);
                sb.append(" rows ");
                if (rowsSize != 0) {
                    sb.append("with cids: ");
                    sb.append(rows.get(0).get(PgFields.OPERATION_ID).asLong());
                    sb.append('-');
                    sb.append(
                        rows.get(rowsSize - 1)
                            .get(PgFields.OPERATION_ID).asLong());
                    sb.append(' ');
                }
                sb.append("which were converted into ");
                sb.append(envelopes.size());
                sb.append(" envelopes in ");
                sb.append(System.currentTimeMillis() - start);
                sb.append(MS);
                logger.fine(new String(sb));
                mdb.context().envelopesReceived(mdb.name(), envelopes.size());
            } catch (Throwable t) {
                throw new IOException("Can't parse MSAL response: " + data, t);
            }
        } finally {
            mdb.context().queueResponse(status, start);
        }
    }

    private List<Envelope> prepareEnvelopes() {
        if (envelopes.isEmpty()) {
            mdb.markActive();
        } else {
            setLastOperationId(
                envelopes.get(envelopes.size() - 1).operationId().longValue());
        }
        return envelopes;
    }

    private DoublePair<Double> getMinTransactionDate()
        throws IOException, HttpException
    {
        int status = YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST;
        long start = System.currentTimeMillis();
        try (CloseableHttpResponse response = client.execute(
                host,
                new BasicHttpRequest(
                    HttpGet.METHOD_NAME,
                    minTransactionDateUri)))
        {
            status = response.getStatusLine().getStatusCode();
            if (status != YandexHttpStatus.SC_OK) {
                throw new BadResponseException(
                    minTransactionDateUri,
                    response);
            }
            try {
                JsonObject data = TypesafeValueContentHandler.parse(
                    CharsetUtils.content(response.getEntity()));
                JsonMap map = data.asMap();
                return new DoublePair<>(
                    map.getDouble("server_timestamp"),
                    map.getDouble("min_transaction_date", null));
            } catch (JsonException e) {
                throw new IOException(
                    "Failed to parse minimal transaction date",
                    e);
            }
        } finally {
            mdb.context().minTransactionResponse(status, start);
        }
    }

    // CSOFF: ReturnCount
    public List<Envelope> next(final EnvelopesContext context)
        throws IOException, HttpException
    {
        requestEnvelopes(context, selectLength, false);
        long pinhole = detectPinhole(false);
        if (pinhole != -1L) {
            requestEnvelopes(context, selectLength, true);
            pinhole = detectPinhole(true);
            if (pinhole != -1L) {
                long start = 0L;
                long totalPinholeTime = 0L;
                DoublePair<Double> minTransactionDate =
                    getMinTransactionDate();
                double recoveryStartTime = minTransactionDate.first();
                logger.info(
                    String.format(
                        "Recovery start time: %.6f",
                        recoveryStartTime));
                try {
                    while (!mdb.isInterrupted()) {
                        if (start == 0L) {
                            this.pinhole = true;
                            start = System.currentTimeMillis();
                        } else {
                            long now = System.currentTimeMillis();
                            long diff = now - start;
                            totalPinholeTime += diff;
                            mdb.context().pinhole(diff, totalPinholeTime);
                            start = now;
                        }
                        Double minXact = minTransactionDate.second();
                        logger.info(
                            String.format(
                                "Minimal transaction date: %.6f",
                                minXact));
                        if (minXact == null || minXact > recoveryStartTime) {
                            logger.info("All affecting transactions gone");
                            requestEnvelopes(context, selectLength, true);
                            truncateEnvelopes(pinhole);
                            break;
                        } else {
                            logger.warning(
                                "Waiting for " + envelopesCheckInterval + MS);
                            try {
                                Thread.sleep(envelopesCheckInterval);
                            } catch (InterruptedException e) {
                                mdb.interrupt();
                                break;
                            }
                            requestEnvelopes(context, selectLength, true);
                            if (detectPinhole(false) == -1L) {
                                logger.info("Pinhole has gone");
                                break;
                            }
                            // TODO: Fail-fast checking that pinhole has gone?
                            minTransactionDate = getMinTransactionDate();
                        }
                    }
                } finally {
                    if (start != 0L) {
                        this.pinhole = false;
                        long diff = System.currentTimeMillis() - start;
                        totalPinholeTime += diff;
                        mdb.context().pinhole(diff, totalPinholeTime);
                        logger.info("Total time wasted: " + totalPinholeTime);
                    }
                }
            }
        }
        return prepareEnvelopes();
    }
    // CSON: ReturnCount

    // Returns -1 if no pinhole detected. Optionally truncates envelopes to
    // have no pinholes.
    // Returns max operation_id if pinhole presents
    private long detectPinhole(final boolean log) {
        long maxOperationId = -1L;
        int size = envelopes.size();
        if (size != 0) {
            long operationId = envelopes.get(0).operationId().longValue();
            if (operationId != this.operationId) {
                maxOperationId =
                    envelopes.get(size - 1).operationId().longValue();
                if (log) {
                    logger.info(
                        "Pinhole detected: ["
                        + this.operationId + ',' + operationId + ')');
                }
            } else {
                long lastOperationId = operationId;
                for (int i = 1; i < size; ++i) {
                    operationId = envelopes.get(i).operationId().longValue();
                    if (operationId > lastOperationId + 1) {
                        envelopes.subList(i, size).clear();
                        break;
                    }
                    lastOperationId = operationId;
                }
            }
        }
        return maxOperationId;
    }

    private void truncateEnvelopes(
        final long maxOperationId)
        throws IOException
    {
        int size = envelopes.size();
        if (size != 0) {
            long operationId = envelopes.get(0).operationId().longValue();
            if (operationId > maxOperationId) {
                throw new IOException(
                    "operation_id " + maxOperationId
                    + " wasn't found, first operation_id is " + operationId);
            }
            int pos = 0;
            for (; pos < size; ++pos) {
                operationId =
                    envelopes.get(pos).operationId().longValue();
                if (operationId == maxOperationId) {
                    long lastOperationId = maxOperationId;
                    int next = pos + 1;
                    while (next < size) {
                        operationId =
                            envelopes.get(next).operationId().longValue();
                        if (lastOperationId + 1 >= operationId) {
                            pos = next++;
                            lastOperationId = operationId;
                        } else {
                            break;
                        }
                    }
                    break;
                }
            }
            if (pos + 1 < size) {
                envelopes.subList(pos + 1, size).clear();
            }
        }
    }
}

